"use strict";
/**
 * @author    Yannick Deubel (https://github.com/yandeu)
 * @copyright Copyright (c) 2021 Yannick Deubel
 * @license   {@link https://github.com/yandeu/five-server/blob/main/LICENSE LICENSE}
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.preview = void 0;
const path_1 = require("path");
const fs_1 = require("fs");
const public_1 = require("../public");
const fileTypes_1 = require("../fileTypes");
const explorer_1 = require("./explorer");
const preview = (root, injectToAny) => {
    return (req, res, next) => {
        if (!injectToAny)
            return next();
        if (!['.preview', '.php'].includes(path_1.extname(req.url)))
            return next();
        // remove .preview
        req.url = req.url.replace(/\.preview$/, '');
        const URL = decodeURI(req.url);
        const isPHP = path_1.extname(req.url) === '.php';
        const phpMsg = isPHP
            ? 'Why this preview? Five Server could not detect any head, body or html tag in your file.<br><br>'
            : '';
        try {
            const filePath = path_1.resolve(path_1.join(root + URL));
            const isFile = fs_1.statSync(filePath).isFile();
            if (!isFile)
                return next();
            let ext = path_1.extname(URL).replace(/^\./, '').toLowerCase();
            const fileName = path_1.basename(filePath, ext);
            const isImage = fileTypes_1.fileTypes.isImage(ext);
            const isVideo = fileTypes_1.fileTypes.isVideo(ext);
            const isAudio = fileTypes_1.fileTypes.isAudio(ext);
            const isPDF = fileTypes_1.fileTypes.isPDF(ext);
            let preview = '';
            if (isImage)
                preview = `<div class="image" text-align:center; line-height: 0; padding: 0;">
        
        <img style="max-width: 100%;" src="${URL}"></div>`;
            else if (isVideo) {
                const format = ext === 'ogg' ? 'ogg' : ext === 'webm' ? 'webm' : 'mp4';
                preview = `
            <video style="max-width: 100%;" controls>
            <source src="${URL}" type="video/${format}">
            Your browser does not support the video tag.
            </video>`;
            }
            else if (isAudio) {
                const format = ext === 'ogg' ? 'ogg' : ext === 'wav' ? 'wav' : 'mpeg';
                preview = `
            <div style="margin-top: 72px;">
            <audio controls>
            <source src="${URL}" type="audio/${format}">
            Your browser does not support the audio element.
            </audio>
            </div>`;
            }
            else if (isPDF) {
                preview = `
            <div>
            <iframe 
            style="min-height: calc(100vh - 260px)"
            frameborder="0" 
            scrolling="no"                
            width="100%" height="100%"
            src="${URL}">
            </iframe>
            </div>`;
            }
            else {
                const MAX_FILE_SIZE = 250; // KB
                const fileSize = Math.round(fs_1.statSync(filePath).size / 1024); // KB
                const tooBig = fileSize > MAX_FILE_SIZE;
                if (tooBig)
                    ext = 'txt';
                let fileContent = !tooBig
                    ? fs_1.readFileSync(filePath, { encoding: 'utf-8' })
                    : `File is too big for a preview!\n\n\nFile Size: ${fileSize}KB\nAllowed Size: ${MAX_FILE_SIZE}KB`;
                // check for .rc file (can be yml or json)
                if (/^\.[\w]+rc$/.test(fileName)) {
                    const content = fileContent.trim();
                    ext = content[0] === '{' ? 'json' : 'yml';
                }
                // replace all < with &lt;
                fileContent = fileContent.replace(/</gm, '&lt;');
                preview = `
            <div>
            <pre margin="0px;"><code class="">${fileContent}</code></pre>
            </div>`;
            }
            const html = public_1.PREVIEW.replace('{linked-path}', explorer_1.htmlPath(URL))
                .replace('{fileName}', fileName)
                .replace('{ext}', ext)
                .replace('{phpMsg}', phpMsg)
                .replace('{preview}', preview);
            return res.type('html').send(html);
        }
        catch (error) {
            return next();
        }
    };
};
exports.preview = preview;
