"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.donate = exports.getConfigFile = exports.removeTrailingSlash = exports.removeLeadingSlash = exports.escape = exports.fileDoesExist = exports.createError = void 0;
const colors_1 = require("./colors");
const escape_html_1 = __importDefault(require("./dependencies/escape-html"));
const fs_1 = __importDefault(require("fs"));
const msg_1 = require("./msg");
const path_1 = __importDefault(require("path"));
// just a fallback for removing http-errors dependency
const createError = (code, msg = 'unknown', _nothing) => {
    if (code !== 404)
        msg_1.message.log(`ERROR: ${code} ${msg}`);
    return { message: msg, code, status: code, statusCode: code, name: code };
};
exports.createError = createError;
const fileDoesExist = (path) => {
    return new Promise(resolve => {
        fs_1.default.stat(path, (err, stat) => {
            if (err && err.code === 'ENOENT') {
                return resolve(false);
            }
            else
                return resolve(true);
        });
    });
};
exports.fileDoesExist = fileDoesExist;
const escape = html => escape_html_1.default(html);
exports.escape = escape;
const removeLeadingSlash = (str) => {
    return str.replace(/^\/+/g, '');
};
exports.removeLeadingSlash = removeLeadingSlash;
const removeTrailingSlash = (str) => {
    return str.replace(/\/+$/g, '');
};
exports.removeTrailingSlash = removeTrailingSlash;
/**
 * Get and parse the configFile.
 * @param configFile Absolute path of configFile, or true, or false.
 * @param workspace Absolute path to the current workspace.
 * @returns LiveServerParams
 */
const getConfigFile = (configFile = true, workspace) => {
    let options = {};
    if (configFile === false)
        return options;
    const dirs = [];
    const files = [
        '.fiveserverrc',
        '.fiveserverrc.json',
        '.fiveserverrc.js',
        '.fiveserverrc.cjs',
        '.fiveserverrc.mjs',
        'fiveserver.config.js',
        'fiveserver.config.cjs',
        'fiveserver.config.mjs',
        '.live-server.json'
    ];
    if (typeof configFile === 'string') {
        // TODO: Add support for this
        files.unshift(configFile);
    }
    if (workspace)
        dirs.push(workspace);
    dirs.push(path_1.default.resolve());
    const homeDir = process.env[process.platform === 'win32' ? 'USERPROFILE' : 'HOME'];
    if (homeDir)
        dirs.push(homeDir);
    dirs.push(process.cwd());
    const isJSReg = /\.c?m?js$/;
    loop: for (const d of dirs) {
        for (const f of files) {
            const configPath = path_1.default.join(d, f);
            if (fs_1.default.existsSync(configPath)) {
                const isJS = isJSReg.test(path_1.default.extname(configPath));
                if (isJS) {
                    try {
                        delete require.cache[configPath];
                        const config = require(configPath);
                        if (Object.keys(config).length === 0) {
                            msg_1.message.warn(`Config file "${f}" is empty or has issues`);
                        }
                        options = Object.assign(Object.assign({}, options), config);
                    }
                    catch (err) {
                        msg_1.message.error(err.message, f, false);
                    }
                }
                else {
                    const config = fs_1.default.readFileSync(configPath, 'utf8');
                    try {
                        options = Object.assign(Object.assign({}, options), JSON.parse(config));
                    }
                    catch (err) {
                        msg_1.message.error(err.message, f, false);
                    }
                }
                if (options.ignorePattern)
                    options.ignorePattern = new RegExp(options.ignorePattern);
                break loop;
            }
        }
    }
    // some small adjustments
    if (options.root)
        options.root = options.root.replace(/^\/+/, '');
    if (options.open === 'true')
        options.open = true;
    if (options.open === 'false')
        options.open = false;
    if (options.https === 'true')
        options.https = true;
    return options;
};
exports.getConfigFile = getConfigFile;
const donate = () => {
    // show only 2% of the time
    if (Math.random() > 0.02)
        return;
    msg_1.message.log('');
    msg_1.message.log(`        ${colors_1.colors('Thank you for using ', 'gray')}Five Server${colors_1.colors('!', 'gray')}`);
    msg_1.message.log(`     ${colors_1.colors('Please consider supporting it on GitHub.', 'gray')}`);
    msg_1.message.log('');
    msg_1.message.log(`    ${colors_1.colors('Donate: https://github.com/sponsors/yandeu', 'yellow')}`);
    msg_1.message.log('');
};
exports.donate = donate;
