'use strict';

var core = require('./core.js');
var fs = require('fs');
require('semver');
require('kleur');
require('@sidvind/better-ajv-errors');
require('ajv');
var deepmerge = require('deepmerge');
require('json-merge-patch');
var path = require('path');
var glob = require('glob');
var ignore = require('ignore');
var prompts = require('prompts');
require('@babel/code-frame');
require('@html-validate/stylish');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var fs__default = /*#__PURE__*/_interopDefaultLegacy(fs);
var deepmerge__default = /*#__PURE__*/_interopDefaultLegacy(deepmerge);
var path__default = /*#__PURE__*/_interopDefaultLegacy(path);
var glob__default = /*#__PURE__*/_interopDefaultLegacy(glob);
var ignore__default = /*#__PURE__*/_interopDefaultLegacy(ignore);
var prompts__default = /*#__PURE__*/_interopDefaultLegacy(prompts);

const DEFAULT_EXTENSIONS = ["html"];
function isDirectory(filename) {
    const st = fs__default['default'].statSync(filename);
    return st.isDirectory();
}
function join(stem, filename) {
    if (path__default['default'].isAbsolute(filename)) {
        return path__default['default'].normalize(filename);
    }
    else {
        return path__default['default'].normalize(path__default['default'].join(stem, filename));
    }
}
function directoryPattern(extensions) {
    switch (extensions.length) {
        case 0:
            return "**";
        case 1:
            return path__default['default'].join("**", `*.${extensions[0]}`);
        default:
            return path__default['default'].join("**", `*.{${extensions.join(",")}}`);
    }
}
/**
 * Takes a number of file patterns (globs) and returns array of expanded
 * filenames.
 */
function expandFiles(patterns, options) {
    const cwd = options.cwd || process.cwd();
    const extensions = options.extensions || DEFAULT_EXTENSIONS;
    const files = patterns.reduce((result, pattern) => {
        /* process - as standard input */
        if (pattern === "-") {
            result.push("/dev/stdin");
            return result;
        }
        for (const filename of glob__default['default'].sync(pattern, { cwd })) {
            /* if file is a directory recursively expand files from it */
            const fullpath = join(cwd, filename);
            if (isDirectory(fullpath)) {
                const dir = expandFiles([directoryPattern(extensions)], Object.assign(Object.assign({}, options), { cwd: fullpath }));
                result = result.concat(dir.map((cur) => join(filename, cur)));
                continue;
            }
            result.push(fullpath);
        }
        return result;
    }, []);
    /* only return unique matches */
    return Array.from(new Set(files));
}

function wrap(formatter, dst) {
    return (results) => {
        const output = formatter(results);
        if (dst) {
            const dir = path__default['default'].dirname(dst);
            if (!fs__default['default'].existsSync(dir)) {
                fs__default['default'].mkdirSync(dir, { recursive: true });
            }
            fs__default['default'].writeFileSync(dst, output, "utf-8");
            return "";
        }
        else {
            return output;
        }
    };
}
function loadFormatter(name) {
    const fn = core.getFormatter(name);
    if (fn) {
        return fn;
    }
    try {
        return core.legacyRequire(name);
    }
    catch (error) {
        throw new core.UserError(`No formatter named "${name}"`, error);
    }
}
function getFormatter(formatters) {
    const fn = formatters.split(",").map((cur) => {
        const [name, dst] = cur.split("=", 2);
        const fn = loadFormatter(name);
        return wrap(fn, dst);
    });
    return (report) => {
        return fn
            .map((formatter) => formatter(report.results))
            .filter(Boolean)
            .join("\n");
    };
}

class IsIgnored {
    constructor() {
        this.cacheIgnore = new Map();
    }
    /**
     * Searches ".htmlvalidateignore" files from filesystem and returns `true` if
     * one of them contains a pattern matching given filename.
     */
    isIgnored(filename) {
        return this.match(filename);
    }
    /**
     * Clear cache
     */
    clearCache() {
        this.cacheIgnore.clear();
    }
    match(target) {
        let current = path__default['default'].dirname(target);
        // eslint-disable-next-line no-constant-condition
        while (true) {
            const relative = path__default['default'].relative(current, target);
            const filename = path__default['default'].join(current, ".htmlvalidateignore");
            /* test filename (relative to the ignore file) against the patterns */
            const ig = this.parseFile(filename);
            if (ig && ig.ignores(relative)) {
                return true;
            }
            /* get the parent directory */
            const child = current;
            current = path__default['default'].dirname(current);
            /* stop if this is the root directory */
            if (current === child) {
                break;
            }
        }
        return false;
    }
    parseFile(filename) {
        if (this.cacheIgnore.has(filename)) {
            return this.cacheIgnore.get(filename);
        }
        if (!fs__default['default'].existsSync(filename)) {
            this.cacheIgnore.set(filename, undefined);
            return undefined;
        }
        const content = fs__default['default'].readFileSync(filename, "utf-8");
        const ig = ignore__default['default']().add(content);
        this.cacheIgnore.set(filename, ig);
        return ig;
    }
}

var Frameworks;
(function (Frameworks) {
    Frameworks["angularjs"] = "AngularJS";
    Frameworks["vuejs"] = "Vue.js";
    Frameworks["markdown"] = "Markdown";
})(Frameworks || (Frameworks = {}));
const frameworkConfig = {
    [Frameworks.angularjs]: {
        transform: {
            "^.*\\.js$": "html-validate-angular/js",
            "^.*\\.html$": "html-validate-angular/html",
        },
    },
    [Frameworks.vuejs]: {
        plugins: ["html-validate-vue"],
        extends: ["html-validate-vue:recommended"],
        transform: {
            "^.*\\.vue$": "html-validate-vue",
        },
    },
    [Frameworks.markdown]: {
        transform: {
            "^.*\\.md$": "html-validate-markdown",
        },
    },
};
function addFrameworks(src, frameworks) {
    let config = src;
    for (const framework of frameworks) {
        config = deepmerge__default['default'](config, frameworkConfig[framework]);
    }
    return config;
}
function writeConfig(dst, config) {
    return new Promise((resolve, reject) => {
        fs__default['default'].writeFile(dst, JSON.stringify(config, null, 2), (err) => {
            if (err)
                reject(err);
            resolve();
        });
    });
}
async function init(cwd) {
    const filename = `${cwd}/.htmlvalidate.json`;
    const exists = fs__default['default'].existsSync(filename);
    const initialConfig = {
        elements: ["html5"],
        extends: ["html-validate:recommended"],
    };
    /* confirm overwrite */
    if (exists) {
        const result = await prompts__default['default']({
            name: "overwrite",
            type: "confirm",
            message: "A .htmlvalidate.json file already exists, do you want to overwrite it?",
        });
        if (!result.overwrite) {
            return Promise.reject();
        }
    }
    const questions = [
        {
            name: "frameworks",
            type: "multiselect",
            choices: [
                { title: Frameworks.angularjs, value: Frameworks.angularjs },
                { title: Frameworks.vuejs, value: Frameworks.vuejs },
                { title: Frameworks.markdown, value: Frameworks.markdown },
            ],
            message: "Support additional frameworks?",
        },
    ];
    /* prompt user for questions */
    const answers = await prompts__default['default'](questions);
    /* write configuration to file */
    let config = initialConfig;
    config = addFrameworks(config, answers.frameworks);
    await writeConfig(filename, config);
    return {
        filename,
    };
}

const defaultConfig = {
    extends: ["html-validate:recommended"],
};
class CLI {
    /**
     * Create new CLI helper.
     *
     * Can be used to create tooling with similar properties to bundled CLI
     * script.
     */
    constructor(options) {
        this.options = options || {};
        this.config = this.getConfig();
        this.ignored = new IsIgnored();
    }
    /**
     * Returns list of files matching patterns and are not ignored. Filenames will
     * have absolute paths.
     *
     * @public
     */
    expandFiles(patterns, options = {}) {
        return expandFiles(patterns, options).filter((filename) => !this.isIgnored(filename));
    }
    getFormatter(formatters) {
        return getFormatter(formatters);
    }
    /**
     * Initialize project with a new configuration.
     *
     * A new `.htmlvalidate.json` file will be placed in the path provided by
     * `cwd`.
     */
    init(cwd) {
        return init(cwd);
    }
    /**
     * Searches ".htmlvalidateignore" files from filesystem and returns `true` if
     * one of them contains a pattern matching given filename.
     */
    isIgnored(filename) {
        return this.ignored.isIgnored(filename);
    }
    /**
     * Clear cache.
     *
     * Previously fetched [[HtmlValidate]] instances must either be fetched again
     * or call [[HtmlValidate.flushConfigCache]].
     */
    clearCache() {
        this.ignored.clearCache();
    }
    /**
     * Get HtmlValidate instance with configuration based on options passed to the
     * constructor.
     */
    getValidator() {
        return new core.HtmlValidate(this.config);
    }
    /**
     * @internal
     */
    getConfig() {
        const { options } = this;
        const config = options.configFile
            ? JSON.parse(fs.readFileSync(options.configFile, "utf-8"))
            : defaultConfig;
        let rules = options.rules;
        if (rules) {
            if (Array.isArray(rules)) {
                rules = rules.join(",");
            }
            const raw = rules
                .split(",")
                .map((x) => x.replace(/ *(.*):/, '"$1":'))
                .join(",");
            try {
                const rules = JSON.parse(`{${raw}}`);
                config.extends = [];
                config.rules = rules;
            }
            catch (e) {
                // istanbul ignore next
                throw new core.UserError(`Error while parsing --rule option "{${raw}}": ${e.message}.\n`);
            }
        }
        return config;
    }
}

const jsonFiltered = ["parent", "children", "meta"];
/* eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types */
function eventReplacer(key, value) {
    if (value && key === "location") {
        return `${value.filename}:${value.line}:${value.column}`;
    }
    return jsonFiltered.includes(key) ? "[truncated]" : value;
}
function eventFormatter(entry) {
    const strdata = JSON.stringify(entry.data, eventReplacer, 2);
    return `${entry.event}: ${strdata}`;
}

exports.CLI = CLI;
exports.eventFormatter = eventFormatter;
//# sourceMappingURL=cli.js.map
