import { SchemaObject, ErrorObject } from 'ajv';
import { SchemaObject as SchemaObject$1 } from 'ajv/dist/types';

/** @internal */
declare type EventCallback = (event: string, data: any) => void;
declare class EventHandler {
    listeners: {
        [event: string]: EventCallback[];
    };
    constructor();
    /**
     * Add an event listener.
     *
     * @param event - Event names (comma separated) or '*' for any event.
     * @param callback - Called any time even triggers.
     * @returns Unregistration function.
     */
    on(event: string, callback: EventCallback): () => void;
    /**
     * Add a onetime event listener. The listener will automatically be removed
     * after being triggered once.
     *
     * @param event - Event names (comma separated) or '*' for any event.
     * @param callback - Called any time even triggers.
     * @returns Unregistration function.
     */
    once(event: string, callback: EventCallback): () => void;
    /**
     * Trigger event causing all listeners to be called.
     *
     * @param event - Event name.
     * @param data - Event data.
     */
    trigger(event: string, data: any): void;
}

declare enum TokenType {
    UNICODE_BOM = 1,
    WHITESPACE = 2,
    NEWLINE = 3,
    DOCTYPE_OPEN = 4,
    DOCTYPE_VALUE = 5,
    DOCTYPE_CLOSE = 6,
    TAG_OPEN = 7,
    TAG_CLOSE = 8,
    ATTR_NAME = 9,
    ATTR_VALUE = 10,
    TEXT = 11,
    TEMPLATING = 12,
    SCRIPT = 13,
    COMMENT = 14,
    CONDITIONAL = 15,
    DIRECTIVE = 16,
    EOF = 17
}
interface Token {
    type: TokenType;
    location: Location;
    data?: any;
}

declare type TokenStream = IterableIterator<Token>;

declare type RuleSeverity = "off" | "warn" | "error" | number;
declare type RuleOptions = string | number | Record<string, any>;
declare type RuleConfig = Record<string, RuleSeverity | [RuleSeverity] | [RuleSeverity, RuleOptions]>;
interface TransformMap {
    [key: string]: string;
}
interface ConfigData {
    /**
     * If set to true no new configurations will be searched.
     */
    root?: boolean;
    extends?: string[];
    /**
     * List of sources for element metadata.
     *
     * The following sources are allowed:
     *
     * - "html5" (default) for the builtin metadata.
     * - node module which export metadata
     * - local path to json or js file exporting metadata.
     * - object with inline metadata
     *
     * If elements isn't specified it defaults to `["html5"]`
     */
    elements?: Array<string | Record<string, unknown>>;
    /**
     * List of plugins.
     *
     * Each plugin must be resolvable be require and export the plugin interface.
     */
    plugins?: string[];
    /**
     * List of source file transformations. A transformer takes a filename and
     * returns Source instances with extracted HTML-templates.
     *
     * Example:
     *
     * ```js
     * "transform": {
     *   "^.*\\.foo$": "my-transform"
     * }
     * ```
     *
     * To run the "my-transform" module on all .foo files.
     */
    transform?: TransformMap;
    rules?: RuleConfig;
}

/**
 * Reported error message.
 */
interface Message {
    /** Rule that triggered this message */
    ruleId: string;
    /** URL to description of error */
    ruleUrl?: string;
    /** Severity of the message */
    severity: number;
    /** Message text */
    message: string;
    /** Offset (number of characters) into the source */
    offset: number;
    /** Line number */
    line: number;
    /** Column number */
    column: number;
    /** From start offset, how many characters is this message relevant for */
    size: number;
    /** DOM selector */
    selector: string | null;
    /**
     * Optional error context used to provide context-aware documentation.
     *
     * This context can be passed to [[HtmlValidate#getRuleDocumentation]].
     */
    context?: any;
}
interface Result {
    messages: Message[];
    filePath: string;
    errorCount: number;
    warningCount: number;
    source: string | null;
}
/**
 * Report object returned by [[HtmlValidate]].
 */
interface Report {
    /** `true` if validation was successful */
    valid: boolean;
    /** Detailed results per validated source */
    results: Result[];
    /** Total number of errors across all sources */
    errorCount: number;
    /** Total warnings of errors across all sources */
    warningCount: number;
}
declare class Reporter {
    protected result: {
        [filename: string]: Message[];
    };
    constructor();
    /**
     * Merge two or more reports into a single one.
     */
    static merge(reports: Report[]): Report;
    add<ContextType, OptionsType>(rule: Rule<ContextType, OptionsType>, message: string, severity: number, node: DOMNode | null, location: Location, context?: ContextType): void;
    addManual(filename: string, message: Message): void;
    save(sources?: Source[]): Report;
    protected isValid(): boolean;
}

interface RuleDocumentation {
    description: string;
    url?: string;
}
interface RuleConstructor<T, U> {
    new (options?: any): Rule<T, U>;
    schema(): SchemaObject | null | undefined;
}
interface IncludeExcludeOptions {
    include: string[] | null;
    exclude: string[] | null;
}
declare abstract class Rule<ContextType = void, OptionsType = void> {
    private reporter;
    private parser;
    private meta;
    private enabled;
    private severity;
    private event;
    /**
     * Rule name. Defaults to filename without extension but can be overwritten by
     * subclasses.
     */
    name: string;
    /**
     * Rule options.
     */
    readonly options: OptionsType;
    constructor(options: OptionsType);
    getSeverity(): number;
    setServerity(severity: number): void;
    setEnabled(enabled: boolean): void;
    /**
     * Returns `true` if rule is deprecated.
     *
     * Overridden by subclasses.
     */
    get deprecated(): boolean;
    /**
     * Test if rule is enabled.
     *
     * To be considered enabled the enabled flag must be true and the severity at
     * least warning.
     */
    isEnabled(): boolean;
    /**
     * Check if keyword is being ignored by the current rule configuration.
     *
     * This method requires the [[RuleOption]] type to include two properties:
     *
     * - include: string[] | null
     * - exclude: string[] | null
     *
     * This methods checks if the given keyword is included by "include" but not
     * excluded by "exclude". If any property is unset it is skipped by the
     * condition. Usually the user would use either one but not both but there is
     * no limitation to use both but the keyword must satisfy both conditions. If
     * either condition fails `true` is returned.
     *
     * For instance, given `{ include: ["foo"] }` the keyword `"foo"` would match
     * but not `"bar"`.
     *
     * Similarly, given `{ exclude: ["foo"] }` the keyword `"bar"` would match but
     * not `"foo"`.
     *
     * @param keyword - Keyword to match against `include` and `exclude` options.
     * @returns `true` if keyword is not present in `include` or is present in
     * `exclude`.
     */
    isKeywordIgnored<T extends IncludeExcludeOptions>(this: {
        options: T;
    }, keyword: string): boolean;
    /**
     * Find all tags which has enabled given property.
     */
    getTagsWithProperty(propName: MetaLookupableProperty): string[];
    /**
     * Find tag matching tagName or inheriting from it.
     */
    getTagsDerivedFrom(tagName: string): string[];
    /**
     * JSON schema for rule options.
     *
     * Rules should override this to return an object with JSON schema to validate
     * rule options. If `null` or `undefined` is returned no validation is
     * performed.
     */
    static schema(): SchemaObject | null | undefined;
    /**
     * Report a new error.
     *
     * Rule must be enabled both globally and on the specific node for this to
     * have any effect.
     */
    report(node: DOMNode | null, message: string, location?: Location | null, context?: ContextType): void;
    private findLocation;
    /**
     * Listen for events.
     *
     * Adding listeners can be done even if the rule is disabled but for the
     * events to be delivered the rule must be enabled.
     *
     * If the optional filter callback is used it must be a function taking an
     * event of the same type as the listener. The filter is called before the
     * listener and if the filter returns false the event is discarded.
     *
     * @param event - Event name
     * @param filter - Optional filter function. Callback is only called if filter functions return true.
     * @param callback - Callback to handle event.
     * @returns A function to unregister the listener
     */
    on<K extends keyof ListenEventMap>(event: K, callback: (event: ListenEventMap[K]) => void): () => void;
    on<K extends keyof ListenEventMap>(event: K, filter: (event: ListenEventMap[K]) => boolean, callback: (event: ListenEventMap[K]) => void): () => void;
    /**
     * Called by [[Engine]] when initializing the rule.
     *
     * Do not override this, use the `setup` callback instead.
     *
     * @internal
     */
    init(parser: Parser, reporter: Reporter, severity: number, meta: MetaTable): void;
    /**
     * Validate rule options against schema. Throws error if object does not validate.
     *
     * For rules without schema this function does nothing.
     *
     * @throws {@link SchemaValidationError}
     * Thrown when provided options does not validate against rule schema.
     *
     * @param cls - Rule class (constructor)
     * @param ruleId - Rule identifier
     * @param jsonPath - JSON path from which [[options]] can be found in [[config]]
     * @param options - User configured options to be validated
     * @param filename - Filename from which options originated
     * @param config - Configuration from which options originated
     *
     * @internal
     */
    static validateOptions(cls: RuleConstructor<unknown, unknown> | undefined, ruleId: string, jsonPath: string, options: unknown, filename: string | null, config: ConfigData): void;
    /**
     * Rule setup callback.
     *
     * Override this to provide rule setup code.
     */
    abstract setup(): void;
    /**
     * Rule documentation callback.
     *
     * Called when requesting additional documentation for a rule. Some rules
     * provide additional context to provide context-aware suggestions.
     *
     * @param context - Error context given by a reported error.
     * @returns Rule documentation and url with additional details or `null` if no
     * additional documentation is available.
     */
    documentation(context?: ContextType): RuleDocumentation | null;
}

/**
 * @internal
 */
interface Event {
    /** Event location. */
    location: Location | null;
}
/**
 * Configuration ready event.
 */
interface ConfigReadyEvent extends Event {
    config: ConfigData;
    rules: {
        [ruleId: string]: Rule;
    };
}
/**
 * Source ready event. Emitted after source has been transformed but before any
 * markup is processed.
 *
 * The source object must not be modified (use a transformer if modifications
 * are required)
 */
interface SourceReadyEvent extends Event {
    source: Source;
}
/**
 * Token event.
 */
interface TokenEvent extends Event {
    type: TokenType;
    data?: any;
}
/**
 * Event emitted when starting tags are encountered.
 */
interface TagStartEvent extends Event {
    /** Event location. */
    location: Location;
    /** The node being started. */
    target: HtmlElement;
}
/** Deprecated alias for TagStartEvent */
declare type TagOpenEvent = TagStartEvent;
/**
 * Event emitted when end tags `</..>` are encountered.
 */
interface TagEndEvent extends Event {
    /** Event location. */
    location: Location;
    /** Temporary node for the end tag. Can be null for elements left unclosed
     * when document ends */
    target: HtmlElement | null;
    /** The node being closed. */
    previous: HtmlElement;
}
/** Deprecated alias for TagEndEvent */
declare type TagCloseEvent = TagEndEvent;
/**
 * Event emitted when a tag is ready (i.e. all the attributes has been
 * parsed). The children of the element will not yet be finished.
 */
interface TagReadyEvent extends Event {
    /** Event location. */
    location: Location;
    /** The node that is finished parsing. */
    target: HtmlElement;
}
/**
 * Event emitted when an element is fully constructed (including its children).
 */
interface ElementReadyEvent extends Event {
    /** Event location. */
    location: Location;
    /** HTML element */
    target: HtmlElement;
}
/**
 * Event emitted when attributes are encountered.
 */
interface AttributeEvent extends Event {
    /** Location of the full attribute (key, quotes and value) */
    location: Location;
    /** Attribute name. */
    key: string;
    /** Attribute value. */
    value: string | DynamicValue | null;
    /** Quotemark used. */
    quote: '"' | "'" | null;
    /** Set to original attribute when a transformer dynamically added this
     * attribute. */
    originalAttribute?: string;
    /** HTML element this attribute belongs to. */
    target: HtmlElement;
    /** Location of the attribute key */
    keyLocation: Location;
    /** Location of the attribute value */
    valueLocation: Location | null;
}
/**
 * Event emitted when whitespace content is parsed.
 */
interface WhitespaceEvent extends Event {
    /** Event location. */
    location: Location;
    /** Text content. */
    text: string;
}
/**
 * Event emitted when Internet Explorer conditionals `<![if ...]>` are
 * encountered.
 */
interface ConditionalEvent extends Event {
    /** Event location. */
    location: Location;
    /** Condition including markers. */
    condition: string;
}
/**
 * Event emitted when html-validate directives `<!-- [html-validate-...] -->`
 * are encountered.
 */
interface DirectiveEvent extends Event {
    /** Event location. */
    location: Location;
    /** Directive action. */
    action: string;
    /** Directive options. */
    data: string;
    /** Directive comment. */
    comment: string;
}
/**
 * Event emitted when doctypes `<!DOCTYPE ..>` are encountered.
 */
interface DoctypeEvent extends Event {
    /** Event location. */
    location: Location;
    /** Tag */
    tag: string;
    /** Selected doctype */
    value: string;
    /** Location of doctype value */
    valueLocation: Location;
}
/**
 * Event emitted after initialization but before tokenization and parsing occurs.
 * Can be used to initialize state in rules.
 */
interface DOMLoadEvent extends Event {
    source: Source;
}
/**
 * Event emitted when DOM tree is fully constructed.
 */
interface DOMReadyEvent extends Event {
    /** DOM Tree */
    document: DOMTree;
    source: Source;
}
interface TriggerEventMap {
    "config:ready": ConfigReadyEvent;
    "source:ready": SourceReadyEvent;
    token: TokenEvent;
    "tag:start": TagStartEvent;
    "tag:end": TagEndEvent;
    "tag:ready": TagReadyEvent;
    "element:ready": ElementReadyEvent;
    "dom:load": DOMLoadEvent;
    "dom:ready": DOMReadyEvent;
    doctype: DoctypeEvent;
    attr: AttributeEvent;
    whitespace: WhitespaceEvent;
    conditional: ConditionalEvent;
    directive: DirectiveEvent;
}
interface ListenEventMap {
    "config:ready": ConfigReadyEvent;
    "source:ready": SourceReadyEvent;
    token: TokenEvent;
    "tag:open": TagOpenEvent;
    "tag:start": TagStartEvent;
    "tag:close": TagCloseEvent;
    "tag:end": TagEndEvent;
    "tag:ready": TagReadyEvent;
    "element:ready": ElementReadyEvent;
    "dom:load": DOMLoadEvent;
    "dom:ready": DOMReadyEvent;
    doctype: DoctypeEvent;
    attr: AttributeEvent;
    whitespace: WhitespaceEvent;
    conditional: ConditionalEvent;
    directive: DirectiveEvent;
    "*": Event;
}

/**
 * Parse HTML document into a DOM tree.
 */
declare class Parser {
    private readonly event;
    private readonly metaTable;
    private dom;
    /**
     * Create a new parser instance.
     *
     * @param config - Configuration
     */
    constructor(config: ResolvedConfig);
    /**
     * Parse HTML markup.
     *
     * @param source - HTML markup.
     * @returns DOM tree representing the HTML markup.
     */
    parseHtml(source: string | Source): HtmlElement;
    /**
     * Detect optional end tag.
     *
     * Some tags have optional end tags (e.g. <ul><li>foo<li>bar</ul> is
     * valid). The parser handles this by checking if the element on top of the
     * stack when is allowed to omit.
     */
    private closeOptional;
    protected consumeTag(source: Source, startToken: Token, tokenStream: TokenStream): void;
    protected closeElement(source: Source, node: HtmlElement | null, active: HtmlElement, location: Location): void;
    private processElement;
    /**
     * Discard tokens until the end tag for the foreign element is found.
     */
    protected discardForeignBody(source: Source, foreignTagName: string, tokenStream: TokenStream, errorLocation: Location): void;
    protected consumeAttribute(source: Source, node: HtmlElement, token: Token, next?: Token): void;
    /**
     * Takes attribute key token an returns location.
     */
    private getAttributeKeyLocation;
    /**
     * Take attribute value token and return a new location referring to only the
     * value.
     *
     * foo="bar"    foo='bar'    foo=bar    foo      foo=""
     *      ^^^          ^^^         ^^^    (null)   (null)
     */
    private getAttributeValueLocation;
    /**
     * Take attribute key and value token an returns a new location referring to
     * an aggregate location covering key, quotes if present and value.
     */
    private getAttributeLocation;
    protected consumeDirective(token: Token): void;
    /**
     * Consumes comment token.
     *
     * Tries to find IE conditional comments and emits conditional token if found.
     */
    protected consumeComment(token: Token): void;
    /**
     * Consumes doctype tokens. Emits doctype event.
     */
    protected consumeDoctype(startToken: Token, tokenStream: TokenStream): void;
    /**
     * Return a list of tokens found until the expected token was found.
     *
     * @param errorLocation - What location to use if an error occurs
     */
    protected consumeUntil(tokenStream: TokenStream, search: TokenType, errorLocation: Location): IterableIterator<Token>;
    private next;
    /**
     * Listen on events.
     *
     * @param event - Event name.
     * @param listener - Event callback.
     * @returns A function to unregister the listener.
     */
    on<K extends keyof ListenEventMap>(event: K, listener: (event: string, data: ListenEventMap[K]) => void): () => void;
    on(event: string, listener: EventCallback): () => void;
    /**
     * Listen on single event. The listener is automatically unregistered once the
     * event has been received.
     *
     * @param event - Event name.
     * @param listener - Event callback.
     * @returns A function to unregister the listener.
     */
    once<K extends keyof ListenEventMap>(event: K, listener: (event: string, data: ListenEventMap[K]) => void): () => void;
    once(event: string, listener: EventCallback): () => void;
    /**
     * Defer execution. Will call function sometime later.
     *
     * @param cb - Callback to execute later.
     */
    defer(cb: () => void): void;
    /**
     * Trigger event.
     *
     * @param event - Event name
     * @param data - Event data
     */
    trigger<K extends keyof TriggerEventMap>(event: K, data: TriggerEventMap[K]): void;
    /**
     * @internal
     */
    getEventHandler(): EventHandler;
    /**
     * Appends a text node to the current element on the stack.
     */
    private appendText;
    /**
     * Trigger close events for any still open elements.
     */
    private closeTree;
}

/**
 * Raw attribute data.
 */
interface AttributeData {
    /** Attribute name */
    key: string;
    /** Attribute value */
    value: string | DynamicValue | null;
    /** Quotation mark (if present) */
    quote: '"' | "'" | null;
    /** Original attribute name (when a dynamic attribute is used), e.g
     * "ng-attr-foo" or "v-bind:foo" */
    originalAttribute?: string;
}

declare type ProcessAttributeCallback = (this: unknown, attr: AttributeData) => Iterable<AttributeData>;
interface ProcessElementContext {
    getMetaFor(tagName: string): MetaElement | null;
}
declare type ProcessElementCallback = (this: ProcessElementContext, node: HtmlElement) => void;
interface SourceHooks {
    /**
     * Called for every attribute.
     *
     * The original attribute must be yielded as well or no attribute will be
     * added.
     *
     * @returns Attribute data for an attribute to be added to the element.
     */
    processAttribute?: ProcessAttributeCallback | null;
    /**
     * Called for every element after element is created but before any children.
     *
     * May modify the element.
     */
    processElement?: ProcessElementCallback | null;
}
/**
 * Source interface.
 *
 * HTML source with file, line and column context.
 *
 * Optional hooks can be attached. This is usually added by transformers to
 * postprocess.
 */
interface Source {
    data: string;
    filename: string;
    /**
     * Line in the original data.
     *
     * Starts at 1 (first line).
     */
    line: number;
    /**
     * Column in the original data.
     *
     * Starts at 1 (first column).
     */
    column: number;
    /**
     * Offset in the original data.
     *
     * Starts at 0 (first character).
     */
    offset: number;
    /**
     * Original data. When a transformer extracts a portion of the original source
     * this must be set to the full original source.
     *
     * Since the transformer might be chained always test if the input source
     * itself has `originalData` set, e.g.:
     *
     * `originalData = input.originalData || input.data`.
     */
    originalData?: string;
    /**
     * Hooks for processing the source as it is being parsed.
     */
    hooks?: SourceHooks;
    /**
     * Internal property to keep track of what transformers has run on this
     * source. Entries are in reverse-order, e.g. the last applied transform is
     * first.
     */
    transformedBy?: string[];
}

interface Location {
    /**
     * The filemane this location refers to.
     */
    readonly filename: string;
    /**
     * The string offset (number of characters into the string) this location
     * refers to.
     */
    readonly offset: number;
    /**
     * The line number in the file.
     */
    readonly line: number;
    /**
     * The column number in the file. Tabs counts as 1 (not expanded).
     */
    readonly column: number;
    /**
     * The number of characters this location refers to. This includes any
     * whitespace characters such as newlines.
     */
    readonly size: number;
}

declare class DynamicValue {
    readonly expr: string;
    constructor(expr: string);
    toString(): string;
}

/**
 * DOM Attribute.
 *
 * Represents a HTML attribute. Can contain either a fixed static value or a
 * placeholder for dynamic values (e.g. interpolated).
 */
declare class Attribute {
    /** Attribute name */
    readonly key: string;
    readonly value: string | DynamicValue | null;
    readonly keyLocation: Location;
    readonly valueLocation: Location | null;
    readonly originalAttribute?: string;
    /**
     * @param key - Attribute name.
     * @param value - Attribute value. Set to `null` for boolean attributes.
     * @param keyLocation - Source location of attribute name.
     * @param valueLocation - Source location of attribute value.
     * @param originalAttribute - If this attribute was dynamically added via a
     * transformation (e.g. vuejs `:id` generating the `id` attribute) this
     * parameter should be set to the attribute name of the source attribute (`:id`).
     */
    constructor(key: string, value: null | string | DynamicValue | null, keyLocation: Location, valueLocation: Location | null, originalAttribute?: string);
    /**
     * Flag set to true if the attribute value is static.
     */
    get isStatic(): boolean;
    /**
     * Flag set to true if the attribute value is dynamic.
     */
    get isDynamic(): boolean;
    /**
     * Test attribute value.
     *
     * @param pattern - Pattern to match value against. RegExp or a string (===)
     * @param dynamicMatches - If true `DynamicValue` will always match, if false
     * it never matches.
     * @returns `true` if attribute value matches pattern.
     */
    valueMatches(pattern: RegExp, dynamicMatches?: boolean): boolean;
    valueMatches(pattern: string, dynamicMatches?: boolean): boolean;
}

interface PermittedGroup {
    exclude?: string | string[];
}
declare type PropertyExpression = string | [string, any];
declare type PermittedEntry = string | any[] | PermittedGroup;
declare type Permitted = PermittedEntry[];
declare type PermittedOrder = string[];
declare type RequiredAncestors = string[];
declare type RequiredContent = string[];
declare enum TextContent {
    NONE = "none",
    DEFAULT = "default",
    REQUIRED = "required",
    ACCESSIBLE = "accessible"
}
interface PermittedAttribute {
    [key: string]: Array<string | RegExp>;
}
interface DeprecatedElement {
    message?: string;
    documentation?: string;
    source?: string;
}
interface MetaData {
    inherit?: string;
    metadata?: boolean | PropertyExpression;
    flow?: boolean | PropertyExpression;
    sectioning?: boolean | PropertyExpression;
    heading?: boolean | PropertyExpression;
    phrasing?: boolean | PropertyExpression;
    embedded?: boolean | PropertyExpression;
    interactive?: boolean | PropertyExpression;
    deprecated?: boolean | string | DeprecatedElement;
    foreign?: boolean;
    void?: boolean;
    transparent?: boolean | string[];
    implicitClosed?: string[];
    scriptSupporting?: boolean;
    form?: boolean;
    labelable?: boolean | PropertyExpression;
    deprecatedAttributes?: string[];
    requiredAttributes?: string[];
    attributes?: PermittedAttribute;
    permittedContent?: Permitted;
    permittedDescendants?: Permitted;
    permittedOrder?: PermittedOrder;
    requiredAncestors?: RequiredAncestors;
    requiredContent?: RequiredContent;
    textContent?: TextContent;
}
/**
 * Properties listed here can be used to reverse search elements with the given
 * property enabled. See [[MetaTable.getTagsWithProperty]].
 */
declare type MetaLookupableProperty = "metadata" | "flow" | "sectioning" | "heading" | "phrasing" | "embedded" | "interactive" | "deprecated" | "foreign" | "void" | "transparent" | "scriptSupporting" | "form" | "labelable";
/**
 * Properties listed here can be copied (loaded) onto another element using
 * [[HtmlElement.loadMeta]].
 */
declare const MetaCopyableProperty: string[];
interface MetaElement extends MetaData {
    tagName: string;
    [key: string]: undefined | boolean | DeprecatedElement | PropertyExpression | Permitted | PermittedOrder | PermittedAttribute | RequiredAncestors;
}
interface ElementTable {
    [tagName: string]: MetaElement;
}

declare enum NodeType {
    ELEMENT_NODE = 1,
    TEXT_NODE = 3,
    DOCUMENT_NODE = 9
}

interface DOMNodeCache {
}

declare type DOMInternalID = number;
declare const TEXT_CONTENT: unique symbol;
declare module "./cache" {
    interface DOMNodeCache {
        [TEXT_CONTENT]: string;
    }
}
declare class DOMNode {
    readonly nodeName: string;
    readonly nodeType: NodeType;
    readonly childNodes: DOMNode[];
    readonly location: Location;
    readonly unique: DOMInternalID;
    private cache;
    /**
     * Set of disabled rules for this node.
     *
     * Rules disabled by using directives are added here.
     */
    private disabledRules;
    /**
     * Create a new DOMNode.
     *
     * @param nodeType - What node type to create.
     * @param nodeName - What node name to use. For `HtmlElement` this corresponds
     * to the tagName but other node types have specific predefined values.
     * @param location - Source code location of this node.
     */
    constructor(nodeType: NodeType, nodeName: string | undefined, location: Location);
    /**
     * Enable cache for this node.
     *
     * Should not be called before the node and all children are fully constructed.
     */
    cacheEnable(): void;
    /**
     * Fetch cached value from this DOM node.
     *
     * Cache is not enabled until `cacheEnable()` is called by [[Parser]] (when
     * the element is fully constructed).
     *
     * @returns value or `undefined` if the value doesn't exist.
     */
    cacheGet<K extends keyof DOMNodeCache>(key: K): DOMNodeCache[K] | undefined;
    cacheGet(key: string | number | symbol): any | undefined;
    /**
     * Store a value in cache.
     *
     * @returns the value itself is returned.
     */
    cacheSet<K extends keyof DOMNodeCache>(key: K, value: DOMNodeCache[K]): DOMNodeCache[K];
    cacheSet<T>(key: string | number | symbol, value: T): T;
    /**
     * Remove a value by key from cache.
     *
     * @returns `true` if the entry existed and has been removed.
     */
    cacheRemove<K extends keyof DOMNodeCache>(key: K): boolean;
    cacheRemove(key: string | number | symbol): boolean;
    /**
     * Check if key exists in cache.
     */
    cacheExists<K extends keyof DOMNodeCache>(key: K): boolean;
    cacheExists(key: string | number | symbol): boolean;
    /**
     * Get the text (recursive) from all child nodes.
     */
    get textContent(): string;
    append(node: DOMNode): void;
    isRootElement(): boolean;
    /**
     * Tests if two nodes are the same (references the same object).
     */
    isSameNode(otherNode: DOMNode): boolean;
    /**
     * Returns a DOMNode representing the first direct child node or `null` if the
     * node has no children.
     */
    get firstChild(): DOMNode;
    /**
     * Returns a DOMNode representing the last direct child node or `null` if the
     * node has no children.
     */
    get lastChild(): DOMNode;
    /**
     * Disable a rule for this node.
     */
    disableRule(ruleId: string): void;
    /**
     * Disables multiple rules.
     */
    disableRules(rules: string[]): void;
    /**
     * Enable a previously disabled rule for this node.
     */
    enableRule(ruleId: string): void;
    /**
     * Enables multiple rules.
     */
    enableRules(rules: string[]): void;
    /**
     * Test if a rule is enabled for this node.
     */
    ruleEnabled(ruleId: string): boolean;
    generateSelector(): string | null;
}

declare class DOMTokenList extends Array<string> {
    readonly value: string;
    private readonly locations;
    constructor(value: string | DynamicValue | null, location: Location | null);
    item(n: number): string | undefined;
    location(n: number): Location | undefined;
    contains(token: string): boolean;
    iterator(): Generator<{
        index: number;
        item: string;
        location: Location;
    }>;
}

declare enum NodeClosed {
    Open = 0,
    EndTag = 1,
    VoidOmitted = 2,
    VoidSelfClosed = 3,
    ImplicitClosed = 4
}
declare class HtmlElement extends DOMNode {
    readonly tagName: string;
    readonly parent: HtmlElement | null;
    readonly voidElement: boolean;
    readonly depth: number;
    closed: NodeClosed;
    protected readonly attr: {
        [key: string]: Attribute[];
    };
    private metaElement;
    private annotation;
    constructor(tagName: string | undefined, parent: HtmlElement | null, closed: NodeClosed, meta: MetaElement | null, location: Location);
    static rootNode(location: Location): HtmlElement;
    static fromTokens(startToken: Token, endToken: Token, parent: HtmlElement | null, metaTable: MetaTable | null): HtmlElement;
    /**
     * Returns annotated name if set or defaults to `<tagName>`.
     *
     * E.g. `my-annotation` or `<div>`.
     */
    get annotatedName(): string;
    /**
     * Similar to childNodes but only elements.
     */
    get childElements(): HtmlElement[];
    /**
     * Find the first ancestor matching a selector.
     *
     * Implementation of DOM specification of Element.closest(selectors).
     */
    closest(selectors: string): HtmlElement | null;
    /**
     * Generate a DOM selector for this element. The returned selector will be
     * unique inside the current document.
     */
    generateSelector(): string | null;
    /**
     * Tests if this element has given tagname.
     *
     * If passing "*" this test will pass if any tagname is set.
     */
    is(tagName: string): boolean;
    /**
     * Load new element metadata onto this element.
     *
     * Do note that semantics such as `void` cannot be changed (as the element has
     * already been created). In addition the element will still "be" the same
     * element, i.e. even if loading meta for a `<p>` tag upon a `<div>` tag it
     * will still be a `<div>` as far as the rest of the validator is concerned.
     *
     * In fact only certain properties will be copied onto the element:
     *
     * - content categories (flow, phrasing, etc)
     * - required attributes
     * - attribute allowed values
     * - permitted/required elements
     *
     * Properties *not* loaded:
     *
     * - inherit
     * - deprecated
     * - foreign
     * - void
     * - implicitClosed
     * - scriptSupporting
     * - deprecatedAttributes
     *
     * Changes to element metadata will only be visible after `element:ready` (and
     * the subsequent `dom:ready` event).
     */
    loadMeta(meta: MetaElement): void;
    /**
     * Match this element against given selectors. Returns true if any selector
     * matches.
     *
     * Implementation of DOM specification of Element.matches(selectors).
     */
    matches(selector: string): boolean;
    get meta(): MetaElement | null;
    /**
     * Set annotation for this element.
     */
    setAnnotation(text: string): void;
    /**
     * Set attribute. Stores all attributes set even with the same name.
     *
     * @param key - Attribute name
     * @param value - Attribute value. Use `null` if no value is present.
     * @param keyLocation - Location of the attribute name.
     * @param valueLocation - Location of the attribute value (excluding quotation)
     * @param originalAttribute - If attribute is an alias for another attribute
     * (dynamic attributes) set this to the original attribute name.
     */
    setAttribute(key: string, value: string | DynamicValue | null, keyLocation: Location, valueLocation: Location | null, originalAttribute?: string): void;
    /**
     * Get a list of all attributes on this node.
     */
    get attributes(): Attribute[];
    hasAttribute(key: string): boolean;
    /**
     * Get attribute.
     *
     * By default only the first attribute is returned but if the code needs to
     * handle duplicate attributes the `all` parameter can be set to get all
     * attributes with given key.
     *
     * This usually only happens when code contains duplicate attributes (which
     * `no-dup-attr` will complain about) or when a static attribute is combined
     * with a dynamic, consider:
     *
     * <p class="foo" dynamic-class="bar">
     *
     * @param key - Attribute name
     * @param all - Return single or all attributes.
     */
    getAttribute(key: string): Attribute | null;
    getAttribute(key: string, all: true): Attribute[];
    /**
     * Get attribute value.
     *
     * Returns the attribute value if present.
     *
     * - Missing attributes return `null`.
     * - Boolean attributes return `null`.
     * - `DynamicValue` returns attribute expression.
     *
     * @param key - Attribute name
     * @returns Attribute value or null.
     */
    getAttributeValue(key: string): string | null;
    /**
     * Add text as a child node to this element.
     *
     * @param text - Text to add.
     * @param location - Source code location of this text.
     */
    appendText(text: string | DynamicValue, location: Location): void;
    /**
     * Return a list of all known classes on the element. Dynamic values are
     * ignored.
     */
    get classList(): DOMTokenList;
    /**
     * Get element ID if present.
     */
    get id(): string | null;
    /**
     * Returns the first child element or null if there are no child elements.
     */
    get firstElementChild(): HtmlElement | null;
    /**
     * Returns the last child element or null if there are no child elements.
     */
    get lastElementChild(): HtmlElement | null;
    get siblings(): HtmlElement[];
    get previousSibling(): HtmlElement | null;
    get nextSibling(): HtmlElement | null;
    getElementsByTagName(tagName: string): HtmlElement[];
    querySelector(selector: string): HtmlElement;
    querySelectorAll(selector: string): HtmlElement[];
    private querySelectorImpl;
    /**
     * Visit all nodes from this node and down. Depth first.
     */
    visitDepthFirst(callback: (node: HtmlElement) => void): void;
    /**
     * Evaluates callbackk on all descendants, returning true if any are true.
     */
    someChildren(callback: (node: HtmlElement) => boolean): boolean;
    /**
     * Evaluates callbackk on all descendants, returning true if all are true.
     */
    everyChildren(callback: (node: HtmlElement) => boolean): boolean;
    /**
     * Visit all nodes from this node and down. Breadth first.
     *
     * The first node for which the callback evaluates to true is returned.
     */
    find(callback: (node: HtmlElement) => boolean): HtmlElement | null;
}

declare class DOMTree {
    readonly root: HtmlElement;
    private active;
    doctype: string | null;
    constructor(location: Location);
    pushActive(node: HtmlElement): void;
    popActive(): void;
    getActive(): HtmlElement;
    /**
     * Resolve dynamic meta expressions.
     */
    resolveMeta(table: MetaTable): void;
    getElementsByTagName(tagName: string): HtmlElement[];
    visitDepthFirst(callback: (node: HtmlElement) => void): void;
    find(callback: (node: HtmlElement) => boolean): HtmlElement | null;
    querySelector(selector: string): HtmlElement;
    querySelectorAll(selector: string): HtmlElement[];
}

/**
 * Represents a text in the HTML document.
 *
 * Text nodes are appended as children of `HtmlElement` and cannot have childen
 * of its own.
 */
declare class TextNode extends DOMNode {
    private readonly text;
    /**
     * @param text - Text to add. When a `DynamicValue` is used the expression is
     * used as "text".
     * @param location - Source code location of this node.
     */
    constructor(text: string | DynamicValue, location: Location);
    /**
     * Get the text from node.
     */
    get textContent(): string;
    /**
     * Flag set to true if the attribute value is static.
     */
    get isStatic(): boolean;
    /**
     * Flag set to true if the attribute value is dynamic.
     */
    get isDynamic(): boolean;
}

interface TransformContext {
    /**
     * Test if an additional chainable transformer is present.
     *
     * Returns true only if there is a transformer configured for the given
     * filename.
     *
     * @param filename - Filename to use to match next transformer.
     */
    hasChain(filename: string): boolean;
    /**
     * Chain transformations.
     *
     * Sometimes multiple transformers must be applied. For instance, a Markdown
     * file with JSX in a code-block.
     *
     * @param source - Source to chain transformations on.
     * @param filename - Filename to use to match next transformer (unrelated to
     * filename set in source)
     */
    chain(source: Source, filename: string): Iterable<Source>;
}

declare module "estree" {
    interface TemplateElement {
        start: number;
        end: number;
    }
}
declare class TemplateExtractor {
    private ast;
    private filename;
    private data;
    private constructor();
    static fromFilename(filename: string): TemplateExtractor;
    /**
     * Create a new [[TemplateExtractor]] from javascript source code.
     *
     * `Source` offsets will be relative to the string, i.e. offset 0 is the first
     * character of the string. If the string is only a subset of a larger string
     * the offsets must be adjusted manually.
     *
     * @param source - Source code.
     * @param filename - Optional filename to set in the resulting
     * `Source`. Defauls to `"inline"`.
     */
    static fromString(source: string, filename?: string): TemplateExtractor;
    /**
     * Convenience function to create a [[Source]] instance from an existing file.
     *
     * @param filename - Filename with javascript source code. The file must exist
     * and be readable by the user.
     * @returns An array of Source's suitable for passing to [[Engine]] linting
     * functions.
     */
    static createSource(filename: string): Source[];
    /**
     * Extract object properties.
     *
     * Given a key `"template"` this method finds all objects literals with a
     * `"template"` property and creates a [[Source]] instance with proper offsets
     * with the value of the property. For instance:
     *
     * ```
     * const myObj = {
     *   foo: 'bar',
     * };
     * ```
     *
     * The above snippet would yield a `Source` with the content `bar`.
     *
     */
    extractObjectProperty(key: string): Source[];
}

declare type Transformer = (this: TransformContext, source: Source) => Iterable<Source>;

interface SchemaValidationPatch {
    properties?: Record<string, unknown>;
    definitions?: Record<string, unknown>;
}
interface Plugin {
    /**
     * Name of the plugin.
     *
     * If specified this is the name used when referring to the plugin. Default is
     * to use the name/path the user used when loading the plugin. To be less
     * confusing for users you should use the same name as your package.
     *
     * The name must be a valid package name according to NPM (basically lowercase
     * characters, must not begin with dot, slash or non-url safe characters).
     *
     * Hint: import and use the name from `package.json`.
     */
    name?: string | null;
    /**
     * Initialization callback.
     *
     * Called once per plugin during initialization.
     */
    init?: () => void | null;
    /**
     * Setup callback.
     *
     * Called once per source after engine is initialized.
     *
     * @param source - The source about to be validated. Readonly.
     * @param eventhandler - Eventhandler from parser. Can be used to listen for
     * parser events.
     */
    setup?: (source: Source, eventhandler: EventHandler) => void | null;
    /**
     * Configuration presets.
     *
     * Each key should be the unprefixed name which a configuration later can
     * access using `${plugin}:${key}`, e.g. if a plugin named "my-plugin" exposes
     * a preset named "foobar" it can be accessed using:
     *
     * "extends": ["my-plugin:foobar"]
     */
    configs?: Record<string, ConfigData | null> | null;
    /**
     * List of new rules present.
     */
    rules?: Record<string, RuleConstructor<any, any> | null> | null;
    /**
     * Transformer available in this plugin.
     *
     * Can be given either as a single unnamed transformer or an object with
     * multiple named.
     *
     * Unnamed transformers use the plugin name similar to how a standalone
     * transformer would work:
     *
     * ```
     * "transform": {
     *   "^.*\\.foo$": "my-plugin"
     * }
     * ```
     *
     * For named transformers each key should be the unprefixed name which a
     * configuration later can access using `${plugin}:${key}`, e.g. if a plugin
     * named "my-plugin" exposes a transformer named "foobar" it can be accessed
     * using:
     *
     * ```
     * "transform": {
     *   "^.*\\.foo$": "my-plugin:foobar"
     * }
     * ```
     */
    transformer?: Transformer | Record<string, Transformer | null> | null;
    /**
     * Extend metadata validation schema.
     */
    elementSchema?: SchemaValidationPatch | null;
}

declare class MetaTable {
    readonly elements: ElementTable;
    private schema;
    constructor();
    init(): void;
    /**
     * Extend validation schema.
     */
    extendValidationSchema(patch: SchemaValidationPatch): void;
    /**
     * Load metadata table from object.
     *
     * @param obj - Object with metadata to load
     * @param filename - Optional filename used when presenting validation error
     */
    loadFromObject(obj: unknown, filename?: string | null): void;
    /**
     * Load metadata table from filename
     *
     * @param filename - Filename to load
     */
    loadFromFile(filename: string): void;
    /**
     * Get [[MetaElement]] for the given tag or null if the element doesn't exist.
     *
     * @returns A shallow copy of metadata.
     */
    getMetaFor(tagName: string): MetaElement | null;
    /**
     * Find all tags which has enabled given property.
     */
    getTagsWithProperty(propName: MetaLookupableProperty): string[];
    /**
     * Find tag matching tagName or inheriting from it.
     */
    getTagsDerivedFrom(tagName: string): string[];
    private addEntry;
    /**
     * Construct a new AJV schema validator.
     */
    private getSchemaValidator;
    getJSONSchema(): SchemaObject$1;
    /**
     * Finds the global element definition and merges each known element with the
     * global, e.g. to assign global attributes.
     */
    private resolveGlobal;
    private mergeElement;
    resolve(node: HtmlElement): void;
}

declare enum Severity {
    DISABLED = 0,
    WARN = 1,
    ERROR = 2
}

interface TransformerEntry {
    pattern: RegExp;
    name: string;
    fn: Transformer;
}
interface ResolvedConfigData {
    metaTable: MetaTable;
    plugins: Plugin[];
    rules: Map<string, [Severity, RuleOptions]>;
    transformers: TransformerEntry[];
}
/**
 * A resolved configuration is a normalized configuration with all extends,
 * plugins etc resolved.
 */
declare class ResolvedConfig {
    private metaTable;
    private plugins;
    private rules;
    private transformers;
    constructor({ metaTable, plugins, rules, transformers }: ResolvedConfigData);
    getMetaTable(): MetaTable;
    getPlugins(): Plugin[];
    getRules(): Map<string, [Severity, RuleOptions]>;
    /**
     * Transform a source.
     *
     * When transforming zero or more new sources will be generated.
     *
     * @param source - Current source to transform.
     * @param filename - If set it is the filename used to match
     * transformer. Default is to use filename from source.
     * @returns A list of transformed sources ready for validation.
     */
    transformSource(source: Source, filename?: string): Source[];
    /**
     * Wrapper around [[transformSource]] which reads a file before passing it
     * as-is to transformSource.
     *
     * @param source - Filename to transform (according to configured
     * transformations)
     * @returns A list of transformed sources ready for validation.
     */
    transformFilename(filename: string): Source[];
    /**
     * Returns true if a transformer matches given filename.
     */
    canTransform(filename: string): boolean;
    private findTransformer;
}

/**
 * Internal interface for a loaded plugin.
 */
interface LoadedPlugin extends Plugin {
    name: string;
    originalName: string;
}
/**
 * Configuration holder.
 *
 * Each file being validated will have a unique instance of this class.
 */
declare class Config {
    private config;
    private configurations;
    private initialized;
    protected metaTable: MetaTable | null;
    protected plugins: LoadedPlugin[];
    protected transformers: TransformerEntry[];
    protected rootDir: string;
    /**
     * Create a new blank configuration. See also `Config.defaultConfig()`.
     */
    static empty(): Config;
    /**
     * Create configuration from object.
     */
    static fromObject(options: ConfigData, filename?: string | null): Config;
    /**
     * Read configuration from filename.
     *
     * Note: this reads configuration data from a file. If you intent to load
     * configuration for a file to validate use `ConfigLoader.fromTarget()`.
     *
     * @param filename - The file to read from or one of the presets such as
     * `html-validate:recommended`.
     */
    static fromFile(filename: string): Config;
    /**
     * Validate configuration data.
     *
     * Throws SchemaValidationError if invalid.
     */
    static validate(configData: ConfigData, filename?: string | null): void;
    /**
     * Load a default configuration object.
     */
    static defaultConfig(): Config;
    constructor(options?: ConfigData);
    /**
     * Initialize plugins, transforms etc.
     *
     * Must be called before trying to use config. Can safely be called multiple
     * times.
     */
    init(): void;
    /**
     * Returns true if this configuration is marked as "root".
     */
    isRootFound(): boolean;
    /**
     * Returns a new configuration as a merge of the two. Entries from the passed
     * object takes priority over this object.
     *
     * @param rhs - Configuration to merge with this one.
     */
    merge(rhs: Config): Config;
    private extendConfig;
    /**
     * Get element metadata.
     */
    getMetaTable(): MetaTable;
    /**
     * @internal exposed for testing only
     */
    static expandRelative(src: string, currentPath: string): string;
    /**
     * Get a copy of internal configuration data.
     *
     * @internal primary purpose is unittests
     */
    get(): ConfigData;
    /**
     * Get all configured rules, their severity and options.
     */
    getRules(): Map<string, [Severity, RuleOptions]>;
    private static getRulesObject;
    /**
     * Get all configured plugins.
     */
    getPlugins(): Plugin[];
    private loadPlugins;
    private loadConfigurations;
    private extendMeta;
    /**
     * Resolve all configuration and return a [[ResolvedConfig]] instance.
     *
     * A resolved configuration will merge all extended configs and load all
     * plugins and transformers, and normalize the rest of the configuration.
     */
    resolve(): ResolvedConfig;
    /**
     * Same as [[resolve]] but returns the raw configuration data instead of
     * [[ResolvedConfig]] instance. Mainly used for testing.
     *
     * @internal
     */
    resolveData(): ResolvedConfigData;
    private precompileTransformers;
    /**
     * Get transformation function requested by configuration.
     *
     * Searches:
     *
     * - Named transformers from plugins.
     * - Unnamed transformer from plugin.
     * - Standalone modules (local or node_modules)
     *
     * @param name - Key from configuration
     */
    private getTransformFunction;
    /**
     * @param name - Original name from configuration
     * @param pluginName - Name of plugin
     * @param key - Name of transform (from plugin)
     */
    private getNamedTransformerFromPlugin;
    /**
     * @param name - Original name from configuration
     * @param plugin - Plugin instance
     */
    private getUnnamedTransformerFromPlugin;
    private getTransformerFromModule;
    /**
     * @internal
     */
    protected get rootDirCache(): string | null;
    protected set rootDirCache(value: string | null);
    protected findRootDir(): string;
}

/**
 * @internal
 */
interface ConfigFactory {
    defaultConfig(): Config;
    empty(): Config;
    fromObject(options: ConfigData, filename?: string | null): Config;
    fromFile(filename: string): Config;
}
/**
 * Configuration loader interface.
 *
 * A configuration loader takes a handle (typically a filename) and returns a
 * configuration for it.
 *
 * @public
 */
declare abstract class ConfigLoader {
    protected readonly configFactory: ConfigFactory;
    protected readonly globalConfig: Config;
    constructor(config?: ConfigData, configFactory?: ConfigFactory);
    /**
     * Get configuration for given handle.
     *
     * Handle is typically a filename but it is up to the loader to interpret the
     * handle as something useful.
     *
     * If [[configOverride]] is set it is merged with the final result.
     *
     * @param handle - Unique handle to get configuration for.
     * @param configOverride - Optional configuration to merge final results with.
     */
    abstract getConfigFor(handle: string, configOverride?: ConfigData): Config;
    /**
     * Flush configuration cache.
     *
     * Flushes all cached entries unless a specific handle is given.
     *
     * @param handle - If given only the cache for given handle will be flushed.
     */
    abstract flushCache(handle?: string): void;
    /**
     * Default configuration used when no explicit configuration is passed to constructor.
     */
    protected abstract defaultConfig(): Config;
    protected empty(): Config;
    protected loadFromObject(options: ConfigData, filename?: string | null): Config;
    protected loadFromFile(filename: string): Config;
}

declare class NestedError extends Error {
    constructor(message: string, nested?: Error);
}

declare class UserError extends NestedError {
}

declare class SchemaValidationError extends UserError {
    filename: string | null;
    private obj;
    private schema;
    private errors;
    constructor(filename: string | null, message: string, obj: unknown, schema: SchemaObject, errors: ErrorObject[]);
    prettyError(): string;
    private getRawJSON;
}

declare class ConfigError extends UserError {
}

declare const presets: Record<string, ConfigData>;

interface EventDump {
    event: string;
    data: any;
}
interface TokenDump {
    token: string;
    data: string;
    location: string;
}

/**
 * Primary API for using HTML-validate.
 *
 * Provides high-level abstractions for common operations.
 */
declare class HtmlValidate {
    protected configLoader: ConfigLoader;
    /**
     * Create a new validator.
     *
     * @public
     * @param configLoader - Use a custom configuration loader.
     * @param config - If set it provides the global default configuration. By
     * default `Config.defaultConfig()` is used.
     */
    constructor(config?: ConfigData);
    constructor(configLoader: ConfigLoader);
    /**
     * Parse and validate HTML from string.
     *
     * @public
     * @param str - Text to parse.
     * @param filename - If set configuration is loaded for given filename.
     * @param hooks - Optional hooks (see [[Source]]) for definition.
     * @returns Report output.
     */
    validateString(str: string): Report;
    validateString(str: string, filename: string): Report;
    validateString(str: string, hooks: SourceHooks): Report;
    validateString(str: string, options: ConfigData): Report;
    validateString(str: string, filename: string, hooks: SourceHooks): Report;
    validateString(str: string, filename: string, options: ConfigData): Report;
    validateString(str: string, filename: string, options: ConfigData, hooks: SourceHooks): Report;
    /**
     * Parse and validate HTML from [[Source]].
     *
     * @public
     * @param input - Source to parse.
     * @returns Report output.
     */
    validateSource(input: Source, configOverride?: ConfigData): Report;
    /**
     * Parse and validate HTML from file.
     *
     * @public
     * @param filename - Filename to read and parse.
     * @returns Report output.
     */
    validateFile(filename: string): Report;
    /**
     * Parse and validate HTML from multiple files. Result is merged together to a
     * single report.
     *
     * @param filenames - Filenames to read and parse.
     * @returns Report output.
     */
    validateMultipleFiles(filenames: string[]): Report;
    /**
     * Returns true if the given filename can be validated.
     *
     * A file is considered to be validatable if the extension is `.html` or if a
     * transformer matches the filename.
     *
     * This is mostly useful for tooling to determine whenever to validate the
     * file or not. CLI tools will run on all the given files anyway.
     */
    canValidate(filename: string): boolean;
    /**
     * Tokenize filename and output all tokens.
     *
     * Using CLI this is enabled with `--dump-tokens`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to tokenize.
     */
    dumpTokens(filename: string): TokenDump[];
    /**
     * Parse filename and output all events.
     *
     * Using CLI this is enabled with `--dump-events`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to dump events from.
     */
    dumpEvents(filename: string): EventDump[];
    /**
     * Parse filename and output DOM tree.
     *
     * Using CLI this is enabled with `--dump-tree`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to dump DOM tree from.
     */
    dumpTree(filename: string): string[];
    /**
     * Transform filename and output source data.
     *
     * Using CLI this is enabled with `--dump-source`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to dump source from.
     */
    dumpSource(filename: string): string[];
    /**
     * Get effective configuration schema.
     */
    getConfigurationSchema(): SchemaObject;
    /**
     * Get effective metadata element schema.
     *
     * If a filename is given the configured plugins can extend the
     * schema. Filename must not be an existing file or a filetype normally
     * handled by html-validate but the path will be used when resolving
     * configuration. As a rule-of-thumb, set it to the elements json file.
     */
    getElementsSchema(filename?: string): SchemaObject;
    /**
     * Get contextual documentation for the given rule.
     *
     * Typical usage:
     *
     * ```js
     * const report = htmlvalidate.validateFile("my-file.html");
     * for (const result of report.results){
     *   const config = htmlvalidate.getConfigFor(result.filePath);
     *   for (const message of result.messages){
     *     const documentation = htmlvalidate.getRuleDocumentation(message.ruleId, config, message.context);
     *     // do something with documentation
     *   }
     * }
     * ```
     *
     * @param ruleId - Rule to get documentation for.
     * @param config - If set it provides more accurate description by using the
     * correct configuration for the file.
     * @param context - If set to `Message.context` some rules can provide
     * contextual details and suggestions.
     */
    getRuleDocumentation(ruleId: string, config?: Config | null, context?: any | null): RuleDocumentation | null;
    /**
     * Create a parser configured for given filename.
     *
     * @param source - Source to use.
     */
    getParserFor(source: Source): Parser;
    /**
     * Get configuration for given filename.
     *
     * See [[FileSystemConfigLoader]] for details.
     *
     * @public
     * @param filename - Filename to get configuration for.
     * @param configOverride - Configuration to apply last.
     */
    getConfigFor(filename: string, configOverride?: ConfigData): Config;
    /**
     * Flush configuration cache. Clears full cache unless a filename is given.
     *
     * See [[FileSystemConfigLoader]] for details.
     *
     * @public
     * @param filename - If set, only flush cache for given filename.
     */
    flushConfigCache(filename?: string): void;
}

/**
 * The static configuration loader does not do any per-handle lookup. Only the
 * global or per-call configuration is used.
 *
 * In practice this means no configuration is fetch by traversing the
 * filesystem.
 */
declare class StaticConfigLoader extends ConfigLoader {
    getConfigFor(handle: string, configOverride?: ConfigData): Config;
    flushCache(): void;
    protected defaultConfig(): Config;
}

interface CompatibilityOptions {
    /** If `true` nothing no output will be generated on console. Default: `false` */
    silent: boolean;
    /** Use this version number instead of running version. Default: running version */
    version: string;
    /** Use custom logging callback. Default: `console.error` */
    logger(message: string): void;
}
/**
 * Tests if plugin is compatible with html-validate library. Unless the `silent`
 * option is used a warning is displayed on the console.
 *
 * @param name - Name of plugin
 * @param declared - What library versions the plugin support (e.g. declared peerDependencies)
 * @returns - `true` if version is compatible
 */
declare function compatibilityCheck(name: string, declared: string, options?: Partial<CompatibilityOptions>): boolean;

/**
 * Returns true if given ruleId is an existing builtin rule. It does not handle
 * rules loaded via plugins.
 *
 * Can be used to create forward/backward compatibility by checking if a rule
 * exists to enable/disable it.
 *
 * @param ruleId - Rule id to check
 * @returns `true` if rule exists
 */
declare function ruleExists(ruleId: string): boolean;

/**
 * Loads configuration by traversing filesystem.
 *
 * Configuration is read from three sources and in the following order:
 *
 * 1. Global configuration passed to constructor.
 * 2. Configuration files found when traversing the directory structure.
 * 3. Override passed to this function.
 *
 * The following configuration filenames are searched:
 *
 * - `.htmlvalidate.json`
 * - `.htmlvalidate.js`
 * - `.htmlvalidate.cjs`
 *
 * Global configuration is used when no configuration file is found. The
 * result is always merged with override if present.
 *
 * The `root` property set to `true` affects the configuration as following:
 *
 * 1. If set in override the override is returned as-is.
 * 2. If set in the global config the override is merged into global and
 * returned. No configuration files are searched.
 * 3. Setting `root` in configuration file only stops directory traversal.
 */
declare class FileSystemConfigLoader extends ConfigLoader {
    protected cache: Map<string, Config | null>;
    /**
     * @param config - Global configuration
     * @param configFactory - Optional configuration factory
     */
    constructor(config?: ConfigData, configFactory?: ConfigFactory);
    /**
     * Get configuration for given filename.
     *
     * @param filename - Filename to get configuration for.
     * @param configOverride - Configuration to merge final result with.
     */
    getConfigFor(filename: string, configOverride?: ConfigData): Config;
    /**
     * Flush configuration cache.
     *
     * @param filename - If given only the cache for that file is flushed.
     */
    flushCache(filename?: string): void;
    /**
     * Load raw configuration from directory traversal.
     *
     * This configuration is not merged with global configuration and may return
     * `null` if no configuration files are found.
     */
    fromFilename(filename: string): Config | null;
    protected defaultConfig(): Config;
}

declare type Formatter = (results: Result[]) => string;

/**
 * Get formatter function by name.
 *
 * @param name - Name of formatter.
 * @returns Formatter function or null if it doesn't exist.
 */
declare function getFormatter(name: string): Formatter | null;

export { DOMLoadEvent as $, AttributeData as A, ConfigReadyEvent as B, Config as C, DynamicValue as D, EventDump as E, SourceReadyEvent as F, TokenEvent as G, HtmlValidate as H, TagStartEvent as I, TagOpenEvent as J, TagEndEvent as K, Location as L, MetaData as M, NodeClosed as N, TagCloseEvent as O, ProcessElementContext as P, TagReadyEvent as Q, Rule as R, Severity as S, TextNode as T, UserError as U, ElementReadyEvent as V, AttributeEvent as W, WhitespaceEvent as X, ConditionalEvent as Y, DirectiveEvent as Z, DoctypeEvent as _, ConfigData as a, DOMReadyEvent as a0, TriggerEventMap as a1, ListenEventMap as a2, FileSystemConfigLoader as a3, Formatter as a4, getFormatter as a5, compatibilityCheck as a6, CompatibilityOptions as a7, TokenType as a8, ConfigError as b, ConfigLoader as c, StaticConfigLoader as d, HtmlElement as e, TokenDump as f, SchemaValidationError as g, MetaElement as h, MetaTable as i, MetaCopyableProperty as j, RuleDocumentation as k, Source as l, Report as m, Reporter as n, Message as o, presets as p, Result as q, TransformContext as r, Transformer as s, TemplateExtractor as t, Plugin as u, Parser as v, ruleExists as w, EventHandler as x, EventCallback as y, Event as z };
