import { i as getFormatter$1, l as legacyRequire, U as UserError, H as HtmlValidate } from './core.js';
import fs, { readFileSync } from 'fs';
import 'semver';
import 'kleur';
import '@sidvind/better-ajv-errors';
import 'ajv';
import deepmerge from 'deepmerge';
import 'json-merge-patch';
import path from 'path';
import glob from 'glob';
import ignore from 'ignore';
import prompts from 'prompts';
import '@babel/code-frame';
import '@html-validate/stylish';

const DEFAULT_EXTENSIONS = ["html"];
function isDirectory(filename) {
    const st = fs.statSync(filename);
    return st.isDirectory();
}
function join(stem, filename) {
    if (path.isAbsolute(filename)) {
        return path.normalize(filename);
    }
    else {
        return path.normalize(path.join(stem, filename));
    }
}
function directoryPattern(extensions) {
    switch (extensions.length) {
        case 0:
            return "**";
        case 1:
            return path.join("**", `*.${extensions[0]}`);
        default:
            return path.join("**", `*.{${extensions.join(",")}}`);
    }
}
/**
 * Takes a number of file patterns (globs) and returns array of expanded
 * filenames.
 */
function expandFiles(patterns, options) {
    const cwd = options.cwd || process.cwd();
    const extensions = options.extensions || DEFAULT_EXTENSIONS;
    const files = patterns.reduce((result, pattern) => {
        /* process - as standard input */
        if (pattern === "-") {
            result.push("/dev/stdin");
            return result;
        }
        for (const filename of glob.sync(pattern, { cwd })) {
            /* if file is a directory recursively expand files from it */
            const fullpath = join(cwd, filename);
            if (isDirectory(fullpath)) {
                const dir = expandFiles([directoryPattern(extensions)], Object.assign(Object.assign({}, options), { cwd: fullpath }));
                result = result.concat(dir.map((cur) => join(filename, cur)));
                continue;
            }
            result.push(fullpath);
        }
        return result;
    }, []);
    /* only return unique matches */
    return Array.from(new Set(files));
}

function wrap(formatter, dst) {
    return (results) => {
        const output = formatter(results);
        if (dst) {
            const dir = path.dirname(dst);
            if (!fs.existsSync(dir)) {
                fs.mkdirSync(dir, { recursive: true });
            }
            fs.writeFileSync(dst, output, "utf-8");
            return "";
        }
        else {
            return output;
        }
    };
}
function loadFormatter(name) {
    const fn = getFormatter$1(name);
    if (fn) {
        return fn;
    }
    try {
        return legacyRequire(name);
    }
    catch (error) {
        throw new UserError(`No formatter named "${name}"`, error);
    }
}
function getFormatter(formatters) {
    const fn = formatters.split(",").map((cur) => {
        const [name, dst] = cur.split("=", 2);
        const fn = loadFormatter(name);
        return wrap(fn, dst);
    });
    return (report) => {
        return fn
            .map((formatter) => formatter(report.results))
            .filter(Boolean)
            .join("\n");
    };
}

class IsIgnored {
    constructor() {
        this.cacheIgnore = new Map();
    }
    /**
     * Searches ".htmlvalidateignore" files from filesystem and returns `true` if
     * one of them contains a pattern matching given filename.
     */
    isIgnored(filename) {
        return this.match(filename);
    }
    /**
     * Clear cache
     */
    clearCache() {
        this.cacheIgnore.clear();
    }
    match(target) {
        let current = path.dirname(target);
        // eslint-disable-next-line no-constant-condition
        while (true) {
            const relative = path.relative(current, target);
            const filename = path.join(current, ".htmlvalidateignore");
            /* test filename (relative to the ignore file) against the patterns */
            const ig = this.parseFile(filename);
            if (ig && ig.ignores(relative)) {
                return true;
            }
            /* get the parent directory */
            const child = current;
            current = path.dirname(current);
            /* stop if this is the root directory */
            if (current === child) {
                break;
            }
        }
        return false;
    }
    parseFile(filename) {
        if (this.cacheIgnore.has(filename)) {
            return this.cacheIgnore.get(filename);
        }
        if (!fs.existsSync(filename)) {
            this.cacheIgnore.set(filename, undefined);
            return undefined;
        }
        const content = fs.readFileSync(filename, "utf-8");
        const ig = ignore().add(content);
        this.cacheIgnore.set(filename, ig);
        return ig;
    }
}

var Frameworks;
(function (Frameworks) {
    Frameworks["angularjs"] = "AngularJS";
    Frameworks["vuejs"] = "Vue.js";
    Frameworks["markdown"] = "Markdown";
})(Frameworks || (Frameworks = {}));
const frameworkConfig = {
    [Frameworks.angularjs]: {
        transform: {
            "^.*\\.js$": "html-validate-angular/js",
            "^.*\\.html$": "html-validate-angular/html",
        },
    },
    [Frameworks.vuejs]: {
        plugins: ["html-validate-vue"],
        extends: ["html-validate-vue:recommended"],
        transform: {
            "^.*\\.vue$": "html-validate-vue",
        },
    },
    [Frameworks.markdown]: {
        transform: {
            "^.*\\.md$": "html-validate-markdown",
        },
    },
};
function addFrameworks(src, frameworks) {
    let config = src;
    for (const framework of frameworks) {
        config = deepmerge(config, frameworkConfig[framework]);
    }
    return config;
}
function writeConfig(dst, config) {
    return new Promise((resolve, reject) => {
        fs.writeFile(dst, JSON.stringify(config, null, 2), (err) => {
            if (err)
                reject(err);
            resolve();
        });
    });
}
async function init(cwd) {
    const filename = `${cwd}/.htmlvalidate.json`;
    const exists = fs.existsSync(filename);
    const initialConfig = {
        elements: ["html5"],
        extends: ["html-validate:recommended"],
    };
    /* confirm overwrite */
    if (exists) {
        const result = await prompts({
            name: "overwrite",
            type: "confirm",
            message: "A .htmlvalidate.json file already exists, do you want to overwrite it?",
        });
        if (!result.overwrite) {
            return Promise.reject();
        }
    }
    const questions = [
        {
            name: "frameworks",
            type: "multiselect",
            choices: [
                { title: Frameworks.angularjs, value: Frameworks.angularjs },
                { title: Frameworks.vuejs, value: Frameworks.vuejs },
                { title: Frameworks.markdown, value: Frameworks.markdown },
            ],
            message: "Support additional frameworks?",
        },
    ];
    /* prompt user for questions */
    const answers = await prompts(questions);
    /* write configuration to file */
    let config = initialConfig;
    config = addFrameworks(config, answers.frameworks);
    await writeConfig(filename, config);
    return {
        filename,
    };
}

const defaultConfig = {
    extends: ["html-validate:recommended"],
};
class CLI {
    /**
     * Create new CLI helper.
     *
     * Can be used to create tooling with similar properties to bundled CLI
     * script.
     */
    constructor(options) {
        this.options = options || {};
        this.config = this.getConfig();
        this.ignored = new IsIgnored();
    }
    /**
     * Returns list of files matching patterns and are not ignored. Filenames will
     * have absolute paths.
     *
     * @public
     */
    expandFiles(patterns, options = {}) {
        return expandFiles(patterns, options).filter((filename) => !this.isIgnored(filename));
    }
    getFormatter(formatters) {
        return getFormatter(formatters);
    }
    /**
     * Initialize project with a new configuration.
     *
     * A new `.htmlvalidate.json` file will be placed in the path provided by
     * `cwd`.
     */
    init(cwd) {
        return init(cwd);
    }
    /**
     * Searches ".htmlvalidateignore" files from filesystem and returns `true` if
     * one of them contains a pattern matching given filename.
     */
    isIgnored(filename) {
        return this.ignored.isIgnored(filename);
    }
    /**
     * Clear cache.
     *
     * Previously fetched [[HtmlValidate]] instances must either be fetched again
     * or call [[HtmlValidate.flushConfigCache]].
     */
    clearCache() {
        this.ignored.clearCache();
    }
    /**
     * Get HtmlValidate instance with configuration based on options passed to the
     * constructor.
     */
    getValidator() {
        return new HtmlValidate(this.config);
    }
    /**
     * @internal
     */
    getConfig() {
        const { options } = this;
        const config = options.configFile
            ? JSON.parse(readFileSync(options.configFile, "utf-8"))
            : defaultConfig;
        let rules = options.rules;
        if (rules) {
            if (Array.isArray(rules)) {
                rules = rules.join(",");
            }
            const raw = rules
                .split(",")
                .map((x) => x.replace(/ *(.*):/, '"$1":'))
                .join(",");
            try {
                const rules = JSON.parse(`{${raw}}`);
                config.extends = [];
                config.rules = rules;
            }
            catch (e) {
                // istanbul ignore next
                throw new UserError(`Error while parsing --rule option "{${raw}}": ${e.message}.\n`);
            }
        }
        return config;
    }
}

const jsonFiltered = ["parent", "children", "meta"];
/* eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types */
function eventReplacer(key, value) {
    if (value && key === "location") {
        return `${value.filename}:${value.line}:${value.column}`;
    }
    return jsonFiltered.includes(key) ? "[truncated]" : value;
}
function eventFormatter(entry) {
    const strdata = JSON.stringify(entry.data, eventReplacer, 2);
    return `${entry.event}: ${strdata}`;
}

export { CLI as C, eventFormatter as e };
//# sourceMappingURL=cli.js.map
