import fs from 'fs';
import betterAjvErrors from '@sidvind/better-ajv-errors';
import Ajv from 'ajv';
import deepmerge from 'deepmerge';
import jsonMergePatch from 'json-merge-patch';
import path from 'path';
import semver from 'semver';
import kleur from 'kleur';
import { fileURLToPath } from 'node:url';
import { createRequire } from 'module';
import { codeFrameColumns } from '@babel/code-frame';
import stylishImpl from '@html-validate/stylish';

const $schema$2 = "http://json-schema.org/draft-06/schema#";
const $id$2 = "http://json-schema.org/draft-06/schema#";
const title = "Core schema meta-schema";
const definitions$1 = {
	schemaArray: {
		type: "array",
		minItems: 1,
		items: {
			$ref: "#"
		}
	},
	nonNegativeInteger: {
		type: "integer",
		minimum: 0
	},
	nonNegativeIntegerDefault0: {
		allOf: [
			{
				$ref: "#/definitions/nonNegativeInteger"
			},
			{
				"default": 0
			}
		]
	},
	simpleTypes: {
		"enum": [
			"array",
			"boolean",
			"integer",
			"null",
			"number",
			"object",
			"string"
		]
	},
	stringArray: {
		type: "array",
		items: {
			type: "string"
		},
		uniqueItems: true,
		"default": [
		]
	}
};
const type$2 = [
	"object",
	"boolean"
];
const properties$2 = {
	$id: {
		type: "string",
		format: "uri-reference"
	},
	$schema: {
		type: "string",
		format: "uri"
	},
	$ref: {
		type: "string",
		format: "uri-reference"
	},
	title: {
		type: "string"
	},
	description: {
		type: "string"
	},
	"default": {
	},
	examples: {
		type: "array",
		items: {
		}
	},
	multipleOf: {
		type: "number",
		exclusiveMinimum: 0
	},
	maximum: {
		type: "number"
	},
	exclusiveMaximum: {
		type: "number"
	},
	minimum: {
		type: "number"
	},
	exclusiveMinimum: {
		type: "number"
	},
	maxLength: {
		$ref: "#/definitions/nonNegativeInteger"
	},
	minLength: {
		$ref: "#/definitions/nonNegativeIntegerDefault0"
	},
	pattern: {
		type: "string",
		format: "regex"
	},
	additionalItems: {
		$ref: "#"
	},
	items: {
		anyOf: [
			{
				$ref: "#"
			},
			{
				$ref: "#/definitions/schemaArray"
			}
		],
		"default": {
		}
	},
	maxItems: {
		$ref: "#/definitions/nonNegativeInteger"
	},
	minItems: {
		$ref: "#/definitions/nonNegativeIntegerDefault0"
	},
	uniqueItems: {
		type: "boolean",
		"default": false
	},
	contains: {
		$ref: "#"
	},
	maxProperties: {
		$ref: "#/definitions/nonNegativeInteger"
	},
	minProperties: {
		$ref: "#/definitions/nonNegativeIntegerDefault0"
	},
	required: {
		$ref: "#/definitions/stringArray"
	},
	additionalProperties: {
		$ref: "#"
	},
	definitions: {
		type: "object",
		additionalProperties: {
			$ref: "#"
		},
		"default": {
		}
	},
	properties: {
		type: "object",
		additionalProperties: {
			$ref: "#"
		},
		"default": {
		}
	},
	patternProperties: {
		type: "object",
		additionalProperties: {
			$ref: "#"
		},
		"default": {
		}
	},
	dependencies: {
		type: "object",
		additionalProperties: {
			anyOf: [
				{
					$ref: "#"
				},
				{
					$ref: "#/definitions/stringArray"
				}
			]
		}
	},
	propertyNames: {
		$ref: "#"
	},
	"const": {
	},
	"enum": {
		type: "array",
		minItems: 1,
		uniqueItems: true
	},
	type: {
		anyOf: [
			{
				$ref: "#/definitions/simpleTypes"
			},
			{
				type: "array",
				items: {
					$ref: "#/definitions/simpleTypes"
				},
				minItems: 1,
				uniqueItems: true
			}
		]
	},
	format: {
		type: "string"
	},
	allOf: {
		$ref: "#/definitions/schemaArray"
	},
	anyOf: {
		$ref: "#/definitions/schemaArray"
	},
	oneOf: {
		$ref: "#/definitions/schemaArray"
	},
	not: {
		$ref: "#"
	}
};
var ajvSchemaDraft = {
	$schema: $schema$2,
	$id: $id$2,
	title: title,
	definitions: definitions$1,
	type: type$2,
	properties: properties$2,
	"default": {
}
};

class NestedError extends Error {
    constructor(message, nested) {
        super(message);
        Error.captureStackTrace(this, NestedError);
        if (nested) {
            this.stack += `\nCaused by: ${nested.stack}`;
        }
    }
}

class UserError extends NestedError {
}

function getSummary(schema, obj, errors) {
    const output = betterAjvErrors(schema, obj, errors, {
        format: "js",
    });
    // istanbul ignore next: for safety only
    return output.length > 0 ? output[0].error : "unknown validation error";
}
class SchemaValidationError extends UserError {
    constructor(filename, message, obj, schema, errors) {
        const summary = getSummary(schema, obj, errors);
        super(`${message}: ${summary}`);
        this.filename = filename;
        this.obj = obj;
        this.schema = schema;
        this.errors = errors;
    }
    prettyError() {
        const json = this.getRawJSON();
        return betterAjvErrors(this.schema, this.obj, this.errors, {
            format: "cli",
            indent: 2,
            json,
        });
    }
    getRawJSON() {
        if (this.filename && fs.existsSync(this.filename)) {
            return fs.readFileSync(this.filename, "utf-8");
        }
        else {
            return null;
        }
    }
}

const projectRoot = path.resolve(path.dirname(fileURLToPath(import.meta.url)), "../../");
			const legacyRequire = createRequire(import.meta.url);
			const distFolder = path.resolve(projectRoot, "dist/es");

/**
 * Similar to `require(..)` but removes the cached copy first.
 */
function requireUncached(moduleId) {
    const filename = legacyRequire.resolve(moduleId);
    /* remove references from the parent module to prevent memory leak */
    const m = legacyRequire.cache[filename];
    if (m && m.parent) {
        const { parent } = m;
        for (let i = parent.children.length - 1; i >= 0; i--) {
            if (parent.children[i].id === filename) {
                parent.children.splice(i, 1);
            }
        }
    }
    /* remove old module from cache */
    delete legacyRequire.cache[filename];
    return legacyRequire(filename);
}

const $schema$1 = "http://json-schema.org/draft-06/schema#";
const $id$1 = "https://html-validate.org/schemas/elements.json";
const type$1 = "object";
const properties$1 = {
	$schema: {
		type: "string"
	}
};
const patternProperties = {
	"^[^$].*$": {
		type: "object",
		properties: {
			inherit: {
				title: "Inherit from another element",
				description: "Most properties from the parent element will be copied onto this one",
				type: "string"
			},
			embedded: {
				title: "Mark this element as belonging in the embedded content category",
				$ref: "#/definitions/contentCategory"
			},
			flow: {
				title: "Mark this element as belonging in the flow content category",
				$ref: "#/definitions/contentCategory"
			},
			heading: {
				title: "Mark this element as belonging in the heading content category",
				$ref: "#/definitions/contentCategory"
			},
			interactive: {
				title: "Mark this element as belonging in the interactive content category",
				$ref: "#/definitions/contentCategory"
			},
			metadata: {
				title: "Mark this element as belonging in the metadata content category",
				$ref: "#/definitions/contentCategory"
			},
			phrasing: {
				title: "Mark this element as belonging in the phrasing content category",
				$ref: "#/definitions/contentCategory"
			},
			sectioning: {
				title: "Mark this element as belonging in the sectioning content category",
				$ref: "#/definitions/contentCategory"
			},
			deprecated: {
				title: "Mark element as deprecated",
				description: "Deprecated elements should not be used. If a message is provided it will be included in the error",
				anyOf: [
					{
						type: "boolean"
					},
					{
						type: "string"
					},
					{
						$ref: "#/definitions/deprecatedElement"
					}
				]
			},
			foreign: {
				title: "Mark element as foreign",
				description: "Foreign elements are elements which have a start and end tag but is otherwize not parsed",
				type: "boolean"
			},
			"void": {
				title: "Mark element as void",
				description: "Void elements are elements which cannot have content and thus must not use an end tag",
				type: "boolean"
			},
			transparent: {
				title: "Mark element as transparent",
				description: "Transparent elements follows the same content model as its parent, i.e. the content must be allowed in the parent.",
				anyOf: [
					{
						type: "boolean"
					},
					{
						type: "array",
						items: {
							type: "string"
						}
					}
				]
			},
			implicitClosed: {
				title: "List of elements which implicitly closes this element",
				description: "Some elements are automatically closed when another start tag occurs",
				type: "array",
				items: {
					type: "string"
				}
			},
			scriptSupporting: {
				title: "Mark element as script-supporting",
				description: "Script-supporting elements are elements which can be inserted where othersise not permitted to assist in templating",
				type: "boolean"
			},
			form: {
				title: "Mark element as a submittable form element",
				type: "boolean"
			},
			deprecatedAttributes: {
				title: "List of deprecated attributes",
				type: "array",
				items: {
					type: "string"
				}
			},
			requiredAttributes: {
				title: "List of required attributes",
				type: "array",
				items: {
					type: "string"
				}
			},
			attributes: {
				title: "List of known attributes and allowed values",
				$ref: "#/definitions/PermittedAttribute"
			},
			permittedContent: {
				title: "List of elements or categories allowed as content in this element",
				$ref: "#/definitions/Permitted"
			},
			permittedDescendants: {
				title: "List of elements or categories allowed as descendants in this element",
				$ref: "#/definitions/Permitted"
			},
			permittedOrder: {
				title: "Required order of child elements",
				$ref: "#/definitions/PermittedOrder"
			},
			requiredAncestors: {
				title: "List of required ancestor elements",
				$ref: "#/definitions/RequiredAncestors"
			},
			requiredContent: {
				title: "List of required content elements",
				$ref: "#/definitions/RequiredContent"
			},
			textContent: {
				title: "Allow, disallow or require textual content",
				description: "This property controls whenever an element allows, disallows or requires text. Text from any descendant counts, not only direct children",
				"default": "default",
				type: "string",
				"enum": [
					"none",
					"default",
					"required",
					"accessible"
				]
			}
		},
		additionalProperties: true
	}
};
const definitions = {
	contentCategory: {
		anyOf: [
			{
				type: "boolean"
			},
			{
				$ref: "#/definitions/expression"
			}
		]
	},
	expression: {
		type: "array",
		minItems: 2,
		maxItems: 2,
		items: [
			{
				type: "string",
				"enum": [
					"isDescendant",
					"hasAttribute",
					"matchAttribute"
				]
			},
			{
				anyOf: [
					{
						type: "string"
					},
					{
						$ref: "#/definitions/operation"
					}
				]
			}
		]
	},
	operation: {
		type: "array",
		minItems: 3,
		maxItems: 3,
		items: [
			{
				type: "string"
			},
			{
				type: "string",
				"enum": [
					"!=",
					"="
				]
			},
			{
				type: "string"
			}
		]
	},
	deprecatedElement: {
		type: "object",
		additionalProperties: false,
		properties: {
			message: {
				type: "string",
				title: "A short text message shown next to the regular error message."
			},
			documentation: {
				type: "string",
				title: "An extended markdown formatted message shown with the contextual rule documentation."
			},
			source: {
				type: "string",
				title: "Element source, e.g. what standard or library deprecated this element.",
				"default": "html5"
			}
		}
	},
	Permitted: {
		type: "array",
		items: {
			anyOf: [
				{
					type: "string"
				},
				{
					type: "array",
					items: {
						anyOf: [
							{
								type: "string"
							},
							{
								$ref: "#/definitions/PermittedGroup"
							}
						]
					}
				},
				{
					$ref: "#/definitions/PermittedGroup"
				}
			]
		}
	},
	PermittedAttribute: {
		type: "object",
		patternProperties: {
			"^.*$": {
				type: "array",
				uniqueItems: true,
				items: {
					anyOf: [
						{
							type: "string"
						},
						{
							regexp: true
						}
					]
				}
			}
		}
	},
	PermittedGroup: {
		type: "object",
		additionalProperties: false,
		properties: {
			exclude: {
				anyOf: [
					{
						items: {
							type: "string"
						},
						type: "array"
					},
					{
						type: "string"
					}
				]
			}
		}
	},
	PermittedOrder: {
		type: "array",
		items: {
			type: "string"
		}
	},
	RequiredAncestors: {
		type: "array",
		items: {
			type: "string"
		}
	},
	RequiredContent: {
		type: "array",
		items: {
			type: "string"
		}
	}
};
var schema = {
	$schema: $schema$1,
	$id: $id$1,
	type: type$1,
	properties: properties$1,
	patternProperties: patternProperties,
	definitions: definitions
};

const dynamicKeys = [
    "metadata",
    "flow",
    "sectioning",
    "heading",
    "phrasing",
    "embedded",
    "interactive",
    "labelable",
];
const functionTable = {
    isDescendant,
    hasAttribute,
    matchAttribute,
};
function clone(src) {
    return JSON.parse(JSON.stringify(src));
}
function overwriteMerge$1(a, b) {
    return b;
}
/**
 * AJV keyword "regexp" to validate the type to be a regular expression.
 * Injects errors with the "type" keyword to give the same output.
 */
/* istanbul ignore next: manual testing */
const ajvRegexpValidate = function (data, dataCxt) {
    const valid = data instanceof RegExp;
    if (!valid) {
        ajvRegexpValidate.errors = [
            {
                instancePath: dataCxt === null || dataCxt === void 0 ? void 0 : dataCxt.instancePath,
                schemaPath: undefined,
                keyword: "type",
                message: "should be regexp",
                params: {
                    keyword: "type",
                },
            },
        ];
    }
    return valid;
};
const ajvRegexpKeyword = {
    keyword: "regexp",
    schema: false,
    errors: true,
    validate: ajvRegexpValidate,
};
class MetaTable {
    constructor() {
        this.elements = {};
        this.schema = clone(schema);
    }
    init() {
        this.resolveGlobal();
    }
    /**
     * Extend validation schema.
     */
    extendValidationSchema(patch) {
        if (patch.properties) {
            this.schema = jsonMergePatch.apply(this.schema, {
                patternProperties: {
                    "^[^$].*$": {
                        properties: patch.properties,
                    },
                },
            });
        }
        if (patch.definitions) {
            this.schema = jsonMergePatch.apply(this.schema, {
                definitions: patch.definitions,
            });
        }
    }
    /**
     * Load metadata table from object.
     *
     * @param obj - Object with metadata to load
     * @param filename - Optional filename used when presenting validation error
     */
    loadFromObject(obj, filename = null) {
        var _a;
        const validate = this.getSchemaValidator();
        if (!validate(obj)) {
            throw new SchemaValidationError(filename, `Element metadata is not valid`, obj, this.schema, 
            /* istanbul ignore next: AJV sets .errors when validate returns false */
            (_a = validate.errors) !== null && _a !== void 0 ? _a : []);
        }
        for (const [key, value] of Object.entries(obj)) {
            if (key === "$schema")
                continue;
            this.addEntry(key, value);
        }
    }
    /**
     * Load metadata table from filename
     *
     * @param filename - Filename to load
     */
    loadFromFile(filename) {
        try {
            /* load using require as it can process both js and json */
            const data = requireUncached(filename);
            this.loadFromObject(data, filename);
        }
        catch (err) {
            if (err instanceof SchemaValidationError) {
                throw err;
            }
            throw new UserError(`Failed to load element metadata from "${filename}"`, err);
        }
    }
    /**
     * Get [[MetaElement]] for the given tag or null if the element doesn't exist.
     *
     * @returns A shallow copy of metadata.
     */
    getMetaFor(tagName) {
        tagName = tagName.toLowerCase();
        return this.elements[tagName] ? Object.assign({}, this.elements[tagName]) : null;
    }
    /**
     * Find all tags which has enabled given property.
     */
    getTagsWithProperty(propName) {
        return Object.entries(this.elements)
            .filter(([, entry]) => entry[propName])
            .map(([tagName]) => tagName);
    }
    /**
     * Find tag matching tagName or inheriting from it.
     */
    getTagsDerivedFrom(tagName) {
        return Object.entries(this.elements)
            .filter(([key, entry]) => key === tagName || entry.inherit === tagName)
            .map(([tagName]) => tagName);
    }
    addEntry(tagName, entry) {
        let parent = this.elements[tagName] || {};
        /* handle inheritance */
        if (entry.inherit) {
            const name = entry.inherit;
            parent = this.elements[name];
            if (!parent) {
                throw new UserError(`Element <${tagName}> cannot inherit from <${name}>: no such element`);
            }
        }
        /* merge all sources together */
        const expanded = deepmerge(parent, Object.assign(Object.assign({}, entry), { tagName }), { arrayMerge: overwriteMerge$1 });
        expandRegex(expanded);
        this.elements[tagName] = expanded;
    }
    /**
     * Construct a new AJV schema validator.
     */
    getSchemaValidator() {
        const ajv = new Ajv({ strict: true, strictTuples: true, strictTypes: true });
        ajv.addMetaSchema(ajvSchemaDraft);
        ajv.addKeyword(ajvRegexpKeyword);
        ajv.addKeyword({ keyword: "copyable" });
        return ajv.compile(this.schema);
    }
    getJSONSchema() {
        return this.schema;
    }
    /**
     * Finds the global element definition and merges each known element with the
     * global, e.g. to assign global attributes.
     */
    resolveGlobal() {
        /* skip if there is no global elements */
        if (!this.elements["*"])
            return;
        /* fetch and remove the global element, it should not be resolvable by
         * itself */
        const global = this.elements["*"];
        delete this.elements["*"];
        /* hack: unset default properties which global should not override */
        delete global.tagName;
        delete global.void;
        /* merge elements */
        for (const [tagName, entry] of Object.entries(this.elements)) {
            this.elements[tagName] = this.mergeElement(global, entry);
        }
    }
    mergeElement(a, b) {
        return deepmerge(a, b, { arrayMerge: overwriteMerge$1 });
    }
    resolve(node) {
        if (node.meta) {
            expandProperties(node, node.meta);
        }
    }
}
function expandProperties(node, entry) {
    for (const key of dynamicKeys) {
        const property = entry[key];
        if (property && typeof property !== "boolean") {
            entry[key] = evaluateProperty(node, property);
        }
    }
}
/**
 * Given a string it returns either the string as-is or if the string is wrapped
 * in /../ it creates and returns a regex instead.
 */
function expandRegexValue(value) {
    if (value instanceof RegExp) {
        return value;
    }
    const match = value.match(/^\/\^?([^/$]*)\$?\/([i]*)$/);
    if (match) {
        const [, expr, flags] = match;
        // eslint-disable-next-line security/detect-non-literal-regexp
        return new RegExp(`^${expr}$`, flags);
    }
    else {
        return value;
    }
}
/**
 * Expand all regular expressions in strings ("/../"). This mutates the object.
 */
function expandRegex(entry) {
    if (!entry.attributes)
        return;
    for (const [name, values] of Object.entries(entry.attributes)) {
        if (values) {
            entry.attributes[name] = values.map(expandRegexValue);
        }
        else {
            delete entry.attributes[name];
        }
    }
}
function evaluateProperty(node, expr) {
    const [func, options] = parseExpression(expr);
    return func(node, options);
}
function parseExpression(expr) {
    if (typeof expr === "string") {
        return parseExpression([expr, {}]);
    }
    else {
        const [funcName, options] = expr;
        const func = functionTable[funcName];
        if (!func) {
            throw new Error(`Failed to find function "${funcName}" when evaluating property expression`);
        }
        return [func, options];
    }
}
function isDescendant(node, tagName) {
    if (typeof tagName !== "string") {
        throw new Error(`Property expression "isDescendant" must take string argument when evaluating metadata for <${node.tagName}>`);
    }
    let cur = node.parent;
    while (cur && !cur.isRootElement()) {
        if (cur.is(tagName)) {
            return true;
        }
        cur = cur.parent;
    }
    return false;
}
function hasAttribute(node, attr) {
    if (typeof attr !== "string") {
        throw new Error(`Property expression "hasAttribute" must take string argument when evaluating metadata for <${node.tagName}>`);
    }
    return node.hasAttribute(attr);
}
function matchAttribute(node, match) {
    if (!Array.isArray(match) || match.length !== 3) {
        throw new Error(`Property expression "matchAttribute" must take [key, op, value] array as argument when evaluating metadata for <${node.tagName}>`);
    }
    const [key, op, value] = match.map((x) => x.toLowerCase());
    const nodeValue = (node.getAttributeValue(key) || "").toLowerCase();
    switch (op) {
        case "!=":
            return nodeValue !== value;
        case "=":
            return nodeValue === value;
        default:
            throw new Error(`Property expression "matchAttribute" has invalid operator "${op}" when evaluating metadata for <${node.tagName}>`);
    }
}

var TextContent$1;
(function (TextContent) {
    /* forbid node to have text content, inter-element whitespace is ignored */
    TextContent["NONE"] = "none";
    /* node can have text but not required too */
    TextContent["DEFAULT"] = "default";
    /* node requires text-nodes to be present (direct or by descendant) */
    TextContent["REQUIRED"] = "required";
    /* node requires accessible text (hidden text is ignored, tries to get text from accessibility tree) */
    TextContent["ACCESSIBLE"] = "accessible";
})(TextContent$1 || (TextContent$1 = {}));
/**
 * Properties listed here can be copied (loaded) onto another element using
 * [[HtmlElement.loadMeta]].
 */
const MetaCopyableProperty = [
    "metadata",
    "flow",
    "sectioning",
    "heading",
    "phrasing",
    "embedded",
    "interactive",
    "transparent",
    "form",
    "labelable",
    "requiredAttributes",
    "attributes",
    "permittedContent",
    "permittedDescendants",
    "permittedOrder",
    "requiredAncestors",
    "requiredContent",
];

class DynamicValue {
    constructor(expr) {
        this.expr = expr;
    }
    toString() {
        return this.expr;
    }
}

/**
 * DOM Attribute.
 *
 * Represents a HTML attribute. Can contain either a fixed static value or a
 * placeholder for dynamic values (e.g. interpolated).
 */
class Attribute {
    /**
     * @param key - Attribute name.
     * @param value - Attribute value. Set to `null` for boolean attributes.
     * @param keyLocation - Source location of attribute name.
     * @param valueLocation - Source location of attribute value.
     * @param originalAttribute - If this attribute was dynamically added via a
     * transformation (e.g. vuejs `:id` generating the `id` attribute) this
     * parameter should be set to the attribute name of the source attribute (`:id`).
     */
    constructor(key, value, keyLocation, valueLocation, originalAttribute) {
        this.key = key;
        this.value = value;
        this.keyLocation = keyLocation;
        this.valueLocation = valueLocation;
        this.originalAttribute = originalAttribute;
        /* force undefined to null */
        if (typeof this.value === "undefined") {
            this.value = null;
        }
    }
    /**
     * Flag set to true if the attribute value is static.
     */
    get isStatic() {
        return !this.isDynamic;
    }
    /**
     * Flag set to true if the attribute value is dynamic.
     */
    get isDynamic() {
        return this.value instanceof DynamicValue;
    }
    valueMatches(pattern, dynamicMatches = true) {
        if (this.value === null) {
            return false;
        }
        /* dynamic values matches everything */
        if (this.value instanceof DynamicValue) {
            return dynamicMatches;
        }
        /* test value against pattern */
        if (pattern instanceof RegExp) {
            return this.value.match(pattern) !== null;
        }
        else {
            return this.value === pattern;
        }
    }
}

function sliceSize(size, begin, end) {
    if (typeof size !== "number") {
        return size;
    }
    if (typeof end !== "number") {
        return size - begin;
    }
    if (end < 0) {
        end = size + end;
    }
    return Math.min(size, end - begin);
}
function sliceLocation(location, begin, end, wrap) {
    if (!location)
        return null;
    const size = sliceSize(location.size, begin, end);
    const sliced = {
        filename: location.filename,
        offset: location.offset + begin,
        line: location.line,
        column: location.column + begin,
        size,
    };
    /* if text content is provided try to find all newlines and modify line/column accordingly */
    if (wrap) {
        let index = -1;
        const col = sliced.column;
        do {
            index = wrap.indexOf("\n", index + 1);
            if (index >= 0 && index < begin) {
                sliced.column = col - (index + 1);
                sliced.line++;
            }
            else {
                break;
            }
        } while (true); // eslint-disable-line no-constant-condition
    }
    return sliced;
}

var State;
(function (State) {
    State[State["INITIAL"] = 1] = "INITIAL";
    State[State["DOCTYPE"] = 2] = "DOCTYPE";
    State[State["TEXT"] = 3] = "TEXT";
    State[State["TAG"] = 4] = "TAG";
    State[State["ATTR"] = 5] = "ATTR";
    State[State["CDATA"] = 6] = "CDATA";
    State[State["SCRIPT"] = 7] = "SCRIPT";
})(State || (State = {}));

var ContentModel;
(function (ContentModel) {
    ContentModel[ContentModel["TEXT"] = 1] = "TEXT";
    ContentModel[ContentModel["SCRIPT"] = 2] = "SCRIPT";
})(ContentModel || (ContentModel = {}));
class Context {
    constructor(source) {
        var _a, _b, _c, _d;
        this.state = State.INITIAL;
        this.string = source.data;
        this.filename = (_a = source.filename) !== null && _a !== void 0 ? _a : "";
        this.offset = (_b = source.offset) !== null && _b !== void 0 ? _b : 0;
        this.line = (_c = source.line) !== null && _c !== void 0 ? _c : 1;
        this.column = (_d = source.column) !== null && _d !== void 0 ? _d : 1;
        this.contentModel = ContentModel.TEXT;
    }
    getTruncatedLine(n = 13) {
        return JSON.stringify(this.string.length > n ? `${this.string.slice(0, 10)}...` : this.string);
    }
    consume(n, state) {
        /* if "n" is an regex match the first value is the full matched
         * string so consume that many characters. */
        if (typeof n !== "number") {
            n = n[0].length; /* regex match */
        }
        /* poor mans line counter :( */
        let consumed = this.string.slice(0, n);
        let offset;
        while ((offset = consumed.indexOf("\n")) >= 0) {
            this.line++;
            this.column = 1;
            consumed = consumed.substr(offset + 1);
        }
        this.column += consumed.length;
        this.offset += n;
        /* remove N chars */
        this.string = this.string.substr(n);
        /* change state */
        this.state = state;
    }
    getLocation(size) {
        return {
            filename: this.filename,
            offset: this.offset,
            line: this.line,
            column: this.column,
            size,
        };
    }
}

var NodeType;
(function (NodeType) {
    NodeType[NodeType["ELEMENT_NODE"] = 1] = "ELEMENT_NODE";
    NodeType[NodeType["TEXT_NODE"] = 3] = "TEXT_NODE";
    NodeType[NodeType["DOCUMENT_NODE"] = 9] = "DOCUMENT_NODE";
})(NodeType || (NodeType = {}));

const DOCUMENT_NODE_NAME = "#document";
const TEXT_CONTENT = Symbol("textContent");
let counter = 0;
class DOMNode {
    /**
     * Create a new DOMNode.
     *
     * @param nodeType - What node type to create.
     * @param nodeName - What node name to use. For `HtmlElement` this corresponds
     * to the tagName but other node types have specific predefined values.
     * @param location - Source code location of this node.
     */
    constructor(nodeType, nodeName, location) {
        this.nodeType = nodeType;
        this.nodeName = nodeName !== null && nodeName !== void 0 ? nodeName : DOCUMENT_NODE_NAME;
        this.location = location;
        this.disabledRules = new Set();
        this.childNodes = [];
        this.unique = counter++;
        this.cache = null;
    }
    /**
     * Enable cache for this node.
     *
     * Should not be called before the node and all children are fully constructed.
     */
    cacheEnable() {
        this.cache = new Map();
    }
    cacheGet(key) {
        if (this.cache) {
            return this.cache.get(key);
        }
        else {
            return undefined;
        }
    }
    cacheSet(key, value) {
        if (this.cache) {
            this.cache.set(key, value);
        }
        return value;
    }
    cacheRemove(key) {
        if (this.cache) {
            return this.cache.delete(key);
        }
        else {
            return false;
        }
    }
    cacheExists(key) {
        return Boolean(this.cache && this.cache.has(key));
    }
    /**
     * Get the text (recursive) from all child nodes.
     */
    get textContent() {
        const cached = this.cacheGet(TEXT_CONTENT);
        if (cached) {
            return cached;
        }
        const text = this.childNodes.map((node) => node.textContent).join("");
        this.cacheSet(TEXT_CONTENT, text);
        return text;
    }
    append(node) {
        this.childNodes.push(node);
    }
    isRootElement() {
        return this.nodeType === NodeType.DOCUMENT_NODE;
    }
    /**
     * Tests if two nodes are the same (references the same object).
     */
    isSameNode(otherNode) {
        return this.unique === otherNode.unique;
    }
    /**
     * Returns a DOMNode representing the first direct child node or `null` if the
     * node has no children.
     */
    get firstChild() {
        return this.childNodes[0] || null;
    }
    /**
     * Returns a DOMNode representing the last direct child node or `null` if the
     * node has no children.
     */
    get lastChild() {
        return this.childNodes[this.childNodes.length - 1] || null;
    }
    /**
     * Disable a rule for this node.
     */
    disableRule(ruleId) {
        this.disabledRules.add(ruleId);
    }
    /**
     * Disables multiple rules.
     */
    disableRules(rules) {
        for (const rule of rules) {
            this.disableRule(rule);
        }
    }
    /**
     * Enable a previously disabled rule for this node.
     */
    enableRule(ruleId) {
        this.disabledRules.delete(ruleId);
    }
    /**
     * Enables multiple rules.
     */
    enableRules(rules) {
        for (const rule of rules) {
            this.enableRule(rule);
        }
    }
    /**
     * Test if a rule is enabled for this node.
     */
    ruleEnabled(ruleId) {
        return !this.disabledRules.has(ruleId);
    }
    generateSelector() {
        return null;
    }
}

function parse(text, baseLocation) {
    const tokens = [];
    const locations = baseLocation ? [] : null;
    for (let begin = 0; begin < text.length;) {
        let end = text.indexOf(" ", begin);
        /* if the last space was found move the position to the last character
         * in the string */
        if (end === -1) {
            end = text.length;
        }
        /* handle multiple spaces */
        const size = end - begin;
        if (size === 0) {
            begin++;
            continue;
        }
        /* extract token */
        const token = text.substring(begin, end);
        tokens.push(token);
        /* extract location */
        if (locations && baseLocation) {
            const location = sliceLocation(baseLocation, begin, end);
            locations.push(location);
        }
        /* advance position to the character after the current end position */
        begin += size + 1;
    }
    return { tokens, locations };
}
class DOMTokenList extends Array {
    constructor(value, location) {
        if (value && typeof value === "string") {
            /* replace all whitespace with a single space for easier parsing */
            const condensed = value.replace(/[\t\r\n ]+/g, " ");
            const { tokens, locations } = parse(condensed, location);
            super(...tokens);
            this.locations = locations;
        }
        else {
            super(0);
            this.locations = null;
        }
        if (value instanceof DynamicValue) {
            this.value = value.expr;
        }
        else {
            this.value = value || "";
        }
    }
    item(n) {
        return this[n];
    }
    location(n) {
        if (this.locations) {
            return this.locations[n];
        }
        else {
            throw new Error("Trying to access DOMTokenList location when base location isn't set");
        }
    }
    contains(token) {
        return this.includes(token);
    }
    *iterator() {
        for (let index = 0; index < this.length; index++) {
            /* eslint-disable @typescript-eslint/no-non-null-assertion */
            const item = this.item(index);
            const location = this.location(index);
            /* eslint-enable @typescript-eslint/no-non-null-assertion */
            yield { index, item, location };
        }
    }
}

var Combinator;
(function (Combinator) {
    Combinator[Combinator["DESCENDANT"] = 1] = "DESCENDANT";
    Combinator[Combinator["CHILD"] = 2] = "CHILD";
    Combinator[Combinator["ADJACENT_SIBLING"] = 3] = "ADJACENT_SIBLING";
    Combinator[Combinator["GENERAL_SIBLING"] = 4] = "GENERAL_SIBLING";
    /* special cases */
    Combinator[Combinator["SCOPE"] = 5] = "SCOPE";
})(Combinator || (Combinator = {}));
function parseCombinator(combinator, pattern) {
    /* special case, when pattern is :scope [[Selector]] will handle this
     * "combinator" to match itself instead of descendants */
    if (pattern === ":scope") {
        return Combinator.SCOPE;
    }
    switch (combinator) {
        case undefined:
        case null:
        case "":
            return Combinator.DESCENDANT;
        case ">":
            return Combinator.CHILD;
        case "+":
            return Combinator.ADJACENT_SIBLING;
        case "~":
            return Combinator.GENERAL_SIBLING;
        default:
            throw new Error(`Unknown combinator "${combinator}"`);
    }
}

function firstChild(node) {
    return node.previousSibling === null;
}

function lastChild(node) {
    return node.nextSibling === null;
}

const cache = {};
function getNthChild(node) {
    if (!node.parent) {
        return -1;
    }
    if (!cache[node.unique]) {
        const parent = node.parent;
        const index = parent.childElements.findIndex((cur) => {
            return cur.unique === node.unique;
        });
        cache[node.unique] = index + 1; /* nthChild starts at 1 */
    }
    return cache[node.unique];
}
function nthChild(node, args) {
    if (!args) {
        throw new Error("Missing argument to nth-child");
    }
    const n = parseInt(args.trim(), 10);
    const cur = getNthChild(node);
    return cur === n;
}

function scope(node) {
    return node.isSameNode(this.scope);
}

const table = {
    "first-child": firstChild,
    "last-child": lastChild,
    "nth-child": nthChild,
    scope: scope,
};
function factory(name, context) {
    const fn = table[name];
    if (fn) {
        return fn.bind(context);
    }
    else {
        throw new Error(`Pseudo-class "${name}" is not implemented`);
    }
}

/**
 * Homage to PHP: unescapes slashes.
 *
 * E.g. "foo\:bar" becomes "foo:bar"
 */
function stripslashes(value) {
    return value.replace(/\\(.)/g, "$1");
}
function escapeSelectorComponent(text) {
    return text.toString().replace(/([:[\] ])/g, "\\$1");
}
class Matcher {
}
class ClassMatcher extends Matcher {
    constructor(classname) {
        super();
        this.classname = classname;
    }
    match(node) {
        return node.classList.contains(this.classname);
    }
}
class IdMatcher extends Matcher {
    constructor(id) {
        super();
        this.id = stripslashes(id);
    }
    match(node) {
        return node.id === this.id;
    }
}
class AttrMatcher extends Matcher {
    constructor(attr) {
        super();
        const [, key, op, value] = attr.match(/^(.+?)(?:([~^$*|]?=)"([^"]+?)")?$/);
        this.key = key;
        this.op = op;
        this.value = value;
    }
    match(node) {
        const attr = node.getAttribute(this.key, true) || [];
        return attr.some((cur) => {
            switch (this.op) {
                case undefined:
                    return true; /* attribute exists */
                case "=":
                    return cur.value === this.value;
                default:
                    throw new Error(`Attribute selector operator ${this.op} is not implemented yet`);
            }
        });
    }
}
class PseudoClassMatcher extends Matcher {
    constructor(pseudoclass, context) {
        super();
        const match = pseudoclass.match(/^([^(]+)(?:\((.*)\))?$/);
        if (!match) {
            throw new Error(`Missing pseudo-class after colon in selector pattern "${context}"`);
        }
        const [, name, args] = match;
        this.name = name;
        this.args = args;
    }
    match(node, context) {
        const fn = factory(this.name, context);
        return fn(node, this.args);
    }
}
class Pattern {
    constructor(pattern) {
        const match = pattern.match(/^([~+\->]?)((?:[*]|[^.#[:]+)?)(.*)$/);
        match.shift(); /* remove full matched string */
        this.selector = pattern;
        this.combinator = parseCombinator(match.shift(), pattern);
        this.tagName = match.shift() || "*";
        const p = match[0] ? match[0].split(/(?=(?<!\\)[.#[:])/) : [];
        this.pattern = p.map((cur) => this.createMatcher(cur));
    }
    match(node, context) {
        return node.is(this.tagName) && this.pattern.every((cur) => cur.match(node, context));
    }
    createMatcher(pattern) {
        switch (pattern[0]) {
            case ".":
                return new ClassMatcher(pattern.slice(1));
            case "#":
                return new IdMatcher(pattern.slice(1));
            case "[":
                return new AttrMatcher(pattern.slice(1, -1));
            case ":":
                return new PseudoClassMatcher(pattern.slice(1), this.selector);
            default:
                /* istanbul ignore next: fallback solution, the switch cases should cover
                 * everything and there is no known way to trigger this fallback */
                throw new Error(`Failed to create matcher for "${pattern}"`);
        }
    }
}
/**
 * DOM Selector.
 */
class Selector {
    constructor(selector) {
        this.pattern = Selector.parse(selector);
    }
    /**
     * Match this selector against a HtmlElement.
     *
     * @param root - Element to match against.
     * @returns Iterator with matched elements.
     */
    *match(root) {
        const context = { scope: root };
        yield* this.matchInternal(root, 0, context);
    }
    *matchInternal(root, level, context) {
        if (level >= this.pattern.length) {
            yield root;
            return;
        }
        const pattern = this.pattern[level];
        const matches = Selector.findCandidates(root, pattern);
        for (const node of matches) {
            if (!pattern.match(node, context)) {
                continue;
            }
            yield* this.matchInternal(node, level + 1, context);
        }
    }
    static parse(selector) {
        /* strip whitespace before combinators, "ul > li" becomes "ul >li", for
         * easier parsing */
        selector = selector.replace(/([+~>]) /g, "$1");
        const pattern = selector.split(/(?:(?<!\\) )+/);
        return pattern.map((part) => new Pattern(part));
    }
    static findCandidates(root, pattern) {
        switch (pattern.combinator) {
            case Combinator.DESCENDANT:
                return root.getElementsByTagName(pattern.tagName);
            case Combinator.CHILD:
                return root.childElements.filter((node) => node.is(pattern.tagName));
            case Combinator.ADJACENT_SIBLING:
                return Selector.findAdjacentSibling(root);
            case Combinator.GENERAL_SIBLING:
                return Selector.findGeneralSibling(root);
            case Combinator.SCOPE:
                return [root];
        }
        /* istanbul ignore next: fallback solution, the switch cases should cover
         * everything and there is no known way to trigger this fallback */
        return [];
    }
    static findAdjacentSibling(node) {
        let adjacent = false;
        return node.siblings.filter((cur) => {
            if (adjacent) {
                adjacent = false;
                return true;
            }
            if (cur === node) {
                adjacent = true;
            }
            return false;
        });
    }
    static findGeneralSibling(node) {
        let after = false;
        return node.siblings.filter((cur) => {
            if (after) {
                return true;
            }
            if (cur === node) {
                after = true;
            }
            return false;
        });
    }
}

const TEXT_NODE_NAME = "#text";
/**
 * Represents a text in the HTML document.
 *
 * Text nodes are appended as children of `HtmlElement` and cannot have childen
 * of its own.
 */
class TextNode extends DOMNode {
    /**
     * @param text - Text to add. When a `DynamicValue` is used the expression is
     * used as "text".
     * @param location - Source code location of this node.
     */
    constructor(text, location) {
        super(NodeType.TEXT_NODE, TEXT_NODE_NAME, location);
        this.text = text;
    }
    /**
     * Get the text from node.
     */
    get textContent() {
        return this.text.toString();
    }
    /**
     * Flag set to true if the attribute value is static.
     */
    get isStatic() {
        return !this.isDynamic;
    }
    /**
     * Flag set to true if the attribute value is dynamic.
     */
    get isDynamic() {
        return this.text instanceof DynamicValue;
    }
}

var NodeClosed;
(function (NodeClosed) {
    NodeClosed[NodeClosed["Open"] = 0] = "Open";
    NodeClosed[NodeClosed["EndTag"] = 1] = "EndTag";
    NodeClosed[NodeClosed["VoidOmitted"] = 2] = "VoidOmitted";
    NodeClosed[NodeClosed["VoidSelfClosed"] = 3] = "VoidSelfClosed";
    NodeClosed[NodeClosed["ImplicitClosed"] = 4] = "ImplicitClosed";
})(NodeClosed || (NodeClosed = {}));
function isElement(node) {
    return node.nodeType === NodeType.ELEMENT_NODE;
}
function isValidTagName(tagName) {
    return Boolean(tagName !== "" && tagName !== "*");
}
class HtmlElement extends DOMNode {
    constructor(tagName, parent, closed, meta, location) {
        const nodeType = tagName ? NodeType.ELEMENT_NODE : NodeType.DOCUMENT_NODE;
        super(nodeType, tagName, location);
        if (!isValidTagName(tagName)) {
            throw new Error(`The tag name provided ('${tagName || ""}') is not a valid name`);
        }
        this.tagName = tagName || "#document";
        this.parent = parent !== null && parent !== void 0 ? parent : null;
        this.attr = {};
        this.metaElement = meta !== null && meta !== void 0 ? meta : null;
        this.closed = closed;
        this.voidElement = meta ? Boolean(meta.void) : false;
        this.depth = 0;
        this.annotation = null;
        if (parent) {
            parent.childNodes.push(this);
            /* calculate depth in domtree */
            let cur = parent;
            while (cur.parent) {
                this.depth++;
                cur = cur.parent;
            }
        }
    }
    static rootNode(location) {
        return new HtmlElement(undefined, null, NodeClosed.EndTag, null, location);
    }
    static fromTokens(startToken, endToken, parent, metaTable) {
        const tagName = startToken.data[2];
        if (!tagName) {
            throw new Error("tagName cannot be empty");
        }
        const meta = metaTable ? metaTable.getMetaFor(tagName) : null;
        const open = startToken.data[1] !== "/";
        const closed = isClosed(endToken, meta);
        /* location contains position of '<' so strip it out */
        const location = sliceLocation(startToken.location, 1);
        return new HtmlElement(tagName, open ? parent : null, closed, meta, location);
    }
    /**
     * Returns annotated name if set or defaults to `<tagName>`.
     *
     * E.g. `my-annotation` or `<div>`.
     */
    get annotatedName() {
        if (this.annotation) {
            return this.annotation;
        }
        else {
            return `<${this.tagName}>`;
        }
    }
    /**
     * Similar to childNodes but only elements.
     */
    get childElements() {
        return this.childNodes.filter(isElement);
    }
    /**
     * Find the first ancestor matching a selector.
     *
     * Implementation of DOM specification of Element.closest(selectors).
     */
    closest(selectors) {
        /* eslint-disable-next-line @typescript-eslint/no-this-alias */
        let node = this;
        while (node) {
            if (node.matches(selectors)) {
                return node;
            }
            node = node.parent;
        }
        return null;
    }
    /**
     * Generate a DOM selector for this element. The returned selector will be
     * unique inside the current document.
     */
    generateSelector() {
        /* root element cannot have a selector as it isn't a proper element */
        if (this.isRootElement()) {
            return null;
        }
        const parts = [];
        let root;
        for (root = this; root.parent; root = root.parent) {
            /* .. */
        }
        // eslint-disable-next-line @typescript-eslint/no-this-alias
        for (let cur = this; cur.parent; cur = cur.parent) {
            /* if a unique id is present, use it and short-circuit */
            if (cur.id) {
                const escaped = escapeSelectorComponent(cur.id);
                const matches = root.querySelectorAll(`#${escaped}`);
                if (matches.length === 1) {
                    parts.push(`#${escaped}`);
                    break;
                }
            }
            const parent = cur.parent;
            const child = parent.childElements;
            const index = child.findIndex((it) => it.unique === cur.unique);
            const numOfType = child.filter((it) => it.is(cur.tagName)).length;
            const solo = numOfType === 1;
            /* if this is the only tagName in this level of siblings nth-child isn't needed */
            if (solo) {
                parts.push(cur.tagName.toLowerCase());
                continue;
            }
            /* this will generate the worst kind of selector but at least it will be accurate (optimizations welcome) */
            parts.push(`${cur.tagName.toLowerCase()}:nth-child(${index + 1})`);
        }
        return parts.reverse().join(" > ");
    }
    /**
     * Tests if this element has given tagname.
     *
     * If passing "*" this test will pass if any tagname is set.
     */
    is(tagName) {
        return tagName === "*" || this.tagName.toLowerCase() === tagName.toLowerCase();
    }
    /**
     * Load new element metadata onto this element.
     *
     * Do note that semantics such as `void` cannot be changed (as the element has
     * already been created). In addition the element will still "be" the same
     * element, i.e. even if loading meta for a `<p>` tag upon a `<div>` tag it
     * will still be a `<div>` as far as the rest of the validator is concerned.
     *
     * In fact only certain properties will be copied onto the element:
     *
     * - content categories (flow, phrasing, etc)
     * - required attributes
     * - attribute allowed values
     * - permitted/required elements
     *
     * Properties *not* loaded:
     *
     * - inherit
     * - deprecated
     * - foreign
     * - void
     * - implicitClosed
     * - scriptSupporting
     * - deprecatedAttributes
     *
     * Changes to element metadata will only be visible after `element:ready` (and
     * the subsequent `dom:ready` event).
     */
    loadMeta(meta) {
        if (!this.metaElement) {
            this.metaElement = {};
        }
        for (const key of MetaCopyableProperty) {
            if (typeof meta[key] !== "undefined") {
                this.metaElement[key] = meta[key];
            }
            else {
                delete this.metaElement[key];
            }
        }
    }
    /**
     * Match this element against given selectors. Returns true if any selector
     * matches.
     *
     * Implementation of DOM specification of Element.matches(selectors).
     */
    matches(selector) {
        /* find root element */
        /* eslint-disable-next-line @typescript-eslint/no-this-alias */
        let root = this;
        while (root.parent) {
            root = root.parent;
        }
        /* a bit slow implementation as it finds all candidates for the selector and
         * then tests if any of them are the current element. A better
         * implementation would be to walk the selector right-to-left and test
         * ancestors. */
        for (const match of root.querySelectorAll(selector)) {
            if (match.unique === this.unique) {
                return true;
            }
        }
        return false;
    }
    get meta() {
        return this.metaElement;
    }
    /**
     * Set annotation for this element.
     */
    setAnnotation(text) {
        this.annotation = text;
    }
    /**
     * Set attribute. Stores all attributes set even with the same name.
     *
     * @param key - Attribute name
     * @param value - Attribute value. Use `null` if no value is present.
     * @param keyLocation - Location of the attribute name.
     * @param valueLocation - Location of the attribute value (excluding quotation)
     * @param originalAttribute - If attribute is an alias for another attribute
     * (dynamic attributes) set this to the original attribute name.
     */
    setAttribute(key, value, keyLocation, valueLocation, originalAttribute) {
        key = key.toLowerCase();
        if (!this.attr[key]) {
            this.attr[key] = [];
        }
        this.attr[key].push(new Attribute(key, value, keyLocation, valueLocation, originalAttribute));
    }
    /**
     * Get a list of all attributes on this node.
     */
    get attributes() {
        return Object.values(this.attr).reduce((result, cur) => {
            return result.concat(cur);
        }, []);
    }
    hasAttribute(key) {
        key = key.toLowerCase();
        return key in this.attr;
    }
    getAttribute(key, all = false) {
        key = key.toLowerCase();
        if (key in this.attr) {
            const matches = this.attr[key];
            return all ? matches : matches[0];
        }
        else {
            return null;
        }
    }
    /**
     * Get attribute value.
     *
     * Returns the attribute value if present.
     *
     * - Missing attributes return `null`.
     * - Boolean attributes return `null`.
     * - `DynamicValue` returns attribute expression.
     *
     * @param key - Attribute name
     * @returns Attribute value or null.
     */
    getAttributeValue(key) {
        const attr = this.getAttribute(key);
        if (attr) {
            return attr.value !== null ? attr.value.toString() : null;
        }
        else {
            return null;
        }
    }
    /**
     * Add text as a child node to this element.
     *
     * @param text - Text to add.
     * @param location - Source code location of this text.
     */
    appendText(text, location) {
        this.childNodes.push(new TextNode(text, location));
    }
    /**
     * Return a list of all known classes on the element. Dynamic values are
     * ignored.
     */
    get classList() {
        if (!this.hasAttribute("class")) {
            return new DOMTokenList(null, null);
        }
        const classes = this.getAttribute("class", true)
            .filter((attr) => attr.isStatic)
            .map((attr) => attr.value)
            .join(" ");
        return new DOMTokenList(classes, null);
    }
    /**
     * Get element ID if present.
     */
    get id() {
        return this.getAttributeValue("id");
    }
    /**
     * Returns the first child element or null if there are no child elements.
     */
    get firstElementChild() {
        const children = this.childElements;
        return children.length > 0 ? children[0] : null;
    }
    /**
     * Returns the last child element or null if there are no child elements.
     */
    get lastElementChild() {
        const children = this.childElements;
        return children.length > 0 ? children[children.length - 1] : null;
    }
    get siblings() {
        return this.parent ? this.parent.childElements : [this];
    }
    get previousSibling() {
        const i = this.siblings.findIndex((node) => node.unique === this.unique);
        return i >= 1 ? this.siblings[i - 1] : null;
    }
    get nextSibling() {
        const i = this.siblings.findIndex((node) => node.unique === this.unique);
        return i <= this.siblings.length - 2 ? this.siblings[i + 1] : null;
    }
    getElementsByTagName(tagName) {
        return this.childElements.reduce((matches, node) => {
            return matches.concat(node.is(tagName) ? [node] : [], node.getElementsByTagName(tagName));
        }, []);
    }
    querySelector(selector) {
        const it = this.querySelectorImpl(selector);
        return it.next().value || null;
    }
    querySelectorAll(selector) {
        const it = this.querySelectorImpl(selector);
        const unique = new Set(it);
        return Array.from(unique.values());
    }
    *querySelectorImpl(selectorList) {
        if (!selectorList) {
            return;
        }
        for (const selector of selectorList.split(/,\s*/)) {
            const pattern = new Selector(selector);
            yield* pattern.match(this);
        }
    }
    /**
     * Visit all nodes from this node and down. Depth first.
     */
    visitDepthFirst(callback) {
        function visit(node) {
            node.childElements.forEach(visit);
            if (!node.isRootElement()) {
                callback(node);
            }
        }
        visit(this);
    }
    /**
     * Evaluates callbackk on all descendants, returning true if any are true.
     */
    someChildren(callback) {
        return this.childElements.some(visit);
        function visit(node) {
            if (callback(node)) {
                return true;
            }
            else {
                return node.childElements.some(visit);
            }
        }
    }
    /**
     * Evaluates callbackk on all descendants, returning true if all are true.
     */
    everyChildren(callback) {
        return this.childElements.every(visit);
        function visit(node) {
            if (!callback(node)) {
                return false;
            }
            return node.childElements.every(visit);
        }
    }
    /**
     * Visit all nodes from this node and down. Breadth first.
     *
     * The first node for which the callback evaluates to true is returned.
     */
    find(callback) {
        function visit(node) {
            if (callback(node)) {
                return node;
            }
            for (const child of node.childElements) {
                const match = child.find(callback);
                if (match) {
                    return match;
                }
            }
            return null;
        }
        return visit(this);
    }
}
function isClosed(endToken, meta) {
    let closed = NodeClosed.Open;
    if (meta && meta.void) {
        closed = NodeClosed.VoidOmitted;
    }
    if (endToken.data[0] === "/>") {
        closed = NodeClosed.VoidSelfClosed;
    }
    return closed;
}

class DOMTree {
    constructor(location) {
        this.root = HtmlElement.rootNode(location);
        this.active = this.root;
        this.doctype = null;
    }
    pushActive(node) {
        this.active = node;
    }
    popActive() {
        if (this.active.isRootElement()) {
            /* root element should never be popped, continue as if nothing happened */
            return;
        }
        this.active = this.active.parent || this.root;
    }
    getActive() {
        return this.active;
    }
    /**
     * Resolve dynamic meta expressions.
     */
    resolveMeta(table) {
        this.visitDepthFirst((node) => table.resolve(node));
    }
    getElementsByTagName(tagName) {
        return this.root.getElementsByTagName(tagName);
    }
    visitDepthFirst(callback) {
        this.root.visitDepthFirst(callback);
    }
    find(callback) {
        return this.root.find(callback);
    }
    querySelector(selector) {
        return this.root.querySelector(selector);
    }
    querySelectorAll(selector) {
        return this.root.querySelectorAll(selector);
    }
}

const allowedKeys = ["exclude"];
/**
 * Helper class to validate elements against metadata rules.
 */
class Validator {
    /**
     * Test if element is used in a proper context.
     *
     * @param node - Element to test.
     * @param rules - List of rules.
     * @returns `true` if element passes all tests.
     */
    static validatePermitted(node, rules) {
        if (!rules) {
            return true;
        }
        return rules.some((rule) => {
            return Validator.validatePermittedRule(node, rule);
        });
    }
    /**
     * Test if an element is used the correct amount of times.
     *
     * For instance, a `<table>` element can only contain a single `<tbody>`
     * child. If multiple `<tbody>` exists this test will fail both nodes.
     *
     * @param node - Element to test.
     * @param rules - List of rules.
     * @param numSiblings - How many siblings of the same type as the element
     * exists (including the element itself)
     * @returns `true` if the element passes the test.
     */
    static validateOccurrences(node, rules, numSiblings) {
        if (!rules) {
            return true;
        }
        const category = rules.find((cur) => {
            /** @todo handle complex rules and not just plain arrays (but as of now
             * there is no use-case for it) */
            // istanbul ignore next
            if (typeof cur !== "string") {
                return false;
            }
            const match = cur.match(/^(.*?)[?*]?$/);
            return match && match[1] === node.tagName;
        });
        const limit = parseAmountQualifier(category);
        return limit === null || numSiblings <= limit;
    }
    /**
     * Validate elements order.
     *
     * Given a parent element with children and metadata containing permitted
     * order it will validate each children and ensure each one exists in the
     * specified order.
     *
     * For instance, for a `<table>` element the `<caption>` element must come
     * before a `<thead>` which must come before `<tbody>`.
     *
     * @param children - Array of children to validate.
     */
    static validateOrder(children, rules, cb) {
        if (!rules) {
            return true;
        }
        let i = 0;
        let prev = null;
        for (const node of children) {
            const old = i;
            while (rules[i] && !Validator.validatePermittedCategory(node, rules[i], true)) {
                i++;
            }
            if (i >= rules.length) {
                /* Second check is if the order is specified for this element at all. It
                 * will be unspecified in two cases:
                 * - disallowed elements
                 * - elements where the order doesn't matter
                 * In both of these cases no error should be reported. */
                const orderSpecified = rules.find((cur) => Validator.validatePermittedCategory(node, cur, true));
                if (orderSpecified) {
                    cb(node, prev);
                    return false;
                }
                /* if this element has unspecified order the index is restored so new
                 * elements of the same type can be specified again */
                i = old;
            }
            prev = node;
        }
        return true;
    }
    /**
     * Validate element ancestors.
     *
     * Check if an element has the required set of elements. At least one of the
     * selectors must match.
     */
    static validateAncestors(node, rules) {
        if (!rules || rules.length === 0) {
            return true;
        }
        return rules.some((rule) => node.closest(rule));
    }
    /**
     * Validate element required content.
     *
     * Check if an element has the required set of elements. At least one of the
     * selectors must match.
     *
     * Returns [] when valid or a list of tagNames missing as content.
     */
    static validateRequiredContent(node, rules) {
        if (!rules || rules.length === 0) {
            return [];
        }
        return rules.filter((tagName) => {
            const haveMatchingChild = node.childElements.some((child) => child.is(tagName));
            return !haveMatchingChild;
        });
    }
    /**
     * Test if an attribute has an allowed value and/or format.
     *
     * @param attr - Attribute to test.
     * @param rules - Element attribute metadta.
     * @returns `true` if attribute passes all tests.
     */
    static validateAttribute(attr, rules) {
        const rule = rules[attr.key];
        if (!rule) {
            return true;
        }
        /* consider dynamic values as valid as there is no way to properly test them
         * while using transformed sources, i.e. it must be tested when running in a
         * browser instead */
        const value = attr.value;
        if (value instanceof DynamicValue) {
            return true;
        }
        const empty = value === null || value === "";
        /* consider an empty array as being a boolean attribute */
        if (rule.length === 0) {
            return empty || value === attr.key;
        }
        /* if the empty string is present allow both "" and null
         * (boolean-attribute-style will regulate which is allowed) */
        if (rule.includes("") && empty) {
            return true;
        }
        if (value === null || value === undefined) {
            return false;
        }
        return rule.some((entry) => {
            if (entry instanceof RegExp) {
                return !!value.match(entry);
            }
            else {
                return value === entry;
            }
        });
    }
    static validatePermittedRule(node, rule, isExclude = false) {
        if (typeof rule === "string") {
            return Validator.validatePermittedCategory(node, rule, !isExclude);
        }
        else if (Array.isArray(rule)) {
            return rule.every((inner) => {
                return Validator.validatePermittedRule(node, inner, isExclude);
            });
        }
        else {
            validateKeys(rule);
            if (rule.exclude) {
                if (Array.isArray(rule.exclude)) {
                    return !rule.exclude.some((inner) => {
                        return Validator.validatePermittedRule(node, inner, true);
                    });
                }
                else {
                    return !Validator.validatePermittedRule(node, rule.exclude, true);
                }
            }
            else {
                return true;
            }
        }
    }
    /**
     * Validate node against a content category.
     *
     * When matching parent nodes against permitted parents use the superset
     * parameter to also match for `@flow`. E.g. if a node expects a `@phrasing`
     * parent it should also allow `@flow` parent since `@phrasing` is a subset of
     * `@flow`.
     *
     * @param node - The node to test against
     * @param category - Name of category with `@` prefix or tag name.
     * @param defaultMatch - The default return value when node categories is not known.
     */
    // eslint-disable-next-line complexity
    static validatePermittedCategory(node, category, defaultMatch) {
        /* match tagName when an explicit name is given */
        if (category[0] !== "@") {
            const [, tagName] = category.match(/^(.*?)[?*]?$/);
            return node.tagName === tagName;
        }
        /* if the meta entry is missing assume any content model would match */
        if (!node.meta) {
            return defaultMatch;
        }
        switch (category) {
            case "@meta":
                return node.meta.metadata;
            case "@flow":
                return node.meta.flow;
            case "@sectioning":
                return node.meta.sectioning;
            case "@heading":
                return node.meta.heading;
            case "@phrasing":
                return node.meta.phrasing;
            case "@embedded":
                return node.meta.embedded;
            case "@interactive":
                return node.meta.interactive;
            case "@script":
                return node.meta.scriptSupporting;
            case "@form":
                return node.meta.form;
            default:
                throw new Error(`Invalid content category "${category}"`);
        }
    }
}
function validateKeys(rule) {
    for (const key of Object.keys(rule)) {
        if (!allowedKeys.includes(key)) {
            const str = JSON.stringify(rule);
            throw new Error(`Permitted rule "${str}" contains unknown property "${key}"`);
        }
    }
}
function parseAmountQualifier(category) {
    if (!category) {
        /* content not allowed, catched by another rule so just assume unlimited
         * usage for this purpose */
        return null;
    }
    const [, qualifier] = category.match(/^.*?([?*]?)$/);
    switch (qualifier) {
        case "?":
            return 1;
        case "":
            return null;
        case "*":
            return null;
        /* istanbul ignore next */
        default:
            throw new Error(`Invalid amount qualifier "${qualifier}" used`);
    }
}

const $schema = "http://json-schema.org/draft-06/schema#";
const $id = "https://html-validate.org/schemas/config.json";
const type = "object";
const additionalProperties = false;
const properties = {
	$schema: {
		type: "string"
	},
	root: {
		type: "boolean",
		title: "Mark as root configuration",
		description: "If this is set to true no further configurations will be searched.",
		"default": false
	},
	"extends": {
		type: "array",
		items: {
			type: "string"
		},
		title: "Configurations to extend",
		description: "Array of shareable or builtin configurations to extend."
	},
	elements: {
		type: "array",
		items: {
			anyOf: [
				{
					type: "string"
				},
				{
					type: "object"
				}
			]
		},
		title: "Element metadata to load",
		description: "Array of modules, plugins or files to load element metadata from. Use <rootDir> to refer to the folder with the package.json file.",
		examples: [
			[
				"html-validate:recommended",
				"plugin:recommended",
				"module",
				"./local-file.json"
			]
		]
	},
	plugins: {
		type: "array",
		items: {
			type: "string"
		},
		title: "Plugins to load",
		description: "Array of plugins load. Use <rootDir> to refer to the folder with the package.json file.",
		examples: [
			[
				"my-plugin",
				"./local-plugin"
			]
		]
	},
	transform: {
		type: "object",
		additionalProperties: {
			type: "string"
		},
		title: "File transformations to use.",
		description: "Object where key is regular expression to match filename and value is name of transformer.",
		examples: [
			{
				"^.*\\.foo$": "my-transformer",
				"^.*\\.bar$": "my-plugin",
				"^.*\\.baz$": "my-plugin:named"
			}
		]
	},
	rules: {
		type: "object",
		patternProperties: {
			".*": {
				anyOf: [
					{
						"enum": [
							0,
							1,
							2,
							"off",
							"warn",
							"error"
						]
					},
					{
						type: "array",
						minItems: 1,
						maxItems: 1,
						items: [
							{
								"enum": [
									0,
									1,
									2,
									"off",
									"warn",
									"error"
								]
							}
						]
					},
					{
						type: "array",
						minItems: 2,
						maxItems: 2,
						items: [
							{
								"enum": [
									0,
									1,
									2,
									"off",
									"warn",
									"error"
								]
							},
							{
							}
						]
					}
				]
			}
		},
		title: "Rule configuration.",
		description: "Enable/disable rules, set severity. Some rules have additional configuration like style or patterns to use.",
		examples: [
			{
				foo: "error",
				bar: "off",
				baz: [
					"error",
					{
						style: "camelcase"
					}
				]
			}
		]
	}
};
var configurationSchema = {
	$schema: $schema,
	$id: $id,
	type: type,
	additionalProperties: additionalProperties,
	properties: properties
};

/* eslint-disable @typescript-eslint/no-non-null-assertion */
const espree = legacyRequire("espree");
const walk = legacyRequire("acorn-walk");
function joinTemplateLiteral(nodes) {
    let offset = nodes[0].start;
    let output = "";
    for (const node of nodes) {
        output += " ".repeat(node.start - offset);
        output += node.value.raw;
        offset = node.end;
    }
    return output;
}
/**
 * Compute source offset from line and column and the given markup.
 *
 * @param position - Line and column.
 * @param data - Source markup.
 * @returns The byte offset into the markup which line and column corresponds to.
 */
function computeOffset(position, data) {
    let line = position.line;
    let column = position.column + 1;
    for (let i = 0; i < data.length; i++) {
        if (line > 1) {
            /* not yet on the correct line */
            if (data[i] === "\n") {
                line--;
            }
        }
        else if (column > 1) {
            /* not yet on the correct column */
            column--;
        }
        else {
            /* line/column found, return current position */
            return i;
        }
    }
    /* istanbul ignore next: should never reach this line unless espree passes bad
     * positions, no sane way to test */
    throw new Error("Failed to compute location offset from position");
}
function extractLiteral(node, filename, data) {
    switch (node.type) {
        /* ignored nodes */
        case "FunctionExpression":
        case "Identifier":
            return null;
        case "Literal":
            if (typeof node.value !== "string") {
                return null;
            }
            return {
                data: node.value.toString(),
                filename,
                line: node.loc.start.line,
                column: node.loc.start.column + 1,
                offset: computeOffset(node.loc.start, data) + 1,
            };
        case "TemplateLiteral":
            return {
                data: joinTemplateLiteral(node.quasis),
                filename,
                line: node.loc.start.line,
                column: node.loc.start.column + 1,
                offset: computeOffset(node.loc.start, data) + 1,
            };
        case "TaggedTemplateExpression":
            return {
                data: joinTemplateLiteral(node.quasi.quasis),
                filename,
                line: node.quasi.loc.start.line,
                column: node.quasi.loc.start.column + 1,
                offset: computeOffset(node.quasi.loc.start, data) + 1,
            };
        case "ArrowFunctionExpression": {
            const whitelist = ["Literal", "TemplateLiteral"];
            if (whitelist.includes(node.body.type)) {
                return extractLiteral(node.body, filename, data);
            }
            else {
                return null;
            }
        }
        /* istanbul ignore next: this only provides a better error, all currently known nodes are tested */
        default: {
            const loc = node.loc.start;
            const context = `${filename}:${loc.line}:${loc.column}`;
            throw new Error(`Unhandled node type "${node.type}" at "${context}" in extractLiteral`);
        }
    }
}
function compareKey(node, key, filename) {
    switch (node.type) {
        case "Identifier":
            return node.name === key;
        case "Literal":
            return node.value === key;
        /* istanbul ignore next: this only provides a better error, all currently known nodes are tested */
        default: {
            const loc = node.loc.start;
            const context = `${filename}:${loc.line}:${loc.column}`;
            throw new Error(`Unhandled node type "${node.type}" at "${context}" in compareKey`);
        }
    }
}
class TemplateExtractor {
    constructor(ast, filename, data) {
        this.ast = ast;
        this.filename = filename;
        this.data = data;
    }
    static fromFilename(filename) {
        const source = fs.readFileSync(filename, "utf-8");
        const ast = espree.parse(source, {
            ecmaVersion: 2017,
            sourceType: "module",
            loc: true,
        });
        return new TemplateExtractor(ast, filename, source);
    }
    /**
     * Create a new [[TemplateExtractor]] from javascript source code.
     *
     * `Source` offsets will be relative to the string, i.e. offset 0 is the first
     * character of the string. If the string is only a subset of a larger string
     * the offsets must be adjusted manually.
     *
     * @param source - Source code.
     * @param filename - Optional filename to set in the resulting
     * `Source`. Defauls to `"inline"`.
     */
    static fromString(source, filename) {
        const ast = espree.parse(source, {
            ecmaVersion: 2017,
            sourceType: "module",
            loc: true,
        });
        return new TemplateExtractor(ast, filename || "inline", source);
    }
    /**
     * Convenience function to create a [[Source]] instance from an existing file.
     *
     * @param filename - Filename with javascript source code. The file must exist
     * and be readable by the user.
     * @returns An array of Source's suitable for passing to [[Engine]] linting
     * functions.
     */
    static createSource(filename) {
        const data = fs.readFileSync(filename, "utf-8");
        return [
            {
                column: 1,
                data,
                filename,
                line: 1,
                offset: 0,
            },
        ];
    }
    /**
     * Extract object properties.
     *
     * Given a key `"template"` this method finds all objects literals with a
     * `"template"` property and creates a [[Source]] instance with proper offsets
     * with the value of the property. For instance:
     *
     * ```
     * const myObj = {
     *   foo: 'bar',
     * };
     * ```
     *
     * The above snippet would yield a `Source` with the content `bar`.
     *
     */
    extractObjectProperty(key) {
        const result = [];
        const { filename, data } = this;
        walk.simple(this.ast, {
            Property(node) {
                if (compareKey(node.key, key, filename)) {
                    const source = extractLiteral(node.value, filename, data);
                    if (source) {
                        source.filename = filename;
                        result.push(source);
                    }
                }
            },
        });
        return result;
    }
}

var TRANSFORMER_API;
(function (TRANSFORMER_API) {
    TRANSFORMER_API[TRANSFORMER_API["VERSION"] = 1] = "VERSION";
})(TRANSFORMER_API || (TRANSFORMER_API = {}));

const name = "html-validate";
const version = "5.5.0";
const homepage = "https://html-validate.org";
const bugs = {
	url: "https://gitlab.com/html-validate/html-validate/issues/new"
};

var Severity;
(function (Severity) {
    Severity[Severity["DISABLED"] = 0] = "DISABLED";
    Severity[Severity["WARN"] = 1] = "WARN";
    Severity[Severity["ERROR"] = 2] = "ERROR";
})(Severity || (Severity = {}));
function parseSeverity(value) {
    switch (value) {
        case 0:
        case "off":
            return Severity.DISABLED;
        /* istanbul ignore next: deprecated code which will be removed later */
        case "disable":
            // eslint-disable-next-line no-console
            console.warn(`Deprecated alias "disabled" will be removed, replace with severity "off"`);
            return Severity.DISABLED;
        case 1:
        case "warn":
            return Severity.WARN;
        case 2:
        case "error":
            return Severity.ERROR;
        default:
            throw new Error(`Invalid severity "${value}"`);
    }
}

const remapEvents = {
    "tag:open": "tag:start",
    "tag:close": "tag:end",
};
const ajv$1 = new Ajv({ strict: true, strictTuples: true, strictTypes: true });
ajv$1.addMetaSchema(ajvSchemaDraft);
/**
 * Get (cached) schema validator for rule options.
 *
 * @param ruleId - Rule ID used as key for schema lookups.
 * @param properties - Uncompiled schema.
 */
function getSchemaValidator(ruleId, properties) {
    const $id = `rule/${ruleId}`;
    const cached = ajv$1.getSchema($id);
    if (cached) {
        return cached;
    }
    const schema = {
        $id,
        type: "object",
        additionalProperties: false,
        properties,
    };
    return ajv$1.compile(schema);
}
class Rule {
    constructor(options) {
        /* faux initialization, properly initialized by init(). This is to keep TS happy without adding null-checks everywhere */
        this.reporter = null;
        this.parser = null;
        this.meta = null;
        this.event = null;
        this.options = options;
        this.enabled = true;
        this.severity = 0;
        this.name = "";
    }
    getSeverity() {
        return this.severity;
    }
    setServerity(severity) {
        this.severity = severity;
    }
    setEnabled(enabled) {
        this.enabled = enabled;
    }
    /**
     * Returns `true` if rule is deprecated.
     *
     * Overridden by subclasses.
     */
    get deprecated() {
        return false;
    }
    /**
     * Test if rule is enabled.
     *
     * To be considered enabled the enabled flag must be true and the severity at
     * least warning.
     */
    isEnabled() {
        return this.enabled && this.severity >= Severity.WARN;
    }
    /**
     * Check if keyword is being ignored by the current rule configuration.
     *
     * This method requires the [[RuleOption]] type to include two properties:
     *
     * - include: string[] | null
     * - exclude: string[] | null
     *
     * This methods checks if the given keyword is included by "include" but not
     * excluded by "exclude". If any property is unset it is skipped by the
     * condition. Usually the user would use either one but not both but there is
     * no limitation to use both but the keyword must satisfy both conditions. If
     * either condition fails `true` is returned.
     *
     * For instance, given `{ include: ["foo"] }` the keyword `"foo"` would match
     * but not `"bar"`.
     *
     * Similarly, given `{ exclude: ["foo"] }` the keyword `"bar"` would match but
     * not `"foo"`.
     *
     * @param keyword - Keyword to match against `include` and `exclude` options.
     * @returns `true` if keyword is not present in `include` or is present in
     * `exclude`.
     */
    isKeywordIgnored(keyword) {
        const { include, exclude } = this.options;
        /* ignore keyword if not present in "include" */
        if (include && !include.includes(keyword)) {
            return true;
        }
        /* ignore keyword if present in "excludes" */
        if (exclude && exclude.includes(keyword)) {
            return true;
        }
        return false;
    }
    /**
     * Find all tags which has enabled given property.
     */
    getTagsWithProperty(propName) {
        return this.meta.getTagsWithProperty(propName);
    }
    /**
     * Find tag matching tagName or inheriting from it.
     */
    getTagsDerivedFrom(tagName) {
        return this.meta.getTagsDerivedFrom(tagName);
    }
    /**
     * JSON schema for rule options.
     *
     * Rules should override this to return an object with JSON schema to validate
     * rule options. If `null` or `undefined` is returned no validation is
     * performed.
     */
    static schema() {
        return null;
    }
    /**
     * Report a new error.
     *
     * Rule must be enabled both globally and on the specific node for this to
     * have any effect.
     */
    report(node, message, location, context) {
        if (this.isEnabled() && (!node || node.ruleEnabled(this.name))) {
            const where = this.findLocation({ node, location, event: this.event });
            this.reporter.add(this, message, this.severity, node, where, context);
        }
    }
    findLocation(src) {
        if (src.location) {
            return src.location;
        }
        if (src.event && src.event.location) {
            return src.event.location;
        }
        if (src.node && src.node.location) {
            return src.node.location;
        }
        return {};
    }
    on(event, ...args) {
        var _a;
        /* handle deprecated aliases */
        const remap = remapEvents[event];
        if (remap) {
            event = remap;
        }
        const callback = args.pop();
        const filter = (_a = args.pop()) !== null && _a !== void 0 ? _a : (() => true);
        return this.parser.on(event, (_event, data) => {
            if (this.isEnabled() && filter(data)) {
                this.event = data;
                callback(data);
            }
        });
    }
    /**
     * Called by [[Engine]] when initializing the rule.
     *
     * Do not override this, use the `setup` callback instead.
     *
     * @internal
     */
    init(parser, reporter, severity, meta) {
        this.parser = parser;
        this.reporter = reporter;
        this.severity = severity;
        this.meta = meta;
    }
    /**
     * Validate rule options against schema. Throws error if object does not validate.
     *
     * For rules without schema this function does nothing.
     *
     * @throws {@link SchemaValidationError}
     * Thrown when provided options does not validate against rule schema.
     *
     * @param cls - Rule class (constructor)
     * @param ruleId - Rule identifier
     * @param jsonPath - JSON path from which [[options]] can be found in [[config]]
     * @param options - User configured options to be validated
     * @param filename - Filename from which options originated
     * @param config - Configuration from which options originated
     *
     * @internal
     */
    static validateOptions(cls, ruleId, jsonPath, options, filename, config) {
        var _a;
        if (!cls) {
            return;
        }
        const schema = cls.schema();
        if (!schema) {
            return;
        }
        const isValid = getSchemaValidator(ruleId, schema);
        if (!isValid(options)) {
            /* istanbul ignore next: it is always set when validation fails */
            const errors = (_a = isValid.errors) !== null && _a !== void 0 ? _a : [];
            const mapped = errors.map((error) => {
                error.instancePath = `${jsonPath}${error.instancePath}`;
                return error;
            });
            throw new SchemaValidationError(filename, `Rule configuration error`, config, schema, mapped);
        }
    }
    /**
     * Rule documentation callback.
     *
     * Called when requesting additional documentation for a rule. Some rules
     * provide additional context to provide context-aware suggestions.
     *
     * @param context - Error context given by a reported error.
     * @returns Rule documentation and url with additional details or `null` if no
     * additional documentation is available.
     */
    /* eslint-disable-next-line @typescript-eslint/no-unused-vars */
    documentation(context) {
        return null;
    }
}
function ruleDocumentationUrl(filename) {
    /* during bundling all "@/rule.ts"'s are converted to paths relative to the src
     * folder and with the @/ prefix, by replacing the @ with the dist folder we
     * can resolve the path properly */
    filename = filename.replace("@", distFolder);
    const p = path.parse(filename);
    const root = path.join(distFolder, "rules");
    const rel = path.relative(root, path.join(p.dir, p.name));
    return `${homepage}/rules/${rel}.html`;
}

const defaults$p = {
    allowExternal: true,
    allowRelative: true,
    allowAbsolute: true,
    allowBase: true,
};
const mapping$1 = {
    a: "href",
    img: "src",
    link: "href",
    script: "src",
};
const description = {
    ["external" /* EXTERNAL */]: "External links are not allowed by current configuration.",
    ["relative-base" /* RELATIVE_BASE */]: "Links relative to <base> are not allowed by current configuration.",
    ["relative-path" /* RELATIVE_PATH */]: "Relative links are not allowed by current configuration.",
    ["absolute" /* ABSOLUTE */]: "Absolute links are not allowed by current configuration.",
    ["anchor" /* ANCHOR */]: null,
};
class AllowedLinks extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$p), options));
    }
    static schema() {
        return {
            allowAbsolute: {
                type: "boolean",
            },
            allowBase: {
                type: "boolean",
            },
            allowExternal: {
                type: "boolean",
            },
            allowRelative: {
                type: "boolean",
            },
        };
    }
    documentation(context) {
        const message = description[context] || "This link type is not allowed by current configuration";
        return {
            description: message,
            url: ruleDocumentationUrl("@/rules/allowed-links.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => {
            if (!event.value || !this.isRelevant(event)) {
                return;
            }
            const link = event.value.toString();
            const style = this.getStyle(link);
            switch (style) {
                case "anchor" /* ANCHOR */:
                    /* anchor links are always allowed by this rule */
                    break;
                case "absolute" /* ABSOLUTE */:
                    this.handleAbsolute(event, style);
                    break;
                case "external" /* EXTERNAL */:
                    this.handleExternal(event, style);
                    break;
                case "relative-base" /* RELATIVE_BASE */:
                    this.handleRelativeBase(event, style);
                    break;
                case "relative-path" /* RELATIVE_PATH */:
                    this.handleRelativePath(event, style);
                    break;
            }
        });
    }
    isRelevant(event) {
        const { target, key, value } = event;
        /* don't check links with dynamic values */
        if (value instanceof DynamicValue) {
            return false;
        }
        const attr = mapping$1[target.tagName];
        return Boolean(attr && attr === key);
    }
    getStyle(value) {
        /* http://example.net or //example.net */
        if (value.match(/^([a-z]+:)?\/\//g)) {
            return "external" /* EXTERNAL */;
        }
        switch (value[0]) {
            /* /foo/bar */
            case "/":
                return "absolute" /* ABSOLUTE */;
            /* ../foo/bar */
            case ".":
                return "relative-path" /* RELATIVE_PATH */;
            /* #foo */
            case "#":
                return "anchor" /* ANCHOR */;
            /* foo/bar */
            default:
                return "relative-base" /* RELATIVE_BASE */;
        }
    }
    handleAbsolute(event, style) {
        const { allowAbsolute } = this.options;
        if (!allowAbsolute) {
            this.report(event.target, "Link destination must not be absolute url", event.valueLocation, style);
        }
    }
    handleExternal(event, style) {
        const { allowExternal } = this.options;
        if (!allowExternal) {
            this.report(event.target, "Link destination must not be external url", event.valueLocation, style);
        }
    }
    handleRelativePath(event, style) {
        const { allowRelative } = this.options;
        if (!allowRelative) {
            this.report(event.target, "Link destination must not be relative url", event.valueLocation, style);
        }
    }
    handleRelativeBase(event, style) {
        const { allowRelative, allowBase } = this.options;
        if (!allowRelative) {
            this.report(event.target, "Link destination must not be relative url", event.valueLocation, style);
        }
        else if (!allowBase) {
            this.report(event.target, "Relative links must be relative to current folder", event.valueLocation, style);
        }
    }
}

const whitelisted = [
    "main",
    "nav",
    "table",
    "td",
    "th",
    "aside",
    "header",
    "footer",
    "section",
    "article",
    "form",
    "img",
    "area",
    "fieldset",
    "summary",
    "figure",
];
class AriaLabelMisuse extends Rule {
    documentation() {
        const valid = [
            "Interactive elements",
            "Labelable elements",
            "Landmark elements",
            "Elements with roles inheriting from widget",
            "`<area>`",
            "`<form>` and `<fieldset>`",
            "`<iframe>`",
            "`<img>` and `<figure>`",
            "`<summary>`",
            "`<table>`, `<td>` and `<th>`",
        ];
        const lines = valid.map((it) => `- ${it}\n`).join("");
        return {
            description: `\`aria-label\` can only be used on:\n\n${lines}`,
            url: ruleDocumentationUrl("@/rules/aria-label-misuse.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const { document } = event;
            for (const target of document.querySelectorAll("[aria-label]")) {
                this.validateElement(target);
            }
        });
    }
    validateElement(target) {
        const attr = target.getAttribute("aria-label");
        if (!attr || !attr.value || attr.valueMatches("", false)) {
            return;
        }
        /* ignore elements without meta */
        const meta = target.meta;
        if (!meta) {
            return;
        }
        /* ignore landmark and other whitelisted elements */
        if (whitelisted.includes(target.tagName)) {
            return;
        }
        /* ignore elements with role, @todo check if the role is widget or landmark */
        if (target.hasAttribute("role")) {
            return;
        }
        /* ignore elements with tabindex (implicit interactive) */
        if (target.hasAttribute("tabindex")) {
            return;
        }
        /* ignore interactive and labelable elements */
        if (meta.interactive || meta.labelable) {
            return;
        }
        this.report(target, `"aria-label" cannot be used on this element`, attr.keyLocation);
    }
}

class ConfigError extends UserError {
}

/**
 * Represents casing for a name, e.g. lowercase, uppercase, etc.
 */
class CaseStyle {
    /**
     * @param style - Name of a valid case style.
     */
    constructor(style, ruleId) {
        if (!Array.isArray(style)) {
            style = [style];
        }
        if (style.length === 0) {
            throw new ConfigError(`Missing style for ${ruleId} rule`);
        }
        this.styles = this.parseStyle(style, ruleId);
    }
    /**
     * Test if a text matches this case style.
     */
    match(text) {
        return this.styles.some((style) => text.match(style.pattern));
    }
    get name() {
        const names = this.styles.map((style) => style.name);
        switch (this.styles.length) {
            case 1:
                return names[0];
            case 2:
                return names.join(" or ");
            default: {
                const last = names.slice(-1);
                const rest = names.slice(0, -1);
                return `${rest.join(", ")} or ${last}`;
            }
        }
    }
    parseStyle(style, ruleId) {
        return style.map((cur) => {
            switch (cur.toLowerCase()) {
                case "lowercase":
                    return { pattern: /^[a-z]*$/, name: "lowercase" };
                case "uppercase":
                    return { pattern: /^[A-Z]*$/, name: "uppercase" };
                case "pascalcase":
                    return { pattern: /^[A-Z][A-Za-z]*$/, name: "PascalCase" };
                case "camelcase":
                    return { pattern: /^[a-z][A-Za-z]*$/, name: "camelCase" };
                default:
                    throw new ConfigError(`Invalid style "${style}" for ${ruleId} rule`);
            }
        });
    }
}

const defaults$o = {
    style: "lowercase",
    ignoreForeign: true,
};
class AttrCase extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$o), options));
        this.style = new CaseStyle(this.options.style, "attr-case");
    }
    static schema() {
        const styleEnum = ["lowercase", "uppercase", "pascalcase", "camelcase"];
        return {
            ignoreForeign: {
                type: "boolean",
            },
            style: {
                anyOf: [
                    {
                        enum: styleEnum,
                        type: "string",
                    },
                    {
                        items: {
                            enum: styleEnum,
                            type: "string",
                        },
                        type: "array",
                    },
                ],
            },
        };
    }
    documentation() {
        return {
            description: `Attribute name must be ${this.options.style}.`,
            url: ruleDocumentationUrl("@/rules/attr-case.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => {
            if (this.isIgnored(event.target)) {
                return;
            }
            /* ignore case for dynamic attributes, the original attributes will be
             * checked instead (this prevents duplicated errors for the same source
             * attribute) */
            if (event.originalAttribute) {
                return;
            }
            const letters = event.key.replace(/[^a-z]+/gi, "");
            if (!this.style.match(letters)) {
                this.report(event.target, `Attribute "${event.key}" should be ${this.style.name}`, event.keyLocation);
            }
        });
    }
    isIgnored(node) {
        if (this.options.ignoreForeign) {
            return Boolean(node.meta && node.meta.foreign);
        }
        else {
            return false;
        }
    }
}

var TokenType;
(function (TokenType) {
    TokenType[TokenType["UNICODE_BOM"] = 1] = "UNICODE_BOM";
    TokenType[TokenType["WHITESPACE"] = 2] = "WHITESPACE";
    TokenType[TokenType["NEWLINE"] = 3] = "NEWLINE";
    TokenType[TokenType["DOCTYPE_OPEN"] = 4] = "DOCTYPE_OPEN";
    TokenType[TokenType["DOCTYPE_VALUE"] = 5] = "DOCTYPE_VALUE";
    TokenType[TokenType["DOCTYPE_CLOSE"] = 6] = "DOCTYPE_CLOSE";
    TokenType[TokenType["TAG_OPEN"] = 7] = "TAG_OPEN";
    TokenType[TokenType["TAG_CLOSE"] = 8] = "TAG_CLOSE";
    TokenType[TokenType["ATTR_NAME"] = 9] = "ATTR_NAME";
    TokenType[TokenType["ATTR_VALUE"] = 10] = "ATTR_VALUE";
    TokenType[TokenType["TEXT"] = 11] = "TEXT";
    TokenType[TokenType["TEMPLATING"] = 12] = "TEMPLATING";
    TokenType[TokenType["SCRIPT"] = 13] = "SCRIPT";
    TokenType[TokenType["COMMENT"] = 14] = "COMMENT";
    TokenType[TokenType["CONDITIONAL"] = 15] = "CONDITIONAL";
    TokenType[TokenType["DIRECTIVE"] = 16] = "DIRECTIVE";
    TokenType[TokenType["EOF"] = 17] = "EOF";
})(TokenType || (TokenType = {}));

/* eslint-disable no-useless-escape */
const MATCH_UNICODE_BOM = /^\uFEFF/;
const MATCH_WHITESPACE = /^(?:\r\n|\r|\n|[ \t]+(?:\r\n|\r|\n)?)/;
const MATCH_DOCTYPE_OPEN = /^<!(DOCTYPE)\s/i;
const MATCH_DOCTYPE_VALUE = /^[^>]+/;
const MATCH_DOCTYPE_CLOSE = /^>/;
const MATCH_XML_TAG = /^<\?xml.*?\?>\s+/;
const MATCH_TAG_OPEN = /^<(\/?)([a-zA-Z0-9\-:]+)/; // https://www.w3.org/TR/html/syntax.html#start-tags
const MATCH_TAG_CLOSE = /^\/?>/;
const MATCH_TEXT = /^[^]*?(?=(?:[ \t]*(?:\r\n|\r|\n)|<[^ ]|$))/;
const MATCH_TEMPLATING = /^(?:<%.*?%>|<\?.*?\?>|<\$.*?\$>)/;
const MATCH_TAG_LOOKAHEAD = /^[^]*?(?=<|$)/;
const MATCH_ATTR_START = /^([^\t\r\n\f \/><"'=]+)/; // https://www.w3.org/TR/html/syntax.html#elements-attributes
const MATCH_ATTR_SINGLE = /^(\s*=\s*)'([^']*?)(')/;
const MATCH_ATTR_DOUBLE = /^(\s*=\s*)"([^"]*?)(")/;
const MATCH_ATTR_UNQUOTED = /^(\s*=\s*)([^\t\r\n\f "'<>][^\t\r\n\f <>]*)/;
const MATCH_CDATA_BEGIN = /^<!\[CDATA\[/;
const MATCH_CDATA_END = /^[^]*?]]>/;
const MATCH_SCRIPT_DATA = /^[^]*?(?=<\/script)/;
const MATCH_SCRIPT_END = /^<(\/)(script)/;
const MATCH_DIRECTIVE = /^<!--\s*\[html-validate-(.*?)]\s*-->/;
const MATCH_COMMENT = /^<!--([^]*?)-->/;
const MATCH_CONDITIONAL = /^<!\[([^\]]*?)\]>/;
class InvalidTokenError extends Error {
    constructor(location, message) {
        super(message);
        this.location = location;
    }
}
class Lexer {
    // eslint-disable-next-line complexity
    *tokenize(source) {
        const context = new Context(source);
        /* for sanity check */
        let previousState = context.state;
        let previousLength = context.string.length;
        while (context.string.length > 0) {
            switch (context.state) {
                case State.INITIAL:
                    yield* this.tokenizeInitial(context);
                    break;
                case State.DOCTYPE:
                    yield* this.tokenizeDoctype(context);
                    break;
                case State.TAG:
                    yield* this.tokenizeTag(context);
                    break;
                case State.ATTR:
                    yield* this.tokenizeAttr(context);
                    break;
                case State.TEXT:
                    yield* this.tokenizeText(context);
                    break;
                case State.CDATA:
                    yield* this.tokenizeCDATA(context);
                    break;
                case State.SCRIPT:
                    yield* this.tokenizeScript(context);
                    break;
                /* istanbul ignore next: sanity check: should not happen unless adding new states */
                default:
                    this.unhandled(context);
            }
            /* sanity check: state or string must change, if both are intact
             * we are stuck in an endless loop. */
            /* istanbul ignore next: no easy way to test this as it is a condition which should never happen */
            if (context.state === previousState && context.string.length === previousLength) {
                this.errorStuck(context);
            }
            previousState = context.state;
            previousLength = context.string.length;
        }
        yield this.token(context, TokenType.EOF);
    }
    token(context, type, data) {
        const size = data ? data[0].length : 0;
        const location = context.getLocation(size);
        return {
            type,
            location,
            data: data ? Array.from(data) : null,
        };
    }
    /* istanbul ignore next: used to provide a better error when an unhandled state happens */
    unhandled(context) {
        const truncated = JSON.stringify(context.string.length > 13 ? `${context.string.slice(0, 15)}...` : context.string);
        const state = State[context.state];
        const message = `failed to tokenize ${truncated}, unhandled state ${state}.`;
        throw new InvalidTokenError(context.getLocation(1), message);
    }
    /* istanbul ignore next: used to provide a better error when lexer is detected to be stuck, no known way to reproduce */
    errorStuck(context) {
        const state = State[context.state];
        const message = `failed to tokenize ${context.getTruncatedLine()}, state ${state} failed to consume data or change state.`;
        throw new InvalidTokenError(context.getLocation(1), message);
    }
    evalNextState(nextState, token) {
        if (typeof nextState === "function") {
            return nextState(token);
        }
        else {
            return nextState;
        }
    }
    *match(context, tests, error) {
        let match = null;
        const n = tests.length;
        for (let i = 0; i < n; i++) {
            const [regex, nextState, tokenType] = tests[i];
            if (regex === false || (match = context.string.match(regex))) {
                let token = null;
                if (tokenType !== false) {
                    yield (token = this.token(context, tokenType, match));
                }
                const state = this.evalNextState(nextState, token);
                context.consume(match || 0, state);
                this.enter(context, state, match);
                return;
            }
        }
        const message = `failed to tokenize ${context.getTruncatedLine()}, ${error}.`;
        throw new InvalidTokenError(context.getLocation(1), message);
    }
    /**
     * Called when entering a new state.
     */
    enter(context, state, data) {
        /* script tags require a different content model */
        if (state === State.TAG && data && data[0][0] === "<") {
            if (data[0] === "<script") {
                context.contentModel = ContentModel.SCRIPT;
            }
            else {
                context.contentModel = ContentModel.TEXT;
            }
        }
    }
    *tokenizeInitial(context) {
        yield* this.match(context, [
            [MATCH_UNICODE_BOM, State.INITIAL, TokenType.UNICODE_BOM],
            [MATCH_XML_TAG, State.INITIAL, false],
            [MATCH_DOCTYPE_OPEN, State.DOCTYPE, TokenType.DOCTYPE_OPEN],
            [MATCH_WHITESPACE, State.INITIAL, TokenType.WHITESPACE],
            [MATCH_DIRECTIVE, State.INITIAL, TokenType.DIRECTIVE],
            [MATCH_CONDITIONAL, State.INITIAL, TokenType.CONDITIONAL],
            [MATCH_COMMENT, State.INITIAL, TokenType.COMMENT],
            [false, State.TEXT, false],
        ], "expected doctype");
    }
    *tokenizeDoctype(context) {
        yield* this.match(context, [
            [MATCH_WHITESPACE, State.DOCTYPE, TokenType.WHITESPACE],
            [MATCH_DOCTYPE_VALUE, State.DOCTYPE, TokenType.DOCTYPE_VALUE],
            [MATCH_DOCTYPE_CLOSE, State.TEXT, TokenType.DOCTYPE_CLOSE],
        ], "expected doctype name");
    }
    *tokenizeTag(context) {
        function nextState(token) {
            switch (context.contentModel) {
                case ContentModel.TEXT:
                    return State.TEXT;
                case ContentModel.SCRIPT:
                    if (token && token.data[0][0] !== "/") {
                        return State.SCRIPT;
                    }
                    else {
                        return State.TEXT; /* <script/> (not legal but handle it anyway so the lexer doesn't choke on it) */
                    }
            }
            /* istanbul ignore next: not covered by a test as there is currently no
             * way to trigger this unless new content models are added but this will
             * add a saner default if anyone ever does */
            return context.contentModel !== ContentModel.SCRIPT ? State.TEXT : State.SCRIPT;
        }
        yield* this.match(context, [
            [MATCH_TAG_CLOSE, nextState, TokenType.TAG_CLOSE],
            [MATCH_ATTR_START, State.ATTR, TokenType.ATTR_NAME],
            [MATCH_WHITESPACE, State.TAG, TokenType.WHITESPACE],
        ], 'expected attribute, ">" or "/>"');
    }
    *tokenizeAttr(context) {
        yield* this.match(context, [
            [MATCH_ATTR_SINGLE, State.TAG, TokenType.ATTR_VALUE],
            [MATCH_ATTR_DOUBLE, State.TAG, TokenType.ATTR_VALUE],
            [MATCH_ATTR_UNQUOTED, State.TAG, TokenType.ATTR_VALUE],
            [false, State.TAG, false],
        ], 'expected attribute, ">" or "/>"');
    }
    *tokenizeText(context) {
        yield* this.match(context, [
            [MATCH_WHITESPACE, State.TEXT, TokenType.WHITESPACE],
            [MATCH_CDATA_BEGIN, State.CDATA, false],
            [MATCH_DIRECTIVE, State.TEXT, TokenType.DIRECTIVE],
            [MATCH_CONDITIONAL, State.TEXT, TokenType.CONDITIONAL],
            [MATCH_COMMENT, State.TEXT, TokenType.COMMENT],
            [MATCH_TEMPLATING, State.TEXT, TokenType.TEMPLATING],
            [MATCH_TAG_OPEN, State.TAG, TokenType.TAG_OPEN],
            [MATCH_TEXT, State.TEXT, TokenType.TEXT],
            [MATCH_TAG_LOOKAHEAD, State.TEXT, TokenType.TEXT],
        ], 'expected text or "<"');
    }
    *tokenizeCDATA(context) {
        yield* this.match(context, [[MATCH_CDATA_END, State.TEXT, false]], "expected ]]>");
    }
    *tokenizeScript(context) {
        yield* this.match(context, [
            [MATCH_SCRIPT_END, State.TAG, TokenType.TAG_OPEN],
            [MATCH_SCRIPT_DATA, State.SCRIPT, TokenType.SCRIPT],
        ], "expected </script>");
    }
}

const whitespace = /(\s+)/;
function isRelevant$3(event) {
    return event.type === TokenType.ATTR_VALUE;
}
class AttrDelimiter extends Rule {
    documentation() {
        return {
            description: `Attribute value should be separated by `,
            url: ruleDocumentationUrl("@/rules/attr-delimiter.ts"),
        };
    }
    setup() {
        this.on("token", isRelevant$3, (event) => {
            const delimiter = event.data[1];
            const match = whitespace.exec(delimiter);
            if (match) {
                const location = sliceLocation(event.location, 0, delimiter.length);
                this.report(null, "Attribute value must not be delimited by whitespace", location);
            }
        });
    }
}

const DEFAULT_PATTERN = "[a-z0-9-:]+";
const defaults$n = {
    pattern: DEFAULT_PATTERN,
    ignoreForeign: true,
};
function generateRegexp(pattern) {
    if (Array.isArray(pattern)) {
        /* eslint-disable-next-line security/detect-non-literal-regexp */
        return new RegExp(`^(${pattern.join("|")})$`, "i");
    }
    else {
        /* eslint-disable-next-line security/detect-non-literal-regexp */
        return new RegExp(`^${pattern}$`, "i");
    }
}
function generateMessage(name, pattern) {
    if (Array.isArray(pattern)) {
        const patterns = pattern.map((it) => `/${it}/`).join(", ");
        return `Attribute "${name}" should match one of [${patterns}]`;
    }
    else {
        return `Attribute "${name}" should match /${pattern}/`;
    }
}
function generateDescription(name, pattern) {
    if (Array.isArray(pattern)) {
        return [
            `Attribute "${name}" should match one of the configured regular expressions:`,
            "",
            ...pattern.map((it) => `- \`/${it}/\``),
        ].join("\n");
    }
    else {
        return `Attribute "${name}" should match the regular expression \`/${pattern}/\``;
    }
}
class AttrPattern extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$n), options));
        this.pattern = generateRegexp(this.options.pattern);
    }
    static schema() {
        return {
            pattern: {
                oneOf: [{ type: "array", items: { type: "string" }, minItems: 1 }, { type: "string" }],
            },
            ignoreForeign: {
                type: "boolean",
            },
        };
    }
    documentation(context) {
        let description;
        if (context) {
            description = generateDescription(context.attr, context.pattern);
        }
        else {
            description = `Attribute should match configured pattern`;
        }
        return {
            description,
            url: ruleDocumentationUrl("@/rules/attr-pattern.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => {
            if (this.isIgnored(event.target)) {
                return;
            }
            /* ignore case for dynamic attributes, the original attributes will be
             * checked instead (this prevents duplicated errors for the same source
             * attribute) */
            if (event.originalAttribute) {
                return;
            }
            if (this.pattern.test(event.key)) {
                return;
            }
            const message = generateMessage(event.key, this.options.pattern);
            this.report(event.target, message, event.keyLocation);
        });
    }
    isIgnored(node) {
        if (this.options.ignoreForeign) {
            return Boolean(node.meta && node.meta.foreign);
        }
        else {
            return false;
        }
    }
}

var QuoteStyle;
(function (QuoteStyle) {
    QuoteStyle["SINGLE_QUOTE"] = "'";
    QuoteStyle["DOUBLE_QUOTE"] = "\"";
    QuoteStyle["AUTO_QUOTE"] = "auto";
})(QuoteStyle || (QuoteStyle = {}));
const defaults$m = {
    style: "auto",
    unquoted: false,
};
class AttrQuotes extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$m), options));
        this.style = parseStyle$4(this.options.style);
    }
    static schema() {
        return {
            style: {
                enum: ["auto", "double", "single"],
                type: "string",
            },
            unquoted: {
                type: "boolean",
            },
        };
    }
    documentation() {
        if (this.options.style === "auto") {
            return {
                description: `Attribute values are required to be quoted with doublequotes unless the attribute value itself contains doublequotes in which case singlequotes should be used.`,
                url: ruleDocumentationUrl("@/rules/attr-quotes.ts"),
            };
        }
        else {
            return {
                description: `Attribute values are required to be quoted with ${this.options.style}quotes.`,
                url: ruleDocumentationUrl("@/rules/attr-quotes.ts"),
            };
        }
    }
    setup() {
        this.on("attr", (event) => {
            /* ignore attributes with no value */
            if (event.value === null) {
                return;
            }
            if (!event.quote) {
                if (this.options.unquoted === false) {
                    this.report(event.target, `Attribute "${event.key}" using unquoted value`);
                }
                return;
            }
            const expected = this.resolveQuotemark(event.value.toString(), this.style);
            if (event.quote !== expected) {
                this.report(event.target, `Attribute "${event.key}" used ${event.quote} instead of expected ${expected}`);
            }
        });
    }
    resolveQuotemark(value, style) {
        if (style === QuoteStyle.AUTO_QUOTE) {
            return value.includes('"') ? "'" : '"';
        }
        else {
            return style;
        }
    }
}
function parseStyle$4(style) {
    switch (style.toLowerCase()) {
        case "auto":
            return QuoteStyle.AUTO_QUOTE;
        case "double":
            return QuoteStyle.DOUBLE_QUOTE;
        case "single":
            return QuoteStyle.SINGLE_QUOTE;
        /* istanbul ignore next: covered by schema validation */
        default:
            throw new ConfigError(`Invalid style "${style}" for "attr-quotes" rule`);
    }
}

class AttrSpacing extends Rule {
    documentation() {
        return {
            description: `No space between attributes. At least one whitespace character (commonly space) must be used to separate attributes.`,
            url: ruleDocumentationUrl("@/rules/attr-spacing.ts"),
        };
    }
    setup() {
        let previousToken;
        this.on("token", (event) => {
            if (event.type === TokenType.ATTR_NAME && previousToken !== TokenType.WHITESPACE) {
                this.report(null, "No space between attributes", event.location);
            }
            previousToken = event.type;
        });
    }
}

class AttributeAllowedValues extends Rule {
    documentation(context) {
        const docs = {
            description: "Attribute has invalid value.",
            url: ruleDocumentationUrl("@/rules/attribute-allowed-values.ts"),
        };
        if (!context) {
            return docs;
        }
        if (context.allowed.length > 0) {
            const allowed = context.allowed.map((val) => `- \`${val}\``);
            docs.description = `Element <${context.element}> does not allow attribute \`${context.attribute}\` to have the value \`"${context.value}"\`, it must match one of the following:\n\n${allowed.join("\n")}`;
        }
        else {
            docs.description = `Element <${context.element}> attribute \`${context.attribute}\` must be a boolean attribute, e.g. \`<${context.element} ${context.attribute}>\``;
        }
        return docs;
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            doc.visitDepthFirst((node) => {
                const meta = node.meta;
                /* ignore rule if element has no meta or meta does not specify attribute
                 * allowed values */
                if (!meta || !meta.attributes)
                    return;
                for (const attr of node.attributes) {
                    if (Validator.validateAttribute(attr, meta.attributes)) {
                        continue;
                    }
                    const value = attr.value ? attr.value.toString() : "";
                    const context = {
                        element: node.tagName,
                        attribute: attr.key,
                        value,
                        allowed: meta.attributes[attr.key],
                    };
                    const message = this.getMessage(attr);
                    const location = this.getLocation(attr);
                    this.report(node, message, location, context);
                }
            });
        });
    }
    getMessage(attr) {
        const { key, value } = attr;
        if (value !== null) {
            return `Attribute "${key}" has invalid value "${value.toString()}"`;
        }
        else {
            return `Attribute "${key}" is missing value`;
        }
    }
    getLocation(attr) {
        if (attr.value !== null) {
            return attr.valueLocation;
        }
        else {
            return attr.keyLocation;
        }
    }
}

const defaults$l = {
    style: "omit",
};
class AttributeBooleanStyle extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$l), options));
        this.hasInvalidStyle = parseStyle$3(this.options.style);
    }
    static schema() {
        return {
            style: {
                enum: ["empty", "name", "omit"],
                type: "string",
            },
        };
    }
    documentation() {
        return {
            description: "Require a specific style when writing boolean attributes.",
            url: ruleDocumentationUrl("@/rules/attribute-boolean-style.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            doc.visitDepthFirst((node) => {
                const meta = node.meta;
                /* ignore rule if element has no meta or meta does not specify attribute
                 * allowed values */
                if (!meta || !meta.attributes)
                    return;
                /* check all boolean attributes */
                for (const attr of node.attributes) {
                    if (!this.isBoolean(attr, meta.attributes))
                        continue;
                    /* ignore attribute if it is aliased by a dynamic value,
                     * e.g. ng-required or v-bind:required, since it will probably have a
                     * value despite the target attribute is a boolean. The framework is
                     * assumed to handle it properly */
                    if (attr.originalAttribute) {
                        continue;
                    }
                    if (this.hasInvalidStyle(attr)) {
                        this.report(node, reportMessage$1(attr, this.options.style), attr.keyLocation);
                    }
                }
            });
        });
    }
    isBoolean(attr, rules) {
        return rules[attr.key] && rules[attr.key].length === 0;
    }
}
function parseStyle$3(style) {
    switch (style.toLowerCase()) {
        case "omit":
            return (attr) => attr.value !== null;
        case "empty":
            return (attr) => attr.value !== "";
        case "name":
            return (attr) => attr.value !== attr.key;
        /* istanbul ignore next: covered by schema validation */
        default:
            throw new Error(`Invalid style "${style}" for "attribute-boolean-style" rule`);
    }
}
function reportMessage$1(attr, style) {
    const key = attr.key;
    switch (style.toLowerCase()) {
        case "omit":
            return `Attribute "${key}" should omit value`;
        case "empty":
            return `Attribute "${key}" value should be empty string`;
        case "name":
            return `Attribute "${key}" should be set to ${key}="${key}"`;
    }
    /* istanbul ignore next: the above switch should cover all cases */
    return "";
}

const defaults$k = {
    style: "omit",
};
class AttributeEmptyStyle extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$k), options));
        this.hasInvalidStyle = parseStyle$2(this.options.style);
    }
    static schema() {
        return {
            style: {
                enum: ["empty", "omit"],
                type: "string",
            },
        };
    }
    documentation() {
        return {
            description: "Require a specific style for attributes with empty values.",
            url: ruleDocumentationUrl("@/rules/attribute-empty-style.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            doc.visitDepthFirst((node) => {
                const meta = node.meta;
                /* ignore rule if element has no meta or meta does not specify attribute
                 * allowed values */
                if (!meta || !meta.attributes)
                    return;
                /* check all boolean attributes */
                for (const attr of node.attributes) {
                    /* only handle attributes which allows empty values */
                    if (!allowsEmpty(attr, meta.attributes)) {
                        continue;
                    }
                    /* skip attribute if the attribute is set to non-empty value
                     * (attribute-allowed-values deals with non-empty values)*/
                    if (!isEmptyValue(attr)) {
                        continue;
                    }
                    /* skip attribute if the style is valid */
                    if (!this.hasInvalidStyle(attr)) {
                        continue;
                    }
                    /* report error */
                    this.report(node, reportMessage(attr, this.options.style), attr.keyLocation);
                }
            });
        });
    }
}
function allowsEmpty(attr, rules) {
    return rules[attr.key] && rules[attr.key].includes("");
}
function isEmptyValue(attr) {
    /* dynamic values are ignored, assumed to contain a value */
    if (attr.isDynamic) {
        return false;
    }
    return attr.value === null || attr.value === "";
}
function parseStyle$2(style) {
    switch (style.toLowerCase()) {
        case "omit":
            return (attr) => attr.value !== null;
        case "empty":
            return (attr) => attr.value !== "";
        /* istanbul ignore next: covered by schema validation */
        default:
            throw new Error(`Invalid style "${style}" for "attribute-empty-style" rule`);
    }
}
function reportMessage(attr, style) {
    const key = attr.key;
    switch (style.toLowerCase()) {
        case "omit":
            return `Attribute "${key}" should omit value`;
        case "empty":
            return `Attribute "${key}" value should be empty string`;
    }
    /* istanbul ignore next: the above switch should cover all cases */
    return "";
}

function parsePattern(pattern) {
    switch (pattern) {
        case "kebabcase":
            return /^[a-z0-9-]+$/;
        case "camelcase":
            return /^[a-z][a-zA-Z0-9]+$/;
        case "underscore":
            return /^[a-z0-9_]+$/;
        default:
            // eslint-disable-next-line security/detect-non-literal-regexp
            return new RegExp(pattern);
    }
}
function describePattern(pattern) {
    const regexp = parsePattern(pattern).toString();
    switch (pattern) {
        case "kebabcase":
        case "camelcase":
        case "underscore": {
            return `${regexp} (${pattern})`;
        }
        default:
            return regexp;
    }
}

const defaults$j = {
    pattern: "kebabcase",
};
class ClassPattern extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$j), options));
        this.pattern = parsePattern(this.options.pattern);
    }
    static schema() {
        return {
            pattern: {
                type: "string",
            },
        };
    }
    documentation() {
        const pattern = describePattern(this.options.pattern);
        return {
            description: `For consistency all classes are required to match the pattern ${pattern}.`,
            url: ruleDocumentationUrl("@/rules/class-pattern.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => {
            if (event.key.toLowerCase() !== "class") {
                return;
            }
            const classes = new DOMTokenList(event.value, event.valueLocation);
            classes.forEach((cur, index) => {
                if (!cur.match(this.pattern)) {
                    const location = classes.location(index);
                    this.report(event.target, `Class "${cur}" does not match required pattern "${this.pattern}"`, location);
                }
            });
        });
    }
}

class CloseAttr extends Rule {
    documentation() {
        return {
            description: "HTML disallows end tags to have attributes.",
            url: ruleDocumentationUrl("@/rules/close-attr.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            /* handle unclosed tags */
            if (!event.target) {
                return;
            }
            /* ignore self-closed and void */
            if (event.previous === event.target) {
                return;
            }
            const node = event.target;
            if (Object.keys(node.attributes).length > 0) {
                const first = node.attributes[0];
                this.report(null, "Close tags cannot have attributes", first.keyLocation);
            }
        });
    }
}

class CloseOrder extends Rule {
    documentation() {
        return {
            description: "HTML requires elements to be closed in the same order as they were opened.",
            url: ruleDocumentationUrl("@/rules/close-order.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const current = event.target; // The current element being closed
            const active = event.previous; // The current active element (that is, the current element on the stack)
            /* handle unclosed tags */
            if (!current) {
                this.report(null, `Missing close-tag, expected '</${active.tagName}>' but document ended before it was found.`, event.location);
                return;
            }
            /* void elements are always closed in correct order but if the markup contains
             * an end-tag for it it should be ignored here since the void element is
             * implicitly closed in the right order, so the current active element is the
             * parent. */
            if (current.voidElement) {
                return;
            }
            /* if the active element is implicitly closed when the parent is closed
             * (such as a <li> by </ul>) no error should be reported. */
            if (active.closed === NodeClosed.ImplicitClosed) {
                return;
            }
            /* handle unopened tags */
            if (!active || active.isRootElement()) {
                const location = {
                    filename: current.location.filename,
                    line: current.location.line,
                    column: current.location.column,
                    offset: current.location.offset,
                    size: current.tagName.length + 1,
                };
                this.report(null, "Unexpected close-tag, expected opening tag.", location);
                return;
            }
            /* check for matching tagnames */
            if (current.tagName !== active.tagName) {
                this.report(null, `Mismatched close-tag, expected '</${active.tagName}>' but found '</${current.tagName}>'.`, current.location);
            }
        });
    }
}

const defaults$i = {
    include: null,
    exclude: null,
};
class Deprecated extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$i), options));
    }
    static schema() {
        return {
            exclude: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
            include: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
        };
    }
    documentation(context) {
        const doc = {
            description: "This element is deprecated and should not be used in new code.",
            url: ruleDocumentationUrl("@/rules/deprecated.ts"),
        };
        if (context) {
            const text = [];
            if (context.source) {
                const source = prettySource(context.source);
                const message = `The \`<$tagname>\` element is deprecated ${source} and should not be used in new code.`;
                text.push(message);
            }
            else {
                const message = `The \`<$tagname>\` element is deprecated and should not be used in new code.`;
                text.push(message);
            }
            if (context.documentation) {
                text.push(context.documentation);
            }
            doc.description = text.map((cur) => cur.replace(/\$tagname/g, context.tagName)).join("\n\n");
        }
        return doc;
    }
    setup() {
        this.on("tag:start", (event) => {
            const node = event.target;
            /* cannot validate if meta isn't known */
            if (node.meta === null) {
                return;
            }
            /* ignore if element is not deprecated */
            const deprecated = node.meta.deprecated;
            if (!deprecated) {
                return;
            }
            /* ignore if element is ignored by used configuration */
            if (this.isKeywordIgnored(node.tagName)) {
                return;
            }
            const location = sliceLocation(event.location, 1);
            if (typeof deprecated === "string") {
                this.reportString(deprecated, node, location);
            }
            else if (typeof deprecated === "boolean") {
                this.reportBoolean(node, location);
            }
            else {
                this.reportObject(deprecated, node, location);
            }
        });
    }
    reportString(deprecated, node, location) {
        const context = { tagName: node.tagName };
        const message = `<${node.tagName}> is deprecated: ${deprecated}`;
        this.report(node, message, location, context);
    }
    reportBoolean(node, location) {
        const context = { tagName: node.tagName };
        const message = `<${node.tagName}> is deprecated`;
        this.report(node, message, location, context);
    }
    reportObject(deprecated, node, location) {
        const context = Object.assign(Object.assign({}, deprecated), { tagName: node.tagName });
        const notice = deprecated.message ? `: ${deprecated.message}` : "";
        const message = `<${node.tagName}> is deprecated${notice}`;
        this.report(node, message, location, context);
    }
}
function prettySource(source) {
    const match = source.match(/html(\d)(\d)?/);
    if (match) {
        const [, ...parts] = match;
        const version = parts.filter(Boolean).join(".");
        return `in HTML ${version}`;
    }
    switch (source) {
        case "whatwg":
            return "in HTML Living Standard";
        case "non-standard":
            return "and non-standard";
        default:
            return `by ${source}`;
    }
}

class DeprecatedRule extends Rule {
    documentation(context) {
        const preamble = context ? `The rule "${context}"` : "This rule";
        return {
            description: `${preamble} is deprecated and should not be used any longer, consult documentation for further information.`,
            url: ruleDocumentationUrl("@/rules/deprecated-rule.ts"),
        };
    }
    setup() {
        this.on("config:ready", (event) => {
            for (const rule of this.getDeprecatedRules(event)) {
                if (rule.getSeverity() > Severity.DISABLED) {
                    this.report(null, `Usage of deprecated rule "${rule.name}"`, null, rule.name);
                }
            }
        });
    }
    getDeprecatedRules(event) {
        const rules = Object.values(event.rules);
        return rules.filter((rule) => rule.deprecated);
    }
}

class NoStyleTag$1 extends Rule {
    documentation() {
        return {
            description: [
                'HTML5 documents should use the "html" doctype (short `form`, not legacy string):',
                "",
                "```html",
                "<!DOCTYPE html>",
                "```",
            ].join("\n"),
            url: ruleDocumentationUrl("@/rules/doctype-html.ts"),
        };
    }
    setup() {
        this.on("doctype", (event) => {
            const doctype = event.value.toLowerCase();
            if (doctype !== "html") {
                this.report(null, 'doctype should be "html"', event.valueLocation);
            }
        });
    }
}

const defaults$h = {
    style: "uppercase",
};
class DoctypeStyle extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$h), options));
    }
    static schema() {
        return {
            style: {
                enum: ["lowercase", "uppercase"],
                type: "string",
            },
        };
    }
    documentation(context) {
        const doc = {
            description: `While DOCTYPE is case-insensitive in the standard the current configuration requires a specific style.`,
            url: ruleDocumentationUrl("@/rules/doctype-style.ts"),
        };
        if (context) {
            doc.description = `While DOCTYPE is case-insensitive in the standard the current configuration requires it to be ${context.style}`;
        }
        return doc;
    }
    setup() {
        this.on("doctype", (event) => {
            if (this.options.style === "uppercase" && event.tag !== "DOCTYPE") {
                this.report(null, "DOCTYPE should be uppercase", event.location, this.options);
            }
            if (this.options.style === "lowercase" && event.tag !== "doctype") {
                this.report(null, "DOCTYPE should be lowercase", event.location, this.options);
            }
        });
    }
}

const defaults$g = {
    style: "lowercase",
};
class ElementCase extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$g), options));
        this.style = new CaseStyle(this.options.style, "element-case");
    }
    static schema() {
        const styleEnum = ["lowercase", "uppercase", "pascalcase", "camelcase"];
        return {
            style: {
                anyOf: [
                    {
                        enum: styleEnum,
                        type: "string",
                    },
                    {
                        items: {
                            enum: styleEnum,
                            type: "string",
                        },
                        type: "array",
                    },
                ],
            },
        };
    }
    documentation() {
        return {
            description: `Element tagname must be ${this.options.style}.`,
            url: ruleDocumentationUrl("@/rules/element-case.ts"),
        };
    }
    setup() {
        this.on("tag:start", (event) => {
            const { target, location } = event;
            this.validateCase(target, location);
        });
        this.on("tag:end", (event) => {
            const { target, previous } = event;
            this.validateMatchingCase(previous, target);
        });
    }
    validateCase(target, targetLocation) {
        const letters = target.tagName.replace(/[^a-z]+/gi, "");
        if (!this.style.match(letters)) {
            const location = sliceLocation(targetLocation, 1);
            this.report(target, `Element "${target.tagName}" should be ${this.style.name}`, location);
        }
    }
    validateMatchingCase(start, end) {
        /* handle when elements have have missing start or end tag */
        if (!start || !end || !start.tagName || !end.tagName) {
            return;
        }
        /* only check case if the names are a lowercase match to each other or it
         * will yield false positives when elements are closed in wrong order or
         * otherwise mismatched */
        if (start.tagName.toLowerCase() !== end.tagName.toLowerCase()) {
            return;
        }
        if (start.tagName !== end.tagName) {
            this.report(start, "Start and end tag must not differ in casing", end.location);
        }
    }
}

const defaults$f = {
    pattern: "^[a-z][a-z0-9\\-._]*-[a-z0-9\\-._]*$",
    whitelist: [],
    blacklist: [],
};
class ElementName extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$f), options));
        // eslint-disable-next-line security/detect-non-literal-regexp
        this.pattern = new RegExp(this.options.pattern);
    }
    static schema() {
        return {
            blacklist: {
                items: {
                    type: "string",
                },
                type: "array",
            },
            pattern: {
                type: "string",
            },
            whitelist: {
                items: {
                    type: "string",
                },
                type: "array",
            },
        };
    }
    documentation(context) {
        return {
            description: this.documentationMessages(context).join("\n"),
            url: ruleDocumentationUrl("@/rules/element-name.ts"),
        };
    }
    documentationMessages(context) {
        if (!context) {
            return ["This is not a valid element name."];
        }
        if (context.blacklist.includes(context.tagName)) {
            return [
                `<${context.tagName}> is blacklisted by the project configuration.`,
                "",
                "The following names are blacklisted:",
                ...context.blacklist.map((cur) => `- ${cur}`),
            ];
        }
        if (context.pattern !== defaults$f.pattern) {
            return [
                `<${context.tagName}> is not a valid element name. This project is configured to only allow names matching the following regular expression:`,
                "",
                `- \`${context.pattern}\``,
            ];
        }
        return [
            `<${context.tagName}> is not a valid element name. If this is a custom element HTML requires the name to follow these rules:`,
            "",
            "- The name must begin with `a-z`",
            "- The name must include a hyphen `-`",
            "- It may include alphanumerical characters `a-z0-9` or hyphens `-`, dots `.` or underscores `_`.",
        ];
    }
    setup() {
        const xmlns = /^(.+):.+$/;
        this.on("tag:start", (event) => {
            const target = event.target;
            const tagName = target.tagName;
            const location = sliceLocation(event.location, 1);
            const context = {
                tagName,
                pattern: this.options.pattern,
                blacklist: this.options.blacklist,
            };
            /* check if element is blacklisted */
            if (this.options.blacklist.includes(tagName)) {
                this.report(target, `<${tagName}> element is blacklisted`, location, context);
            }
            /* assume that an element with meta has valid name as it is a builtin
             * element */
            if (target.meta) {
                return;
            }
            /* ignore elements in xml namespaces, they should be validated against a
             * DTD instead */
            if (tagName.match(xmlns)) {
                return;
            }
            /* check if element is whitelisted */
            if (this.options.whitelist.includes(tagName)) {
                return;
            }
            if (!tagName.match(this.pattern)) {
                this.report(target, `<${tagName}> is not a valid element name`, location, context);
            }
        });
    }
}

function getTransparentChildren(node, transparent) {
    if (typeof transparent === "boolean") {
        return node.childElements;
    }
    else {
        /* only return children which matches one of the given content categories */
        return node.childElements.filter((it) => {
            return transparent.some((category) => {
                return Validator.validatePermittedCategory(it, category, false);
            });
        });
    }
}
class ElementPermittedContent extends Rule {
    documentation() {
        return {
            description: "Some elements has restrictions on what content is allowed. This can include both direct children or descendant elements.",
            url: ruleDocumentationUrl("@/rules/element-permitted-content.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            doc.visitDepthFirst((node) => {
                const parent = node.parent;
                /* dont verify root element, assume any element is allowed */
                if (!parent || parent.isRootElement()) {
                    return;
                }
                /* Run each validation step, stop as soon as any errors are
                 * reported. This is to prevent multiple similar errors on the same
                 * element, such as "<dd> is not permitted content under <span>" and
                 * "<dd> has no permitted ancestors". */
                [
                    () => this.validatePermittedContent(node, parent),
                    () => this.validatePermittedDescendant(node, parent),
                    () => this.validatePermittedAncestors(node),
                ].some((fn) => fn());
            });
        });
    }
    validatePermittedContent(cur, parent) {
        var _a;
        /* if parent doesn't have metadata (unknown element) skip checking permitted
         * content */
        if (!parent.meta) {
            return false;
        }
        const rules = (_a = parent.meta.permittedContent) !== null && _a !== void 0 ? _a : null;
        return this.validatePermittedContentImpl(cur, parent, rules);
    }
    validatePermittedContentImpl(cur, parent, rules) {
        if (!Validator.validatePermitted(cur, rules)) {
            this.report(cur, `Element <${cur.tagName}> is not permitted as content in ${parent.annotatedName}`);
            return true;
        }
        /* for transparent elements all/listed children must be validated against
         * the (this elements) parent, i.e. if this node was removed from the DOM it
         * should still be valid. */
        if (cur.meta && cur.meta.transparent) {
            const children = getTransparentChildren(cur, cur.meta.transparent);
            return children
                .map((child) => {
                return this.validatePermittedContentImpl(child, parent, rules);
            })
                .some(Boolean);
        }
        return false;
    }
    validatePermittedDescendant(node, parent) {
        var _a;
        for (let cur = parent; cur && !cur.isRootElement(); cur = /* istanbul ignore next */ (_a = cur === null || cur === void 0 ? void 0 : cur.parent) !== null && _a !== void 0 ? _a : null) {
            const meta = cur.meta;
            /* ignore checking parent without meta */
            if (!meta) {
                continue;
            }
            const rules = meta.permittedDescendants;
            if (!rules) {
                continue;
            }
            if (Validator.validatePermitted(node, rules)) {
                continue;
            }
            this.report(node, `Element <${node.tagName}> is not permitted as descendant of ${cur.annotatedName}`);
            return true;
        }
        return false;
    }
    validatePermittedAncestors(node) {
        if (!node.meta) {
            return false;
        }
        const rules = node.meta.requiredAncestors;
        if (!rules) {
            return false;
        }
        if (!Validator.validateAncestors(node, rules)) {
            this.report(node, `Element <${node.tagName}> requires an "${rules[0]}" ancestor`);
            return true;
        }
        return false;
    }
}

class ElementPermittedOccurrences extends Rule {
    documentation() {
        return {
            description: "Some elements may only be used a fixed amount of times in given context.",
            url: ruleDocumentationUrl("@/rules/element-permitted-occurrences.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            doc.visitDepthFirst((node) => {
                const parent = node.parent;
                if (!parent || !parent.meta) {
                    return;
                }
                const rules = parent.meta.permittedContent;
                if (!rules) {
                    return;
                }
                const siblings = parent.childElements.filter((cur) => cur.tagName === node.tagName);
                const first = node.unique === siblings[0].unique;
                /* the first occurrence should not trigger any errors, only the
                 * subsequent occurrences should. */
                if (first) {
                    return;
                }
                if (parent.meta && !Validator.validateOccurrences(node, rules, siblings.length)) {
                    this.report(node, `Element <${node.tagName}> can only appear once under ${parent.annotatedName}`);
                }
            });
        });
    }
}

class ElementPermittedOrder extends Rule {
    documentation() {
        return {
            description: "Some elements has a specific order the children must use.",
            url: ruleDocumentationUrl("@/rules/element-permitted-order.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            doc.visitDepthFirst((node) => {
                if (!node.meta) {
                    return;
                }
                const rules = node.meta.permittedOrder;
                if (!rules) {
                    return;
                }
                Validator.validateOrder(node.childElements, rules, (child, prev) => {
                    this.report(child, `Element <${child.tagName}> must be used before <${prev.tagName}> in this context`);
                });
            });
        });
    }
}

class ElementRequiredAttributes extends Rule {
    documentation(context) {
        const docs = {
            description: "Element is missing a required attribute",
            url: ruleDocumentationUrl("@/rules/element-required-attributes.ts"),
        };
        if (context) {
            docs.description = `The <${context.element}> element is required to have a "${context.attribute}" attribute.`;
        }
        return docs;
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.previous;
            const meta = node.meta;
            if (!meta || !meta.requiredAttributes)
                return;
            for (const key of meta.requiredAttributes) {
                if (node.hasAttribute(key))
                    continue;
                const context = {
                    element: node.tagName,
                    attribute: key,
                };
                this.report(node, `${node.annotatedName} is missing required "${key}" attribute`, node.location, context);
            }
        });
    }
}

class ElementRequiredContent extends Rule {
    documentation(context) {
        if (context) {
            return {
                description: `The <${context.node} element requires a <${context.missing}> to be present as content.`,
                url: ruleDocumentationUrl("@/rules/element-required-content.ts"),
            };
        }
        else {
            return {
                description: "Some elements has requirements on content that must be present.",
                url: ruleDocumentationUrl("@/rules/element-required-content.ts"),
            };
        }
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            doc.visitDepthFirst((node) => {
                /* if element doesn't have metadata (unknown element) skip checking
                 * required content */
                if (!node.meta) {
                    return;
                }
                const rules = node.meta.requiredContent;
                if (!rules) {
                    return;
                }
                for (const missing of Validator.validateRequiredContent(node, rules)) {
                    const context = {
                        node: node.tagName,
                        missing,
                    };
                    this.report(node, `${node.annotatedName} element must have <${missing}> as content`, null, context);
                }
            });
        });
    }
}

const CACHE_KEY = Symbol(classifyNodeText.name);
var TextClassification;
(function (TextClassification) {
    TextClassification[TextClassification["EMPTY_TEXT"] = 0] = "EMPTY_TEXT";
    TextClassification[TextClassification["DYNAMIC_TEXT"] = 1] = "DYNAMIC_TEXT";
    TextClassification[TextClassification["STATIC_TEXT"] = 2] = "STATIC_TEXT";
})(TextClassification || (TextClassification = {}));
/**
 * Checks text content of an element.
 *
 * Any text is considered including text from descendant elements. Whitespace is
 * ignored.
 *
 * If any text is dynamic `TextClassification.DYNAMIC_TEXT` is returned.
 */
function classifyNodeText(node) {
    if (node.cacheExists(CACHE_KEY)) {
        return node.cacheGet(CACHE_KEY);
    }
    const text = findTextNodes(node);
    /* if any text is dynamic classify as dynamic */
    if (text.some((cur) => cur.isDynamic)) {
        return node.cacheSet(CACHE_KEY, TextClassification.DYNAMIC_TEXT);
    }
    /* if any text has non-whitespace character classify as static */
    if (text.some((cur) => cur.textContent.match(/\S/) !== null)) {
        return node.cacheSet(CACHE_KEY, TextClassification.STATIC_TEXT);
    }
    /* default to empty */
    return node.cacheSet(CACHE_KEY, TextClassification.EMPTY_TEXT);
}
function findTextNodes(node) {
    let text = [];
    for (const child of node.childNodes) {
        switch (child.nodeType) {
            case NodeType.TEXT_NODE:
                text.push(child);
                break;
            case NodeType.ELEMENT_NODE:
                text = text.concat(findTextNodes(child));
                break;
        }
    }
    return text;
}

const selector = ["h1", "h2", "h3", "h4", "h5", "h6"].join(",");
class EmptyHeading extends Rule {
    documentation() {
        return {
            description: `Assistive technology such as screen readers require textual content in headings. Whitespace only is considered empty.`,
            url: ruleDocumentationUrl("@/rules/empty-heading.ts"),
        };
    }
    setup() {
        this.on("dom:ready", ({ document }) => {
            const headings = document.querySelectorAll(selector);
            for (const heading of headings) {
                switch (classifyNodeText(heading)) {
                    case TextClassification.DYNAMIC_TEXT:
                    case TextClassification.STATIC_TEXT:
                        /* have some text content, consider ok */
                        break;
                    case TextClassification.EMPTY_TEXT:
                        /* no content or whitespace only */
                        this.report(heading, `<${heading.tagName}> cannot be empty, must have text content`);
                        break;
                }
            }
        });
    }
}

class EmptyTitle extends Rule {
    documentation() {
        return {
            description: `The <title> element is used to describe the document and is shown in the browser tab and titlebar. WCAG and SEO requires a descriptive title and preferably unique within the site. Whitespace only is considered empty.`,
            url: ruleDocumentationUrl("@/rules/empty-title.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.previous;
            if (node.tagName !== "title")
                return;
            switch (classifyNodeText(node)) {
                case TextClassification.DYNAMIC_TEXT:
                case TextClassification.STATIC_TEXT:
                    /* have some text content, consider ok */
                    break;
                case TextClassification.EMPTY_TEXT:
                    /* no content or whitespace only */
                    this.report(node, `<${node.tagName}> cannot be empty, must have text content`);
                    break;
            }
        });
    }
}

const defaults$e = {
    allowMultipleH1: false,
    minInitialRank: "h1",
    sectioningRoots: ["dialog", '[role="dialog"]'],
};
function isRelevant$2(event) {
    const node = event.target;
    return Boolean(node.meta && node.meta.heading);
}
function extractLevel(node) {
    const match = node.tagName.match(/^[hH](\d)$/);
    if (match) {
        return parseInt(match[1], 10);
    }
    else {
        return null;
    }
}
function parseMaxInitial(value) {
    if (value === false || value === "any") {
        return 6;
    }
    const match = value.match(/^h(\d)$/);
    /* istanbul ignore next: should never happen, schema validation should catch invalid values */
    if (!match) {
        return 1;
    }
    return parseInt(match[1], 10);
}
class HeadingLevel extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$e), options));
        this.stack = [];
        this.minInitialRank = parseMaxInitial(this.options.minInitialRank);
        this.sectionRoots = this.options.sectioningRoots.map((it) => new Pattern(it));
        /* add a global sectioning root used by default */
        this.stack.push({
            node: null,
            current: 0,
            h1Count: 0,
        });
    }
    static schema() {
        return {
            allowMultipleH1: {
                type: "boolean",
            },
            minInitialRank: {
                enum: ["h1", "h2", "h3", "h4", "h5", "h6", "any", false],
            },
            sectioningRoots: {
                items: {
                    type: "string",
                },
                type: "array",
            },
        };
    }
    documentation() {
        const text = [];
        const modality = this.minInitialRank > 1 ? "should" : "must";
        text.push(`Headings ${modality} start at <h1> and can only increase one level at a time.`);
        text.push("The headings should form a table of contents and make sense on its own.");
        if (!this.options.allowMultipleH1) {
            text.push("");
            text.push("Under the current configuration only a single <h1> can be present at a time in the document.");
        }
        return {
            description: text.join("\n"),
            url: ruleDocumentationUrl("@/rules/heading-level.ts"),
        };
    }
    setup() {
        this.on("tag:start", isRelevant$2, (event) => this.onTagStart(event));
        this.on("tag:ready", (event) => this.onTagReady(event));
        this.on("tag:close", (event) => this.onTagClose(event));
    }
    onTagStart(event) {
        /* extract heading level from tagName (e.g "h1" -> 1)*/
        const level = extractLevel(event.target);
        if (!level)
            return;
        /* fetch the current sectioning root */
        const root = this.getCurrentRoot();
        /* do not allow multiple h1 */
        if (!this.options.allowMultipleH1 && level === 1) {
            if (root.h1Count >= 1) {
                const location = sliceLocation(event.location, 1);
                this.report(event.target, `Multiple <h1> are not allowed`, location);
                return;
            }
            root.h1Count++;
        }
        /* allow same level or decreasing to any level (e.g. from h4 to h2) */
        if (level <= root.current) {
            root.current = level;
            return;
        }
        this.checkLevelIncrementation(root, event, level);
        root.current = level;
    }
    /**
     * Validate heading level was only incremented by one.
     */
    checkLevelIncrementation(root, event, level) {
        const expected = root.current + 1;
        /* check if the new level is the expected one (headings with higher ranks
         * are skipped already) */
        if (level === expected) {
            return;
        }
        /* if this is the initial heading of the document it is compared to the
         * minimal allowed (default h1) */
        const isInitial = this.stack.length === 1 && expected === 1;
        if (isInitial && level <= this.minInitialRank) {
            return;
        }
        /* if we reach this far the heading level is not accepted */
        const location = sliceLocation(event.location, 1);
        if (root.current > 0) {
            const msg = `Heading level can only increase by one, expected <h${expected}> but got <h${level}>`;
            this.report(event.target, msg, location);
        }
        else {
            this.checkInitialLevel(event, location, level, expected);
        }
    }
    checkInitialLevel(event, location, level, expected) {
        if (this.stack.length === 1) {
            const msg = this.minInitialRank > 1
                ? `Initial heading level must be <h${this.minInitialRank}> or higher rank but got <h${level}>`
                : `Initial heading level must be <h${expected}> but got <h${level}>`;
            this.report(event.target, msg, location);
        }
        else {
            const prevRoot = this.getPrevRoot();
            const prevRootExpected = prevRoot.current + 1;
            if (level > prevRootExpected) {
                if (expected === prevRootExpected) {
                    const msg = `Initial heading level for sectioning root must be <h${expected}> but got <h${level}>`;
                    this.report(event.target, msg, location);
                }
                else {
                    const msg = `Initial heading level for sectioning root must be between <h${expected}> and <h${prevRootExpected}> but got <h${level}>`;
                    this.report(event.target, msg, location);
                }
            }
        }
    }
    /**
     * Check if the current element is a sectioning root and push a new root entry
     * on the stack if it is.
     */
    onTagReady(event) {
        const { target } = event;
        if (this.isSectioningRoot(target)) {
            this.stack.push({
                node: target.unique,
                current: 0,
                h1Count: 0,
            });
        }
    }
    /**
     * Check if the current element being closed is the element which opened the
     * current sectioning root, in which case the entry is popped from the stack.
     */
    onTagClose(event) {
        const { previous: target } = event;
        const root = this.getCurrentRoot();
        if (target.unique !== root.node) {
            return;
        }
        this.stack.pop();
    }
    getPrevRoot() {
        return this.stack[this.stack.length - 2];
    }
    getCurrentRoot() {
        return this.stack[this.stack.length - 1];
    }
    isSectioningRoot(node) {
        const context = {
            scope: node,
        };
        return this.sectionRoots.some((it) => it.match(node, context));
    }
}

const defaults$d = {
    pattern: "kebabcase",
};
class IdPattern extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$d), options));
        this.pattern = parsePattern(this.options.pattern);
    }
    static schema() {
        return {
            pattern: {
                type: "string",
            },
        };
    }
    documentation() {
        const pattern = describePattern(this.options.pattern);
        return {
            description: `For consistency all IDs are required to match the pattern ${pattern}.`,
            url: ruleDocumentationUrl("@/rules/id-pattern.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => {
            if (event.key.toLowerCase() !== "id") {
                return;
            }
            /* consider dynamic value as always matching the pattern */
            if (event.value instanceof DynamicValue) {
                return;
            }
            if (!event.value || !event.value.match(this.pattern)) {
                this.report(event.target, `ID "${event.value}" does not match required pattern "${this.pattern}"`, event.valueLocation);
            }
        });
    }
}

/* eslint-disable sonarjs/no-duplicate-string */
const restricted = new Map([
    ["accept", ["file"]],
    ["alt", ["image"]],
    [
        "autocomplete",
        [
            "hidden",
            "text",
            "search",
            "url",
            "tel",
            "email",
            "password",
            "date",
            "month",
            "week",
            "time",
            "datetime-local",
            "number",
            "range",
            "color",
        ],
    ],
    ["capture", ["file"]],
    ["checked", ["checkbox", "radio"]],
    ["dirname", ["text", "search"]],
    ["formaction", ["submit", "image"]],
    ["formenctype", ["submit", "image"]],
    ["formmethod", ["submit", "image"]],
    ["formnovalidate", ["submit", "image"]],
    ["formtarget", ["submit", "image"]],
    ["height", ["image"]],
    [
        "list",
        [
            "text",
            "search",
            "url",
            "tel",
            "email",
            "date",
            "month",
            "week",
            "time",
            "datetime-local",
            "number",
            "range",
            "color",
        ],
    ],
    ["max", ["date", "month", "week", "time", "datetime-local", "number", "range"]],
    ["maxlength", ["text", "search", "url", "tel", "email", "password"]],
    ["min", ["date", "month", "week", "time", "datetime-local", "number", "range"]],
    ["minlength", ["text", "search", "url", "tel", "email", "password"]],
    ["multiple", ["email", "file"]],
    ["pattern", ["text", "search", "url", "tel", "email", "password"]],
    ["placeholder", ["text", "search", "url", "tel", "email", "password", "number"]],
    [
        "readonly",
        [
            "text",
            "search",
            "url",
            "tel",
            "email",
            "password",
            "date",
            "month",
            "week",
            "time",
            "datetime-local",
            "number",
        ],
    ],
    [
        "required",
        [
            "text",
            "search",
            "url",
            "tel",
            "email",
            "password",
            "date",
            "month",
            "week",
            "time",
            "datetime-local",
            "number",
            "checkbox",
            "radio",
            "file",
        ],
    ],
    ["size", ["text", "search", "url", "tel", "email", "password"]],
    ["src", ["image"]],
    ["step", ["date", "month", "week", "time", "datetime-local", "number", "range"]],
    ["width", ["image"]],
]);
function isInput(event) {
    const { target } = event;
    return target.is("input");
}
class InputAttributes extends Rule {
    documentation(context) {
        var _a, _b;
        if (context) {
            const { attribute, type } = context;
            const summary = `Attribute \`${attribute}\` is not allowed on \`<input type="${type}">\`\n`;
            const details = `\`${attribute}\` can only be used when \`type\` is:`;
            const list = (_b = (_a = restricted.get(attribute)) === null || _a === void 0 ? void 0 : _a.map((it) => `- \`${it}\``)) !== null && _b !== void 0 ? _b : [];
            return {
                description: [summary, details, ...list].join("\n"),
                url: ruleDocumentationUrl("@/rules/input-attributes.ts"),
            };
        }
        else {
            return {
                description: `This attribute cannot be used with this input type.`,
                url: ruleDocumentationUrl("@/rules/input-attributes.ts"),
            };
        }
    }
    setup() {
        this.on("tag:ready", isInput, (event) => {
            const { target } = event;
            const type = target.getAttribute("type");
            if (!type || type.isDynamic || !type.value) {
                return;
            }
            const typeValue = type.value.toString();
            for (const attr of target.attributes) {
                const validTypes = restricted.get(attr.key);
                if (!validTypes) {
                    continue;
                }
                if (validTypes.includes(typeValue)) {
                    continue;
                }
                const context = {
                    attribute: attr.key,
                    type: typeValue,
                };
                const message = `Attribute "${attr.key}" is not allowed on <input type="${typeValue}">`;
                this.report(target, message, attr.keyLocation, context);
            }
        });
    }
}

const ARIA_HIDDEN_CACHE = Symbol(isAriaHidden.name);
const HTML_HIDDEN_CACHE = Symbol(isHTMLHidden.name);
const ROLE_PRESENTATION_CACHE = Symbol(isPresentation.name);
/**
 * Tests if this element is present in the accessibility tree.
 *
 * In practice it tests whenever the element or its parents has
 * `role="presentation"` or `aria-hidden="false"`. Dynamic values counts as
 * visible since the element might be in the visibility tree sometimes.
 */
function inAccessibilityTree(node) {
    return !isAriaHidden(node) && !isPresentation(node);
}
/**
 * Tests if this element or an ancestor have `aria-hidden="true"`.
 *
 * Dynamic values yields `false` since the element will conditionally be in the
 * accessibility tree and must fulfill it's conditions.
 */
function isAriaHidden(node) {
    if (node.cacheExists(ARIA_HIDDEN_CACHE)) {
        return Boolean(node.cacheGet(ARIA_HIDDEN_CACHE));
    }
    let cur = node;
    do {
        const ariaHidden = cur.getAttribute("aria-hidden");
        /* aria-hidden="true" */
        if (ariaHidden && ariaHidden.value === "true") {
            return cur.cacheSet(ARIA_HIDDEN_CACHE, true);
        }
        /* sanity check: break if no parent is present, normally not an issue as the
         * root element should be found first */
        if (!cur.parent) {
            break;
        }
        /* check parents */
        cur = cur.parent;
    } while (!cur.isRootElement());
    return node.cacheSet(ARIA_HIDDEN_CACHE, false);
}
/**
 * Tests if this element or an ancestor have `hidden` attribute.
 *
 * Dynamic values yields `false` since the element will conditionally be in the
 * DOM tree and must fulfill it's conditions.
 */
function isHTMLHidden(node) {
    if (node.cacheExists(HTML_HIDDEN_CACHE)) {
        return Boolean(node.cacheGet(HTML_HIDDEN_CACHE));
    }
    let cur = node;
    do {
        const hidden = cur.getAttribute("hidden");
        /* hidden present */
        if (hidden !== null && hidden.isStatic) {
            return cur.cacheSet(HTML_HIDDEN_CACHE, true);
        }
        /* sanity check: break if no parent is present, normally not an issue as the
         * root element should be found first */
        if (!cur.parent) {
            break;
        }
        /* check parents */
        cur = cur.parent;
    } while (!cur.isRootElement());
    return node.cacheSet(HTML_HIDDEN_CACHE, false);
}
/**
 * Tests if this element or a parent element has role="presentation".
 *
 * Dynamic values yields `false` just as if the attribute wasn't present.
 */
function isPresentation(node) {
    if (node.cacheExists(ROLE_PRESENTATION_CACHE)) {
        return Boolean(node.cacheGet(ROLE_PRESENTATION_CACHE));
    }
    let cur = node;
    do {
        const role = cur.getAttribute("role");
        /* role="presentation" */
        if (role && role.value === "presentation") {
            return cur.cacheSet(ROLE_PRESENTATION_CACHE, true);
        }
        /* sanity check: break if no parent is present, normally not an issue as the
         * root element should be found first */
        if (!cur.parent) {
            break;
        }
        /* check parents */
        cur = cur.parent;
    } while (!cur.isRootElement());
    return node.cacheSet(ROLE_PRESENTATION_CACHE, false);
}

class InputMissingLabel extends Rule {
    documentation() {
        return {
            description: "Labels are associated with the input element and is required for a17y.",
            url: ruleDocumentationUrl("@/rules/input-missing-label.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const root = event.document;
            for (const elem of root.querySelectorAll("input, textarea, select")) {
                this.validateInput(root, elem);
            }
        });
    }
    validateInput(root, elem) {
        if (isHTMLHidden(elem) || isAriaHidden(elem)) {
            return;
        }
        /* <input type="hidden"> should not have label */
        if (elem.is("input")) {
            const type = elem.getAttributeValue("type");
            if (type && type.toLowerCase() === "hidden") {
                return;
            }
        }
        let label = [];
        /* try to find label by id */
        if ((label = findLabelById(root, elem.id)).length > 0) {
            this.validateLabel(elem, label);
            return;
        }
        /* try to find parent label (input nested in label) */
        if ((label = findLabelByParent(elem)).length > 0) {
            this.validateLabel(elem, label);
            return;
        }
        this.report(elem, `<${elem.tagName}> element does not have a <label>`);
    }
    /**
     * Reports error if none of the labels are accessible.
     */
    validateLabel(elem, labels) {
        const visible = labels.filter(isVisible);
        if (visible.length === 0) {
            this.report(elem, `<${elem.tagName}> element has label but <label> element is hidden`);
        }
    }
}
function isVisible(elem) {
    const hidden = isHTMLHidden(elem) || isAriaHidden(elem);
    return !hidden;
}
function findLabelById(root, id) {
    if (!id)
        return [];
    return root.querySelectorAll(`label[for="${id}"]`);
}
function findLabelByParent(el) {
    let cur = el.parent;
    while (cur) {
        if (cur.is("label")) {
            return [cur];
        }
        cur = cur.parent;
    }
    return [];
}

const defaults$c = {
    maxlength: 70,
};
class LongTitle extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$c), options));
        this.maxlength = this.options.maxlength;
    }
    static schema() {
        return {
            maxlength: {
                type: "number",
            },
        };
    }
    documentation() {
        return {
            description: `Search engines truncates titles with long text, possibly down-ranking the page in the process.`,
            url: ruleDocumentationUrl("@/rules/long-title.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.previous;
            if (node.tagName !== "title")
                return;
            const text = node.textContent;
            if (text.length > this.maxlength) {
                this.report(node, `title text cannot be longer than ${this.maxlength} characters`);
            }
        });
    }
}

class MetaRefresh extends Rule {
    documentation() {
        return {
            description: `Meta refresh directive must use the \`0;url=...\` format. Non-zero values for time interval is disallowed as people with assistive technology might be unable to read and understand the page content before automatically reloading. For the same reason skipping the url is disallowed as it would put the browser in an infinite loop reloading the same page over and over again.`,
            url: ruleDocumentationUrl("@/rules/meta-refresh.ts"),
        };
    }
    setup() {
        this.on("element:ready", ({ target }) => {
            /* only handle <meta> */
            if (!target.is("meta")) {
                return;
            }
            /* only handle refresh */
            const httpEquiv = target.getAttributeValue("http-equiv");
            if (httpEquiv !== "refresh") {
                return;
            }
            /* ensure content attribute is set */
            const content = target.getAttribute("content");
            if (!content || !content.value || content.isDynamic) {
                return;
            }
            /* ensure content attribute is valid */
            const location = content.valueLocation;
            const value = parseContent(content.value.toString());
            if (!value) {
                this.report(target, "Malformed meta refresh directive", location);
                return;
            }
            /* ensure a url is set */
            if (!value.url) {
                this.report(target, "Don't use meta refresh to reload the page", location);
            }
            /* ensure delay is exactly 0 seconds */
            if (value.delay !== 0) {
                this.report(target, "Meta refresh must use 0 second delay", location);
            }
        });
    }
}
function parseContent(text) {
    const match = text.match(/^(\d+)(?:\s*;\s*url=(.*))?/);
    if (match) {
        return {
            delay: parseInt(match[1], 10),
            url: match[2],
        };
    }
    else {
        return null;
    }
}

class MissingDoctype extends Rule {
    documentation() {
        return {
            description: "Requires that the document contains a doctype.",
            url: ruleDocumentationUrl("@/rules/missing-doctype.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const dom = event.document;
            if (!dom.doctype) {
                this.report(dom.root, "Document is missing doctype");
            }
        });
    }
}

class MultipleLabeledControls extends Rule {
    constructor() {
        super(...arguments);
        this.labelable = "";
    }
    documentation() {
        return {
            description: `A \`<label>\` element can only be associated with one control at a time.`,
            url: ruleDocumentationUrl("@/rules/multiple-labeled-controls.ts"),
        };
    }
    setup() {
        this.labelable = this.getTagsWithProperty("labelable").join(",");
        this.on("element:ready", (event) => {
            const { target } = event;
            /* only handle <label> */
            if (target.tagName !== "label") {
                return;
            }
            /* no error if it references 0 or 1 controls */
            const numControls = this.getNumLabledControls(target);
            if (numControls <= 1) {
                return;
            }
            this.report(target, "<label> is associated with multiple controls", target.location);
        });
    }
    getNumLabledControls(src) {
        /* get all controls wrapped by label element */
        const controls = src.querySelectorAll(this.labelable).map((node) => node.id);
        /* only count wrapped controls if the "for" attribute is missing or static,
         * for dynamic "for" attributes it is better to run in document mode later */
        const attr = src.getAttribute("for");
        if (!attr || attr.isDynamic || !attr.value) {
            return controls.length;
        }
        /* if "for" attribute references a wrapped element it should not be counted
         * multiple times */
        const redundant = controls.includes(attr.value.toString());
        if (redundant) {
            return controls.length;
        }
        /* has "for" attribute pointing to element outside wrapped controls */
        return controls.length + 1;
    }
}

const defaults$b = {
    include: null,
    exclude: null,
};
class NoAutoplay extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$b), options));
    }
    documentation(context) {
        const tagName = context ? ` on <${context.tagName}>` : "";
        return {
            description: [
                `The autoplay attribute is not allowed${tagName}.`,
                "Autoplaying content can be disruptive for users and has accessibilty concerns.",
                "Prefer to let the user control playback.",
            ].join("\n"),
            url: ruleDocumentationUrl("@/rules/no-autoplay.ts"),
        };
    }
    static schema() {
        return {
            exclude: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
            include: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
        };
    }
    setup() {
        this.on("attr", (event) => {
            /* only handle autoplay attribute */
            if (event.key.toLowerCase() !== "autoplay") {
                return;
            }
            /* ignore dynamic values */
            if (event.value && event.value instanceof DynamicValue) {
                return;
            }
            /* ignore tagnames configured to be ignored */
            const tagName = event.target.tagName;
            if (this.isKeywordIgnored(tagName)) {
                return;
            }
            /* report error */
            const context = { tagName };
            const location = event.location;
            this.report(event.target, `The autoplay attribute is not allowed on <${tagName}>`, location, context);
        });
    }
}

class NoConditionalComment extends Rule {
    documentation() {
        return {
            description: "Microsoft Internet Explorer previously supported using special HTML comments (conditional comments) for targeting specific versions of IE but since IE 10 it is deprecated and not supported in standards mode.",
            url: ruleDocumentationUrl("@/rules/no-conditional-comment.ts"),
        };
    }
    setup() {
        this.on("conditional", (event) => {
            this.report(null, "Use of conditional comments are deprecated", event.location);
        });
    }
}

class NoDeprecatedAttr extends Rule {
    documentation() {
        return {
            description: "HTML5 deprecated many old attributes.",
            url: ruleDocumentationUrl("@/rules/no-deprecated-attr.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => {
            const node = event.target;
            const meta = node.meta;
            const attr = event.key.toLowerCase();
            /* cannot validate if meta isn't known */
            if (meta === null) {
                return;
            }
            const deprecated = meta.deprecatedAttributes || [];
            if (deprecated.includes(attr)) {
                this.report(node, `Attribute "${event.key}" is deprecated on <${node.tagName}> element`, event.keyLocation);
            }
        });
    }
}

class NoDupAttr extends Rule {
    documentation() {
        return {
            description: "HTML disallows two or more attributes with the same (case-insensitive) name.",
            url: ruleDocumentationUrl("@/rules/no-dup-attr.ts"),
        };
    }
    setup() {
        let attr = {};
        this.on("tag:start", () => {
            /* reset any time a new tag is opened */
            attr = {};
        });
        this.on("attr", (event) => {
            /* ignore dynamic attributes aliasing another, e.g class and ng-class */
            if (event.originalAttribute) {
                return;
            }
            const name = event.key.toLowerCase();
            if (name in attr) {
                this.report(event.target, `Attribute "${name}" duplicated`, event.keyLocation);
            }
            attr[event.key] = true;
        });
    }
}

class NoDupClass extends Rule {
    documentation() {
        return {
            description: "Prevents unnecessary duplication of class names.",
            url: ruleDocumentationUrl("@/rules/no-dup-class.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => {
            if (event.key.toLowerCase() !== "class") {
                return;
            }
            const classes = new DOMTokenList(event.value, event.valueLocation);
            const unique = new Set();
            classes.forEach((cur, index) => {
                if (unique.has(cur)) {
                    const location = classes.location(index);
                    this.report(event.target, `Class "${cur}" duplicated`, location);
                }
                unique.add(cur);
            });
        });
    }
}

class NoDupID extends Rule {
    documentation() {
        return {
            description: "The ID of an element must be unique.",
            url: ruleDocumentationUrl("@/rules/no-dup-id.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const { document } = event;
            const existing = new Set();
            const elements = document.querySelectorAll("[id]");
            const relevant = elements.filter(isRelevant$1);
            for (const el of relevant) {
                const attr = el.getAttribute("id");
                /* istanbul ignore next: this has already been tested in isRelevant once but for type-safety it is checked again */
                if (!attr || !attr.value) {
                    continue;
                }
                const id = attr.value.toString();
                if (existing.has(id)) {
                    this.report(el, `Duplicate ID "${id}"`, attr.valueLocation);
                }
                existing.add(id);
            }
        });
    }
}
function isRelevant$1(element) {
    const attr = element.getAttribute("id");
    /* istanbul ignore next: can not really happen as querySelector will only return elements with id present */
    if (!attr) {
        return false;
    }
    /* id without value is not relevant, e.g. <p id></p> */
    if (!attr.value) {
        return false;
    }
    /* dynamic id (interpolated or otherwise currently unknown value) is not relevant */
    if (attr.isDynamic) {
        return false;
    }
    return true;
}

class NoImplicitClose extends Rule {
    documentation() {
        return {
            description: `Some elements in HTML has optional end tags. When an optional tag is omitted a browser must handle it as if the end tag was present.

Omitted end tags can be ambigious for humans to read and many editors have trouble formatting the markup.`,
            url: ruleDocumentationUrl("@/rules/no-implicit-close.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const closed = event.previous;
            const by = event.target;
            /* not set when unclosed elements are being closed by tree, this rule does
             * not consider such events (handled by close-order instead) */
            if (!by) {
                return;
            }
            if (closed.closed !== NodeClosed.ImplicitClosed) {
                return;
            }
            const closedByParent = closed.parent && closed.parent.tagName === by.tagName; /* <ul><li></ul> */
            const sameTag = closed.tagName === by.tagName; /* <p>foo<p>bar */
            if (closedByParent) {
                this.report(closed, `Element <${closed.tagName}> is implicitly closed by parent </${by.tagName}>`, closed.location);
            }
            else if (sameTag) {
                this.report(closed, `Element <${closed.tagName}> is implicitly closed by sibling`, closed.location);
            }
            else {
                this.report(closed, `Element <${closed.tagName}> is implicitly closed by adjacent <${by.tagName}>`, closed.location);
            }
        });
    }
}

const defaults$a = {
    include: null,
    exclude: null,
    allowedProperties: ["display"],
};
function getCSSDeclarations(value) {
    return value
        .trim()
        .split(";")
        .filter(Boolean)
        .map((it) => {
        const [property, value] = it.split(":", 2);
        return { property: property.trim(), value: value.trim() };
    });
}
class NoInlineStyle extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$a), options));
    }
    static schema() {
        return {
            exclude: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
            include: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
            allowedProperties: {
                items: {
                    type: "string",
                },
                type: "array",
            },
        };
    }
    documentation() {
        const text = [
            "Inline style is not allowed.\n",
            "Inline style is a sign of unstructured CSS. Use class or ID with a separate stylesheet.\n",
        ];
        if (this.options.allowedProperties.length > 0) {
            text.push("Under the current configuration the following CSS properties are allowed:\n");
            text.push(this.options.allowedProperties.map((it) => `- \`${it}\``).join("\n"));
        }
        return {
            description: text.join("\n"),
            url: ruleDocumentationUrl("@/rules/no-inline-style.ts"),
        };
    }
    setup() {
        this.on("attr", (event) => this.isRelevant(event), (event) => {
            const { value } = event;
            if (this.allPropertiesAllowed(value)) {
                return;
            }
            this.report(event.target, "Inline style is not allowed");
        });
    }
    isRelevant(event) {
        if (event.key !== "style") {
            return false;
        }
        const { include, exclude } = this.options;
        const key = event.originalAttribute || event.key;
        /* ignore attributes not present in "include" */
        if (include && !include.includes(key)) {
            return false;
        }
        /* ignore attributes present in "exclude" */
        if (exclude && exclude.includes(key)) {
            return false;
        }
        return true;
    }
    allPropertiesAllowed(value) {
        if (typeof value !== "string") {
            return false;
        }
        const allowProperties = this.options.allowedProperties;
        /* quick path: no properties are allowed, no need to check each one individually */
        if (allowProperties.length === 0) {
            return false;
        }
        const declarations = getCSSDeclarations(value);
        return (declarations.length > 0 &&
            declarations.every((it) => {
                return allowProperties.includes(it.property);
            }));
    }
}

const ARIA = [
    { property: "aria-activedescendant", isList: false },
    { property: "aria-controls", isList: true },
    { property: "aria-describedby", isList: true },
    { property: "aria-details", isList: false },
    { property: "aria-errormessage", isList: false },
    { property: "aria-flowto", isList: true },
    { property: "aria-labelledby", isList: true },
    { property: "aria-owns", isList: true },
];
function idMissing(document, id) {
    const nodes = document.querySelectorAll(`[id="${id}"]`);
    return nodes.length === 0;
}
class NoMissingReferences extends Rule {
    documentation(context) {
        if (context) {
            return {
                description: `The element ID "${context.value}" referenced by the ${context.key} attribute must point to an existing element.`,
                url: ruleDocumentationUrl("@/rules/no-missing-references.ts"),
            };
        }
        else {
            return {
                description: `The element ID referenced by the attribute must point to an existing element.`,
                url: ruleDocumentationUrl("@/rules/no-missing-references.ts"),
            };
        }
    }
    setup() {
        this.on("dom:ready", (event) => {
            const document = event.document;
            /* verify <label for=".."> */
            for (const node of document.querySelectorAll("label[for]")) {
                const attr = node.getAttribute("for");
                this.validateReference(document, node, attr, false);
            }
            /* verify <input list=".."> */
            for (const node of document.querySelectorAll("input[list]")) {
                const attr = node.getAttribute("list");
                this.validateReference(document, node, attr, false);
            }
            /* verify WAI-ARIA properties */
            for (const { property, isList } of ARIA) {
                for (const node of document.querySelectorAll(`[${property}]`)) {
                    const attr = node.getAttribute(property);
                    this.validateReference(document, node, attr, isList);
                }
            }
        });
    }
    validateReference(document, node, attr, isList) {
        /* sanity check: querySelector should never return elements without the attribute */
        /* istanbul ignore next */
        if (!attr) {
            return;
        }
        /* skip dynamic and empty values */
        const value = attr.value;
        if (value instanceof DynamicValue || value === null || value === "") {
            return;
        }
        if (isList) {
            this.validateList(document, node, attr, value);
        }
        else {
            this.validateSingle(document, node, attr, value);
        }
    }
    validateSingle(document, node, attr, id) {
        if (idMissing(document, id)) {
            const context = { key: attr.key, value: id };
            this.report(node, `Element references missing id "${id}"`, attr.valueLocation, context);
        }
    }
    validateList(document, node, attr, values) {
        const parsed = new DOMTokenList(values, attr.valueLocation);
        for (const entry of parsed.iterator()) {
            const id = entry.item;
            if (idMissing(document, id)) {
                const context = { key: attr.key, value: id };
                this.report(node, `Element references missing id "${id}"`, entry.location, context);
            }
        }
    }
}

class NoMultipleMain extends Rule {
    documentation() {
        return {
            description: [
                "Only a single visible `<main>` element can be present at in a document at a time.",
                "",
                "Multiple `<main>` can be present in the DOM as long the others are hidden using the HTML5 `hidden` attribute.",
            ].join("\n"),
            url: ruleDocumentationUrl("@/rules/no-multiple-main.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const { document } = event;
            const main = document.querySelectorAll("main").filter((cur) => !cur.hasAttribute("hidden"));
            main.shift(); /* ignore the first occurrence */
            /* report all other occurrences */
            for (const elem of main) {
                this.report(elem, "Multiple <main> elements present in document");
            }
        });
    }
}

const defaults$9 = {
    relaxed: false,
};
const textRegexp = /([<>]|&(?![a-zA-Z0-9#]+;))/g;
const unquotedAttrRegexp = /([<>"'=`]|&(?![a-zA-Z0-9#]+;))/g;
const matchTemplate = /^(<%.*?%>|<\?.*?\?>|<\$.*?\$>)$/;
const replacementTable = new Map([
    ['"', "&quot;"],
    ["&", "&amp;"],
    ["'", "&apos;"],
    ["<", "&lt;"],
    ["=", "&equals;"],
    [">", "&gt;"],
    ["`", "&grave;"],
]);
class NoRawCharacters extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$9), options));
        this.relaxed = this.options.relaxed;
    }
    static schema() {
        return {
            relaxed: {
                type: "boolean",
            },
        };
    }
    documentation() {
        return {
            description: `Some characters such as \`<\`, \`>\` and \`&\` hold special meaning in HTML and must be escaped using a character reference (html entity).`,
            url: ruleDocumentationUrl("@/rules/no-raw-characters.ts"),
        };
    }
    setup() {
        this.on("element:ready", (event) => {
            const node = event.target;
            /* only iterate over direct descendants */
            for (const child of node.childNodes) {
                if (child.nodeType !== NodeType.TEXT_NODE) {
                    continue;
                }
                /* workaround for templating <% ... %> etc */
                if (child.textContent.match(matchTemplate)) {
                    continue;
                }
                this.findRawChars(node, child.textContent, child.location, textRegexp);
            }
        });
        this.on("attr", (event) => {
            /* boolean attributes has no value so nothing to validate */
            if (!event.value) {
                return;
            }
            /* quoted attribute values can contain most symbols except the quotemark
             * itself but unescaped quotemarks would cause a parsing error */
            if (event.quote) {
                return;
            }
            this.findRawChars(event.target, event.value.toString(), event.valueLocation, unquotedAttrRegexp);
        });
    }
    /**
     * Find raw special characters and report as errors.
     *
     * @param text - The full text to find unescaped raw characters in.
     * @param location - Location of text.
     * @param regexp - Regexp pattern to match using.
     * @param ignore - List of characters to ignore for this text.
     */
    findRawChars(node, text, location, regexp) {
        let match;
        do {
            match = regexp.exec(text);
            if (match) {
                const char = match[0];
                /* In relaxed mode & only needs to be encoded if it is ambiguous,
                 * however this rule will only match either non-ambiguous ampersands or
                 * ampersands part of a character reference. Whenever it is a valid
                 * character reference or not not checked by this rule */
                if (this.relaxed && char === "&") {
                    continue;
                }
                /* determine replacement character and location */
                const replacement = replacementTable.get(char);
                const charLocation = sliceLocation(location, match.index, match.index + 1);
                /* report as error */
                this.report(node, `Raw "${char}" must be encoded as "${replacement}"`, charLocation);
            }
        } while (match);
    }
}

class NoRedundantFor extends Rule {
    documentation() {
        return {
            description: `When the \`<label>\` element wraps the labelable control the \`for\` attribute is redundant and better left out.`,
            url: ruleDocumentationUrl("@/rules/no-redundant-for.ts"),
        };
    }
    setup() {
        this.on("element:ready", (event) => {
            const { target } = event;
            /* only handle <label> */
            if (target.tagName !== "label") {
                return;
            }
            /* ignore label without for or dynamic value */
            const attr = target.getAttribute("for");
            if (!attr || attr.isDynamic) {
                return;
            }
            /* ignore omitted/empty values */
            const id = attr.value;
            if (!id) {
                return;
            }
            /* try to find labeled control */
            const escaped = escapeSelectorComponent(id);
            const control = target.querySelector(`[id="${escaped}"]`);
            if (!control) {
                return;
            }
            this.report(target, 'Redundant "for" attribute', attr.keyLocation);
        });
    }
}

const mapping = {
    article: ["article"],
    header: ["banner"],
    button: ["button"],
    td: ["cell"],
    input: ["checkbox", "radio", "input"],
    aside: ["complementary"],
    footer: ["contentinfo"],
    figure: ["figure"],
    form: ["form"],
    h1: ["heading"],
    h2: ["heading"],
    h3: ["heading"],
    h4: ["heading"],
    h5: ["heading"],
    h6: ["heading"],
    a: ["link"],
    ul: ["list"],
    select: ["listbox"],
    li: ["listitem"],
    main: ["main"],
    nav: ["navigation"],
    progress: ["progressbar"],
    section: ["region"],
    table: ["table"],
    textarea: ["textbox"],
};
class NoRedundantRole extends Rule {
    documentation(context) {
        const doc = {
            description: `Using this role is redundant as it is already implied by the element.`,
            url: ruleDocumentationUrl("@/rules/no-redundant-role.ts"),
        };
        if (context) {
            doc.description = `Using the "${context.role}" role is redundant as it is already implied by the <${context.tagname}> element.`;
        }
        return doc;
    }
    setup() {
        this.on("attr", (event) => {
            const { target } = event;
            /* ignore non-role attributes */
            if (event.key.toLowerCase() !== "role") {
                return;
            }
            /* ignore missing and dynamic values */
            if (!event.value || event.value instanceof DynamicValue) {
                return;
            }
            /* ignore elements without known redundant roles */
            const redundant = mapping[target.tagName];
            if (!redundant) {
                return;
            }
            /* ignore elements with non-redundant roles */
            if (!redundant.includes(event.value)) {
                return;
            }
            /* report error */
            const context = {
                tagname: target.tagName,
                role: event.value,
            };
            this.report(event.target, `Redundant role "${event.value}" on <${target.tagName}>`, event.valueLocation, context);
        });
    }
}

const xmlns = /^(.+):.+$/;
const defaults$8 = {
    ignoreForeign: true,
    ignoreXML: true,
};
class NoSelfClosing extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$8), options));
    }
    static schema() {
        return {
            ignoreForeign: {
                type: "boolean",
            },
            ignoreXML: {
                type: "boolean",
            },
        };
    }
    documentation(tagName) {
        tagName = tagName || "element";
        return {
            description: `Self-closing elements are disallowed. Use regular end tag <${tagName}></${tagName}> instead of self-closing <${tagName}/>.`,
            url: ruleDocumentationUrl("@/rules/no-self-closing.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const active = event.previous; // The current active element (that is, the current element on the stack)
            if (!isRelevant(active, this.options)) {
                return;
            }
            this.validateElement(active);
        });
    }
    validateElement(node) {
        if (node.closed !== NodeClosed.VoidSelfClosed) {
            return;
        }
        this.report(node, `<${node.tagName}> must not be self-closed`, null, node.tagName);
    }
}
function isRelevant(node, options) {
    /* tags in XML namespaces are relevant only if ignoreXml is false, in which
     * case assume all xml elements must not be self-closed */
    if (node.tagName && node.tagName.match(xmlns)) {
        return !options.ignoreXML;
    }
    /* nodes with missing metadata is assumed relevant */
    if (!node.meta) {
        return true;
    }
    if (node.meta.void) {
        return false;
    }
    /* foreign elements are relevant only if ignoreForeign is false, in which case
     * assume all foreign must not be self-closed */
    if (node.meta.foreign) {
        return !options.ignoreForeign;
    }
    return true;
}

class NoStyleTag extends Rule {
    documentation() {
        return {
            description: "Prefer to use external stylesheets with the `<link>` tag instead of inlining the styling.",
            url: ruleDocumentationUrl("@/rules/no-style-tag.ts"),
        };
    }
    setup() {
        this.on("tag:start", (event) => {
            const node = event.target;
            if (node.tagName === "style") {
                this.report(node, "Use external stylesheet with <link> instead of <style> tag");
            }
        });
    }
}

class NoTrailingWhitespace extends Rule {
    documentation() {
        return {
            description: "Lines with trailing whitespace cause unnessecary diff when using version control and usually serve no special purpose in HTML.",
            url: ruleDocumentationUrl("@/rules/no-trailing-whitespace.ts"),
        };
    }
    setup() {
        this.on("whitespace", (event) => {
            if (event.text.match(/^[ \t]+\r?\n$/)) {
                this.report(null, "Trailing whitespace", event.location);
            }
        });
    }
}

class NoUnknownElements extends Rule {
    documentation(context) {
        const element = context ? ` <${context}>` : "";
        return {
            description: `An unknown element${element} was used. If this is a Custom Element you need to supply element metadata for it.`,
            url: ruleDocumentationUrl("@/rules/no-unknown-elements.ts"),
        };
    }
    setup() {
        this.on("tag:start", (event) => {
            const node = event.target;
            if (!node.meta) {
                this.report(node, `Unknown element <${node.tagName}>`, null, node.tagName);
            }
        });
    }
}

class NoUtf8Bom extends Rule {
    documentation() {
        return {
            description: `This file is saved with the UTF-8 byte order mark (BOM) present. It is neither required or recommended to use.\n\nInstead the document should be served with the \`Content-Type: application/javascript; charset=utf-8\` header.`,
            url: ruleDocumentationUrl("@/rules/no-utf8-bom.ts"),
        };
    }
    setup() {
        const unregister = this.on("token", (event) => {
            if (event.type === TokenType.UNICODE_BOM) {
                this.report(null, "File should be saved without UTF-8 BOM", event.location);
            }
            /* since the BOM must be the very first thing the rule can now be disabled for the rest of the run */
            this.setEnabled(false);
            unregister();
        });
    }
}

const types = ["button", "submit", "reset", "image"];
const replacement = {
    button: '<button type="button">',
    submit: '<button type="submit">',
    reset: '<button type="reset">',
    image: '<button type="button">',
};
const defaults$7 = {
    include: null,
    exclude: null,
};
class PreferButton extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$7), options));
    }
    static schema() {
        return {
            exclude: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
            include: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
        };
    }
    documentation(context) {
        const doc = {
            description: `Prefer to use the generic \`<button>\` element instead of \`<input>\`.`,
            url: ruleDocumentationUrl("@/rules/prefer-button.ts"),
        };
        if (context) {
            const src = `<input type="${context.type}">`;
            const dst = replacement[context.type] || `<button>`;
            doc.description = `Prefer to use \`${dst}\` instead of \`"${src}\`.`;
        }
        return doc;
    }
    setup() {
        this.on("attr", (event) => {
            const node = event.target;
            /* only handle input elements */
            if (node.tagName.toLowerCase() !== "input") {
                return;
            }
            /* only handle type attribute */
            if (event.key.toLowerCase() !== "type") {
                return;
            }
            /* sanity check: handle missing, boolean and dynamic attributes */
            if (!event.value || event.value instanceof DynamicValue) {
                return;
            }
            /* ignore types configured to be ignored */
            const type = event.value.toLowerCase();
            if (this.isKeywordIgnored(type)) {
                return;
            }
            /* only values matching known type triggers error */
            if (!types.includes(type)) {
                return;
            }
            const context = { type: type };
            const message = `Prefer to use <button> instead of <input type="${type}"> when adding buttons`;
            this.report(node, message, event.valueLocation, context);
        });
    }
}

const defaults$6 = {
    mapping: {
        article: "article",
        banner: "header",
        button: "button",
        cell: "td",
        checkbox: "input",
        complementary: "aside",
        contentinfo: "footer",
        figure: "figure",
        form: "form",
        heading: "hN",
        input: "input",
        link: "a",
        list: "ul",
        listbox: "select",
        listitem: "li",
        main: "main",
        navigation: "nav",
        progressbar: "progress",
        radio: "input",
        region: "section",
        table: "table",
        textbox: "textarea",
    },
    include: null,
    exclude: null,
};
class PreferNativeElement extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$6), options));
    }
    static schema() {
        return {
            exclude: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
            include: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "null",
                    },
                ],
            },
            mapping: {
                type: "object",
            },
        };
    }
    documentation(context) {
        const doc = {
            description: `Instead of using WAI-ARIA roles prefer to use the native HTML elements.`,
            url: ruleDocumentationUrl("@/rules/prefer-native-element.ts"),
        };
        if (context) {
            doc.description = `Instead of using the WAI-ARIA role "${context.role}" prefer to use the native <${context.replacement}> element.`;
        }
        return doc;
    }
    setup() {
        const { mapping } = this.options;
        this.on("attr", (event) => {
            /* ignore non-role attributes */
            if (event.key.toLowerCase() !== "role") {
                return;
            }
            /* ignore missing and dynamic values */
            if (!event.value || event.value instanceof DynamicValue) {
                return;
            }
            /* ignore roles configured to be ignored */
            const role = event.value.toLowerCase();
            if (this.isIgnored(role)) {
                return;
            }
            /* dont report when the element is already of the right type but has a
             * redundant role, such as <main role="main"> */
            const replacement = mapping[role];
            if (event.target.is(replacement)) {
                return;
            }
            /* report error */
            const context = { role, replacement };
            const location = this.getLocation(event);
            this.report(event.target, `Prefer to use the native <${replacement}> element`, location, context);
        });
    }
    isIgnored(role) {
        const { mapping } = this.options;
        /* ignore roles not mapped to native elements */
        const replacement = mapping[role];
        if (!replacement) {
            return true;
        }
        return this.isKeywordIgnored(role);
    }
    getLocation(event) {
        const begin = event.location;
        const end = event.valueLocation;
        const quote = event.quote ? 1 : 0;
        const size = end.offset + end.size - begin.offset + quote;
        return {
            filename: begin.filename,
            line: begin.line,
            column: begin.column,
            offset: begin.offset,
            size,
        };
    }
}

class PreferTbody extends Rule {
    documentation() {
        return {
            description: `While \`<tbody>\` is optional is relays semantic information about its contents. Where applicable it should also be combined with \`<thead>\` and \`<tfoot>\`.`,
            url: ruleDocumentationUrl("@/rules/prefer-tbody.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const doc = event.document;
            for (const table of doc.querySelectorAll("table")) {
                if (table.querySelector("> tbody")) {
                    continue;
                }
                const tr = table.querySelectorAll("> tr");
                if (tr.length >= 1) {
                    this.report(tr[0], "Prefer to wrap <tr> elements in <tbody>");
                }
            }
        });
    }
}

const defaults$5 = {
    target: "all",
};
const crossorigin = new RegExp("^(\\w+://|//)"); /* e.g. https:// or // */
const supportSri = {
    link: "href",
    script: "src",
};
class RequireSri extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$5), options));
        this.target = this.options.target;
    }
    static schema() {
        return {
            target: {
                enum: ["all", "crossorigin"],
                type: "string",
            },
        };
    }
    documentation() {
        return {
            description: `Subresource Integrity (SRI) \`integrity\` attribute is required to prevent manipulation from Content Delivery Networks or other third-party hosting.`,
            url: ruleDocumentationUrl("@/rules/require-sri.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            /* only handle thats supporting and requires sri */
            const node = event.previous;
            if (!(this.supportSri(node) && this.needSri(node)))
                return;
            /* check if sri attribute is present */
            if (node.hasAttribute("integrity"))
                return;
            this.report(node, `SRI "integrity" attribute is required on <${node.tagName}> element`, node.location);
        });
    }
    supportSri(node) {
        return Object.keys(supportSri).includes(node.tagName);
    }
    needSri(node) {
        if (this.target === "all")
            return true;
        const attr = this.elementSourceAttr(node);
        if (!attr || attr.value === null || attr.isDynamic) {
            return false;
        }
        const url = attr.value.toString();
        return crossorigin.test(url);
    }
    elementSourceAttr(node) {
        const key = supportSri[node.tagName];
        return node.getAttribute(key);
    }
}

class ScriptElement extends Rule {
    documentation() {
        return {
            description: "The end tag for `<script>` is a hard requirement and must never be omitted even when using the `src` attribute.",
            url: ruleDocumentationUrl("@/rules/script-element.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.target; // The current element being closed.
            if (!node || node.tagName !== "script") {
                return;
            }
            if (node.closed !== NodeClosed.EndTag) {
                this.report(node, `End tag for <${node.tagName}> must not be omitted`);
            }
        });
    }
}

const javascript = [
    "",
    "application/ecmascript",
    "application/javascript",
    "text/ecmascript",
    "text/javascript",
];
class ScriptType extends Rule {
    documentation() {
        return {
            description: "While valid the HTML5 standard encourages authors to omit the type element for JavaScript resources.",
            url: ruleDocumentationUrl("@/rules/script-type.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.previous;
            if (!node || node.tagName !== "script") {
                return;
            }
            const attr = node.getAttribute("type");
            if (!attr || attr.isDynamic) {
                return;
            }
            const value = attr.value ? attr.value.toString() : "";
            if (!this.isJavascript(value)) {
                return;
            }
            this.report(node, '"type" attribute is unnecessary for javascript resources', attr.keyLocation);
        });
    }
    isJavascript(mime) {
        /* remove mime parameters, e.g. ";charset=utf-8" */
        const type = mime.replace(/;.*/, "");
        return javascript.includes(type);
    }
}

class SvgFocusable extends Rule {
    documentation() {
        return {
            description: `Inline SVG elements in IE are focusable by default which may cause issues with tab-ordering. The \`focusable\` attribute should explicitly be set to avoid unintended behaviour.`,
            url: ruleDocumentationUrl("@/rules/svg-focusable.ts"),
        };
    }
    setup() {
        this.on("element:ready", (event) => {
            if (event.target.is("svg")) {
                this.validate(event.target);
            }
        });
    }
    validate(svg) {
        if (svg.hasAttribute("focusable")) {
            return;
        }
        this.report(svg, `<${svg.tagName}> is missing required "focusable" attribute`);
    }
}

function hasAltText(image) {
    const alt = image.getAttribute("alt");
    /* missing or boolean */
    if (alt === null || alt.value === null) {
        return false;
    }
    return alt.isDynamic || alt.value.toString() !== "";
}

function hasAriaLabel(node) {
    const label = node.getAttribute("aria-label");
    /* missing or boolean */
    if (label === null || label.value === null) {
        return false;
    }
    return label.isDynamic || label.value.toString() !== "";
}

/**
 * Check if attribute is present and non-empty or dynamic.
 */
function hasNonEmptyAttribute(node, key) {
    const attr = node.getAttribute(key);
    return Boolean(attr && attr.valueMatches(/.+/, true));
}
/**
 * Check if element has default text.
 *
 * Only <input type="submit"> and <input type="reset"> at the moment.
 */
function hasDefaultText(node) {
    /* only input element have default text */
    if (!node.is("input")) {
        return false;
    }
    /* default text is not available if value attribute is present */
    if (node.hasAttribute("value")) {
        return false;
    }
    /* default text is only present when type is submit or reset */
    const type = node.getAttribute("type");
    return Boolean(type && type.valueMatches(/submit|reset/, false));
}
function isTextNode(node) {
    return node.nodeType === NodeType.TEXT_NODE;
}
function isNonEmptyText(node) {
    if (isTextNode(node)) {
        return node.isDynamic || node.textContent.trim() !== "";
    }
    else {
        return false;
    }
}
/**
 * Walk nodes (depth-first, preorder) searching for accessible text. Children
 * hidden from accessibility tree are ignored.
 *
 * For each node the current conditions satisfies as accessible text:
 *
 * - Non-empty or dynamic `aria-label`
 * - Non-empty or dynamic `aria-labelledby` (reference not validated, use [[no-missing-references]]
 * - Image with non-empty or dynamic `alt` text
 * - Elements with default text
 */
function haveAccessibleText(node) {
    if (!inAccessibilityTree(node)) {
        return false;
    }
    /* check direct descendants for non-empty or dynamic text */
    const haveText = node.childNodes.some((child) => isNonEmptyText(child));
    if (haveText) {
        return true;
    }
    if (hasNonEmptyAttribute(node, "aria-label")) {
        return true;
    }
    if (hasNonEmptyAttribute(node, "aria-labelledby")) {
        return true;
    }
    if (node.is("img") && hasNonEmptyAttribute(node, "alt")) {
        return true;
    }
    if (hasDefaultText(node)) {
        return true;
    }
    return node.childElements.some((child) => {
        return haveAccessibleText(child);
    });
}
class TextContent extends Rule {
    documentation(context) {
        const doc = {
            description: `The textual content for this element is not valid.`,
            url: ruleDocumentationUrl("@/rules/text-content.ts"),
        };
        if (context === null || context === void 0 ? void 0 : context.textContent) {
            switch (context.textContent) {
                case TextContent$1.NONE:
                    doc.description = `The \`<${context.tagName}>\` element must not have textual content.`;
                    break;
                case TextContent$1.REQUIRED:
                    doc.description = `The \`<${context.tagName}>\` element must have textual content.`;
                    break;
                case TextContent$1.ACCESSIBLE:
                    doc.description = `The \`<${context.tagName}>\` element must have accessible text.`;
                    break;
            }
        }
        return doc;
    }
    static filter(event) {
        const { target } = event;
        /* skip elements without metadata */
        if (!target.meta) {
            return false;
        }
        /* skip elements without explicit and default textContent */
        const { textContent } = target.meta;
        if (!textContent || textContent === TextContent$1.DEFAULT) {
            return false;
        }
        return true;
    }
    setup() {
        this.on("element:ready", TextContent.filter, (event) => {
            const target = event.target;
            const { textContent } = target.meta;
            switch (textContent) {
                case TextContent$1.NONE:
                    this.validateNone(target);
                    break;
                case TextContent$1.REQUIRED:
                    this.validateRequired(target);
                    break;
                case TextContent$1.ACCESSIBLE:
                    this.validateAccessible(target);
                    break;
            }
        });
    }
    /**
     * Validate element has empty text (inter-element whitespace is not considered text)
     */
    validateNone(node) {
        if (classifyNodeText(node) === TextClassification.EMPTY_TEXT) {
            return;
        }
        this.reportError(node, node.meta, `${node.annotatedName} must not have text content`);
    }
    /**
     * Validate element has any text (inter-element whitespace is not considered text)
     */
    validateRequired(node) {
        if (classifyNodeText(node) !== TextClassification.EMPTY_TEXT) {
            return;
        }
        this.reportError(node, node.meta, `${node.annotatedName} must have text content`);
    }
    /**
     * Validate element has accessible text (either regular text or text only
     * exposed in accessibility tree via aria-label or similar)
     */
    validateAccessible(node) {
        /* skip this element if the element isn't present in accessibility tree */
        if (!inAccessibilityTree(node)) {
            return;
        }
        /* if the element or a child has aria-label, alt or default text, etc the
         * element has accessible text */
        if (haveAccessibleText(node)) {
            return;
        }
        this.reportError(node, node.meta, `${node.annotatedName} must have accessible text`);
    }
    reportError(node, meta, message) {
        this.report(node, message, null, {
            tagName: node.tagName,
            textContent: meta.textContent,
        });
    }
}

var entities$1 = [
	"&aacute;",
	"&abreve;",
	"&ac;",
	"&acd;",
	"&ace;",
	"&acirc;",
	"&acute;",
	"&acy;",
	"&aelig;",
	"&af;",
	"&afr;",
	"&agrave;",
	"&alefsym;",
	"&aleph;",
	"&alpha;",
	"&amacr;",
	"&amalg;",
	"&amp;",
	"&and;",
	"&andand;",
	"&andd;",
	"&andslope;",
	"&andv;",
	"&ang;",
	"&ange;",
	"&angle;",
	"&angmsd;",
	"&angmsdaa;",
	"&angmsdab;",
	"&angmsdac;",
	"&angmsdad;",
	"&angmsdae;",
	"&angmsdaf;",
	"&angmsdag;",
	"&angmsdah;",
	"&angrt;",
	"&angrtvb;",
	"&angrtvbd;",
	"&angsph;",
	"&angst;",
	"&angzarr;",
	"&aogon;",
	"&aopf;",
	"&ap;",
	"&apacir;",
	"&ape;",
	"&apid;",
	"&apos;",
	"&applyfunction;",
	"&approx;",
	"&approxeq;",
	"&aring;",
	"&ascr;",
	"&assign;",
	"&ast;",
	"&asymp;",
	"&asympeq;",
	"&atilde;",
	"&auml;",
	"&awconint;",
	"&awint;",
	"&backcong;",
	"&backepsilon;",
	"&backprime;",
	"&backsim;",
	"&backsimeq;",
	"&backslash;",
	"&barv;",
	"&barvee;",
	"&barwed;",
	"&barwedge;",
	"&bbrk;",
	"&bbrktbrk;",
	"&bcong;",
	"&bcy;",
	"&bdquo;",
	"&becaus;",
	"&because;",
	"&bemptyv;",
	"&bepsi;",
	"&bernou;",
	"&bernoullis;",
	"&beta;",
	"&beth;",
	"&between;",
	"&bfr;",
	"&bigcap;",
	"&bigcirc;",
	"&bigcup;",
	"&bigodot;",
	"&bigoplus;",
	"&bigotimes;",
	"&bigsqcup;",
	"&bigstar;",
	"&bigtriangledown;",
	"&bigtriangleup;",
	"&biguplus;",
	"&bigvee;",
	"&bigwedge;",
	"&bkarow;",
	"&blacklozenge;",
	"&blacksquare;",
	"&blacktriangle;",
	"&blacktriangledown;",
	"&blacktriangleleft;",
	"&blacktriangleright;",
	"&blank;",
	"&blk12;",
	"&blk14;",
	"&blk34;",
	"&block;",
	"&bne;",
	"&bnequiv;",
	"&bnot;",
	"&bopf;",
	"&bot;",
	"&bottom;",
	"&bowtie;",
	"&boxbox;",
	"&boxdl;",
	"&boxdr;",
	"&boxh;",
	"&boxhd;",
	"&boxhu;",
	"&boxminus;",
	"&boxplus;",
	"&boxtimes;",
	"&boxul;",
	"&boxur;",
	"&boxv;",
	"&boxvh;",
	"&boxvl;",
	"&boxvr;",
	"&bprime;",
	"&breve;",
	"&brvbar;",
	"&bscr;",
	"&bsemi;",
	"&bsim;",
	"&bsime;",
	"&bsol;",
	"&bsolb;",
	"&bsolhsub;",
	"&bull;",
	"&bullet;",
	"&bump;",
	"&bumpe;",
	"&bumpeq;",
	"&cacute;",
	"&cap;",
	"&capand;",
	"&capbrcup;",
	"&capcap;",
	"&capcup;",
	"&capdot;",
	"&capitaldifferentiald;",
	"&caps;",
	"&caret;",
	"&caron;",
	"&cayleys;",
	"&ccaps;",
	"&ccaron;",
	"&ccedil;",
	"&ccirc;",
	"&cconint;",
	"&ccups;",
	"&ccupssm;",
	"&cdot;",
	"&cedil;",
	"&cedilla;",
	"&cemptyv;",
	"&cent;",
	"&centerdot;",
	"&cfr;",
	"&chcy;",
	"&check;",
	"&checkmark;",
	"&chi;",
	"&cir;",
	"&circ;",
	"&circeq;",
	"&circlearrowleft;",
	"&circlearrowright;",
	"&circledast;",
	"&circledcirc;",
	"&circleddash;",
	"&circledot;",
	"&circledr;",
	"&circleds;",
	"&circleminus;",
	"&circleplus;",
	"&circletimes;",
	"&cire;",
	"&cirfnint;",
	"&cirmid;",
	"&cirscir;",
	"&clockwisecontourintegral;",
	"&closecurlydoublequote;",
	"&closecurlyquote;",
	"&clubs;",
	"&clubsuit;",
	"&colon;",
	"&colone;",
	"&coloneq;",
	"&comma;",
	"&commat;",
	"&comp;",
	"&compfn;",
	"&complement;",
	"&complexes;",
	"&cong;",
	"&congdot;",
	"&congruent;",
	"&conint;",
	"&contourintegral;",
	"&copf;",
	"&coprod;",
	"&coproduct;",
	"&copy;",
	"&copysr;",
	"&counterclockwisecontourintegral;",
	"&crarr;",
	"&cross;",
	"&cscr;",
	"&csub;",
	"&csube;",
	"&csup;",
	"&csupe;",
	"&ctdot;",
	"&cudarrl;",
	"&cudarrr;",
	"&cuepr;",
	"&cuesc;",
	"&cularr;",
	"&cularrp;",
	"&cup;",
	"&cupbrcap;",
	"&cupcap;",
	"&cupcup;",
	"&cupdot;",
	"&cupor;",
	"&cups;",
	"&curarr;",
	"&curarrm;",
	"&curlyeqprec;",
	"&curlyeqsucc;",
	"&curlyvee;",
	"&curlywedge;",
	"&curren;",
	"&curvearrowleft;",
	"&curvearrowright;",
	"&cuvee;",
	"&cuwed;",
	"&cwconint;",
	"&cwint;",
	"&cylcty;",
	"&dagger;",
	"&daleth;",
	"&darr;",
	"&dash;",
	"&dashv;",
	"&dbkarow;",
	"&dblac;",
	"&dcaron;",
	"&dcy;",
	"&dd;",
	"&ddagger;",
	"&ddarr;",
	"&ddotrahd;",
	"&ddotseq;",
	"&deg;",
	"&del;",
	"&delta;",
	"&demptyv;",
	"&dfisht;",
	"&dfr;",
	"&dhar;",
	"&dharl;",
	"&dharr;",
	"&diacriticalacute;",
	"&diacriticaldot;",
	"&diacriticaldoubleacute;",
	"&diacriticalgrave;",
	"&diacriticaltilde;",
	"&diam;",
	"&diamond;",
	"&diamondsuit;",
	"&diams;",
	"&die;",
	"&differentiald;",
	"&digamma;",
	"&disin;",
	"&div;",
	"&divide;",
	"&divideontimes;",
	"&divonx;",
	"&djcy;",
	"&dlcorn;",
	"&dlcrop;",
	"&dollar;",
	"&dopf;",
	"&dot;",
	"&dotdot;",
	"&doteq;",
	"&doteqdot;",
	"&dotequal;",
	"&dotminus;",
	"&dotplus;",
	"&dotsquare;",
	"&doublebarwedge;",
	"&doublecontourintegral;",
	"&doubledot;",
	"&doubledownarrow;",
	"&doubleleftarrow;",
	"&doubleleftrightarrow;",
	"&doublelefttee;",
	"&doublelongleftarrow;",
	"&doublelongleftrightarrow;",
	"&doublelongrightarrow;",
	"&doublerightarrow;",
	"&doublerighttee;",
	"&doubleuparrow;",
	"&doubleupdownarrow;",
	"&doubleverticalbar;",
	"&downarrow;",
	"&downarrowbar;",
	"&downarrowuparrow;",
	"&downbreve;",
	"&downdownarrows;",
	"&downharpoonleft;",
	"&downharpoonright;",
	"&downleftrightvector;",
	"&downleftteevector;",
	"&downleftvector;",
	"&downleftvectorbar;",
	"&downrightteevector;",
	"&downrightvector;",
	"&downrightvectorbar;",
	"&downtee;",
	"&downteearrow;",
	"&drbkarow;",
	"&drcorn;",
	"&drcrop;",
	"&dscr;",
	"&dscy;",
	"&dsol;",
	"&dstrok;",
	"&dtdot;",
	"&dtri;",
	"&dtrif;",
	"&duarr;",
	"&duhar;",
	"&dwangle;",
	"&dzcy;",
	"&dzigrarr;",
	"&eacute;",
	"&easter;",
	"&ecaron;",
	"&ecir;",
	"&ecirc;",
	"&ecolon;",
	"&ecy;",
	"&eddot;",
	"&edot;",
	"&ee;",
	"&efdot;",
	"&efr;",
	"&eg;",
	"&egrave;",
	"&egs;",
	"&egsdot;",
	"&el;",
	"&element;",
	"&elinters;",
	"&ell;",
	"&els;",
	"&elsdot;",
	"&emacr;",
	"&empty;",
	"&emptyset;",
	"&emptysmallsquare;",
	"&emptyv;",
	"&emptyverysmallsquare;",
	"&emsp13;",
	"&emsp14;",
	"&emsp;",
	"&eng;",
	"&ensp;",
	"&eogon;",
	"&eopf;",
	"&epar;",
	"&eparsl;",
	"&eplus;",
	"&epsi;",
	"&epsilon;",
	"&epsiv;",
	"&eqcirc;",
	"&eqcolon;",
	"&eqsim;",
	"&eqslantgtr;",
	"&eqslantless;",
	"&equal;",
	"&equals;",
	"&equaltilde;",
	"&equest;",
	"&equilibrium;",
	"&equiv;",
	"&equivdd;",
	"&eqvparsl;",
	"&erarr;",
	"&erdot;",
	"&escr;",
	"&esdot;",
	"&esim;",
	"&eta;",
	"&eth;",
	"&euml;",
	"&euro;",
	"&excl;",
	"&exist;",
	"&exists;",
	"&expectation;",
	"&exponentiale;",
	"&fallingdotseq;",
	"&fcy;",
	"&female;",
	"&ffilig;",
	"&fflig;",
	"&ffllig;",
	"&ffr;",
	"&filig;",
	"&filledsmallsquare;",
	"&filledverysmallsquare;",
	"&fjlig;",
	"&flat;",
	"&fllig;",
	"&fltns;",
	"&fnof;",
	"&fopf;",
	"&forall;",
	"&fork;",
	"&forkv;",
	"&fouriertrf;",
	"&fpartint;",
	"&frac12;",
	"&frac13;",
	"&frac14;",
	"&frac15;",
	"&frac16;",
	"&frac18;",
	"&frac23;",
	"&frac25;",
	"&frac34;",
	"&frac35;",
	"&frac38;",
	"&frac45;",
	"&frac56;",
	"&frac58;",
	"&frac78;",
	"&frasl;",
	"&frown;",
	"&fscr;",
	"&gacute;",
	"&gamma;",
	"&gammad;",
	"&gap;",
	"&gbreve;",
	"&gcedil;",
	"&gcirc;",
	"&gcy;",
	"&gdot;",
	"&ge;",
	"&gel;",
	"&geq;",
	"&geqq;",
	"&geqslant;",
	"&ges;",
	"&gescc;",
	"&gesdot;",
	"&gesdoto;",
	"&gesdotol;",
	"&gesl;",
	"&gesles;",
	"&gfr;",
	"&gg;",
	"&ggg;",
	"&gimel;",
	"&gjcy;",
	"&gl;",
	"&gla;",
	"&gle;",
	"&glj;",
	"&gnap;",
	"&gnapprox;",
	"&gne;",
	"&gneq;",
	"&gneqq;",
	"&gnsim;",
	"&gopf;",
	"&grave;",
	"&greaterequal;",
	"&greaterequalless;",
	"&greaterfullequal;",
	"&greatergreater;",
	"&greaterless;",
	"&greaterslantequal;",
	"&greatertilde;",
	"&gscr;",
	"&gsim;",
	"&gsime;",
	"&gsiml;",
	"&gt;",
	"&gtcc;",
	"&gtcir;",
	"&gtdot;",
	"&gtlpar;",
	"&gtquest;",
	"&gtrapprox;",
	"&gtrarr;",
	"&gtrdot;",
	"&gtreqless;",
	"&gtreqqless;",
	"&gtrless;",
	"&gtrsim;",
	"&gvertneqq;",
	"&gvne;",
	"&hacek;",
	"&hairsp;",
	"&half;",
	"&hamilt;",
	"&hardcy;",
	"&harr;",
	"&harrcir;",
	"&harrw;",
	"&hat;",
	"&hbar;",
	"&hcirc;",
	"&hearts;",
	"&heartsuit;",
	"&hellip;",
	"&hercon;",
	"&hfr;",
	"&hilbertspace;",
	"&hksearow;",
	"&hkswarow;",
	"&hoarr;",
	"&homtht;",
	"&hookleftarrow;",
	"&hookrightarrow;",
	"&hopf;",
	"&horbar;",
	"&horizontalline;",
	"&hscr;",
	"&hslash;",
	"&hstrok;",
	"&humpdownhump;",
	"&humpequal;",
	"&hybull;",
	"&hyphen;",
	"&iacute;",
	"&ic;",
	"&icirc;",
	"&icy;",
	"&idot;",
	"&iecy;",
	"&iexcl;",
	"&iff;",
	"&ifr;",
	"&igrave;",
	"&ii;",
	"&iiiint;",
	"&iiint;",
	"&iinfin;",
	"&iiota;",
	"&ijlig;",
	"&im;",
	"&imacr;",
	"&image;",
	"&imaginaryi;",
	"&imagline;",
	"&imagpart;",
	"&imath;",
	"&imof;",
	"&imped;",
	"&implies;",
	"&in;",
	"&incare;",
	"&infin;",
	"&infintie;",
	"&inodot;",
	"&int;",
	"&intcal;",
	"&integers;",
	"&integral;",
	"&intercal;",
	"&intersection;",
	"&intlarhk;",
	"&intprod;",
	"&invisiblecomma;",
	"&invisibletimes;",
	"&iocy;",
	"&iogon;",
	"&iopf;",
	"&iota;",
	"&iprod;",
	"&iquest;",
	"&iscr;",
	"&isin;",
	"&isindot;",
	"&isine;",
	"&isins;",
	"&isinsv;",
	"&isinv;",
	"&it;",
	"&itilde;",
	"&iukcy;",
	"&iuml;",
	"&jcirc;",
	"&jcy;",
	"&jfr;",
	"&jmath;",
	"&jopf;",
	"&jscr;",
	"&jsercy;",
	"&jukcy;",
	"&kappa;",
	"&kappav;",
	"&kcedil;",
	"&kcy;",
	"&kfr;",
	"&kgreen;",
	"&khcy;",
	"&kjcy;",
	"&kopf;",
	"&kscr;",
	"&laarr;",
	"&lacute;",
	"&laemptyv;",
	"&lagran;",
	"&lambda;",
	"&lang;",
	"&langd;",
	"&langle;",
	"&lap;",
	"&laplacetrf;",
	"&laquo;",
	"&larr;",
	"&larrb;",
	"&larrbfs;",
	"&larrfs;",
	"&larrhk;",
	"&larrlp;",
	"&larrpl;",
	"&larrsim;",
	"&larrtl;",
	"&lat;",
	"&latail;",
	"&late;",
	"&lates;",
	"&lbarr;",
	"&lbbrk;",
	"&lbrace;",
	"&lbrack;",
	"&lbrke;",
	"&lbrksld;",
	"&lbrkslu;",
	"&lcaron;",
	"&lcedil;",
	"&lceil;",
	"&lcub;",
	"&lcy;",
	"&ldca;",
	"&ldquo;",
	"&ldquor;",
	"&ldrdhar;",
	"&ldrushar;",
	"&ldsh;",
	"&le;",
	"&leftanglebracket;",
	"&leftarrow;",
	"&leftarrowbar;",
	"&leftarrowrightarrow;",
	"&leftarrowtail;",
	"&leftceiling;",
	"&leftdoublebracket;",
	"&leftdownteevector;",
	"&leftdownvector;",
	"&leftdownvectorbar;",
	"&leftfloor;",
	"&leftharpoondown;",
	"&leftharpoonup;",
	"&leftleftarrows;",
	"&leftrightarrow;",
	"&leftrightarrows;",
	"&leftrightharpoons;",
	"&leftrightsquigarrow;",
	"&leftrightvector;",
	"&lefttee;",
	"&leftteearrow;",
	"&leftteevector;",
	"&leftthreetimes;",
	"&lefttriangle;",
	"&lefttrianglebar;",
	"&lefttriangleequal;",
	"&leftupdownvector;",
	"&leftupteevector;",
	"&leftupvector;",
	"&leftupvectorbar;",
	"&leftvector;",
	"&leftvectorbar;",
	"&leg;",
	"&leq;",
	"&leqq;",
	"&leqslant;",
	"&les;",
	"&lescc;",
	"&lesdot;",
	"&lesdoto;",
	"&lesdotor;",
	"&lesg;",
	"&lesges;",
	"&lessapprox;",
	"&lessdot;",
	"&lesseqgtr;",
	"&lesseqqgtr;",
	"&lessequalgreater;",
	"&lessfullequal;",
	"&lessgreater;",
	"&lessgtr;",
	"&lessless;",
	"&lesssim;",
	"&lessslantequal;",
	"&lesstilde;",
	"&lfisht;",
	"&lfloor;",
	"&lfr;",
	"&lg;",
	"&lge;",
	"&lhar;",
	"&lhard;",
	"&lharu;",
	"&lharul;",
	"&lhblk;",
	"&ljcy;",
	"&ll;",
	"&llarr;",
	"&llcorner;",
	"&lleftarrow;",
	"&llhard;",
	"&lltri;",
	"&lmidot;",
	"&lmoust;",
	"&lmoustache;",
	"&lnap;",
	"&lnapprox;",
	"&lne;",
	"&lneq;",
	"&lneqq;",
	"&lnsim;",
	"&loang;",
	"&loarr;",
	"&lobrk;",
	"&longleftarrow;",
	"&longleftrightarrow;",
	"&longmapsto;",
	"&longrightarrow;",
	"&looparrowleft;",
	"&looparrowright;",
	"&lopar;",
	"&lopf;",
	"&loplus;",
	"&lotimes;",
	"&lowast;",
	"&lowbar;",
	"&lowerleftarrow;",
	"&lowerrightarrow;",
	"&loz;",
	"&lozenge;",
	"&lozf;",
	"&lpar;",
	"&lparlt;",
	"&lrarr;",
	"&lrcorner;",
	"&lrhar;",
	"&lrhard;",
	"&lrm;",
	"&lrtri;",
	"&lsaquo;",
	"&lscr;",
	"&lsh;",
	"&lsim;",
	"&lsime;",
	"&lsimg;",
	"&lsqb;",
	"&lsquo;",
	"&lsquor;",
	"&lstrok;",
	"&lt;",
	"&ltcc;",
	"&ltcir;",
	"&ltdot;",
	"&lthree;",
	"&ltimes;",
	"&ltlarr;",
	"&ltquest;",
	"&ltri;",
	"&ltrie;",
	"&ltrif;",
	"&ltrpar;",
	"&lurdshar;",
	"&luruhar;",
	"&lvertneqq;",
	"&lvne;",
	"&macr;",
	"&male;",
	"&malt;",
	"&maltese;",
	"&map;",
	"&mapsto;",
	"&mapstodown;",
	"&mapstoleft;",
	"&mapstoup;",
	"&marker;",
	"&mcomma;",
	"&mcy;",
	"&mdash;",
	"&mddot;",
	"&measuredangle;",
	"&mediumspace;",
	"&mellintrf;",
	"&mfr;",
	"&mho;",
	"&micro;",
	"&mid;",
	"&midast;",
	"&midcir;",
	"&middot;",
	"&minus;",
	"&minusb;",
	"&minusd;",
	"&minusdu;",
	"&minusplus;",
	"&mlcp;",
	"&mldr;",
	"&mnplus;",
	"&models;",
	"&mopf;",
	"&mp;",
	"&mscr;",
	"&mstpos;",
	"&mu;",
	"&multimap;",
	"&mumap;",
	"&nabla;",
	"&nacute;",
	"&nang;",
	"&nap;",
	"&nape;",
	"&napid;",
	"&napos;",
	"&napprox;",
	"&natur;",
	"&natural;",
	"&naturals;",
	"&nbsp;",
	"&nbump;",
	"&nbumpe;",
	"&ncap;",
	"&ncaron;",
	"&ncedil;",
	"&ncong;",
	"&ncongdot;",
	"&ncup;",
	"&ncy;",
	"&ndash;",
	"&ne;",
	"&nearhk;",
	"&nearr;",
	"&nearrow;",
	"&nedot;",
	"&negativemediumspace;",
	"&negativethickspace;",
	"&negativethinspace;",
	"&negativeverythinspace;",
	"&nequiv;",
	"&nesear;",
	"&nesim;",
	"&nestedgreatergreater;",
	"&nestedlessless;",
	"&newline;",
	"&nexist;",
	"&nexists;",
	"&nfr;",
	"&nge;",
	"&ngeq;",
	"&ngeqq;",
	"&ngeqslant;",
	"&nges;",
	"&ngg;",
	"&ngsim;",
	"&ngt;",
	"&ngtr;",
	"&ngtv;",
	"&nharr;",
	"&nhpar;",
	"&ni;",
	"&nis;",
	"&nisd;",
	"&niv;",
	"&njcy;",
	"&nlarr;",
	"&nldr;",
	"&nle;",
	"&nleftarrow;",
	"&nleftrightarrow;",
	"&nleq;",
	"&nleqq;",
	"&nleqslant;",
	"&nles;",
	"&nless;",
	"&nll;",
	"&nlsim;",
	"&nlt;",
	"&nltri;",
	"&nltrie;",
	"&nltv;",
	"&nmid;",
	"&nobreak;",
	"&nonbreakingspace;",
	"&nopf;",
	"&not;",
	"&notcongruent;",
	"&notcupcap;",
	"&notdoubleverticalbar;",
	"&notelement;",
	"&notequal;",
	"&notequaltilde;",
	"&notexists;",
	"&notgreater;",
	"&notgreaterequal;",
	"&notgreaterfullequal;",
	"&notgreatergreater;",
	"&notgreaterless;",
	"&notgreaterslantequal;",
	"&notgreatertilde;",
	"&nothumpdownhump;",
	"&nothumpequal;",
	"&notin;",
	"&notindot;",
	"&notine;",
	"&notinva;",
	"&notinvb;",
	"&notinvc;",
	"&notlefttriangle;",
	"&notlefttrianglebar;",
	"&notlefttriangleequal;",
	"&notless;",
	"&notlessequal;",
	"&notlessgreater;",
	"&notlessless;",
	"&notlessslantequal;",
	"&notlesstilde;",
	"&notnestedgreatergreater;",
	"&notnestedlessless;",
	"&notni;",
	"&notniva;",
	"&notnivb;",
	"&notnivc;",
	"&notprecedes;",
	"&notprecedesequal;",
	"&notprecedesslantequal;",
	"&notreverseelement;",
	"&notrighttriangle;",
	"&notrighttrianglebar;",
	"&notrighttriangleequal;",
	"&notsquaresubset;",
	"&notsquaresubsetequal;",
	"&notsquaresuperset;",
	"&notsquaresupersetequal;",
	"&notsubset;",
	"&notsubsetequal;",
	"&notsucceeds;",
	"&notsucceedsequal;",
	"&notsucceedsslantequal;",
	"&notsucceedstilde;",
	"&notsuperset;",
	"&notsupersetequal;",
	"&nottilde;",
	"&nottildeequal;",
	"&nottildefullequal;",
	"&nottildetilde;",
	"&notverticalbar;",
	"&npar;",
	"&nparallel;",
	"&nparsl;",
	"&npart;",
	"&npolint;",
	"&npr;",
	"&nprcue;",
	"&npre;",
	"&nprec;",
	"&npreceq;",
	"&nrarr;",
	"&nrarrc;",
	"&nrarrw;",
	"&nrightarrow;",
	"&nrtri;",
	"&nrtrie;",
	"&nsc;",
	"&nsccue;",
	"&nsce;",
	"&nscr;",
	"&nshortmid;",
	"&nshortparallel;",
	"&nsim;",
	"&nsime;",
	"&nsimeq;",
	"&nsmid;",
	"&nspar;",
	"&nsqsube;",
	"&nsqsupe;",
	"&nsub;",
	"&nsube;",
	"&nsubset;",
	"&nsubseteq;",
	"&nsubseteqq;",
	"&nsucc;",
	"&nsucceq;",
	"&nsup;",
	"&nsupe;",
	"&nsupset;",
	"&nsupseteq;",
	"&nsupseteqq;",
	"&ntgl;",
	"&ntilde;",
	"&ntlg;",
	"&ntriangleleft;",
	"&ntrianglelefteq;",
	"&ntriangleright;",
	"&ntrianglerighteq;",
	"&nu;",
	"&num;",
	"&numero;",
	"&numsp;",
	"&nvap;",
	"&nvdash;",
	"&nvge;",
	"&nvgt;",
	"&nvharr;",
	"&nvinfin;",
	"&nvlarr;",
	"&nvle;",
	"&nvlt;",
	"&nvltrie;",
	"&nvrarr;",
	"&nvrtrie;",
	"&nvsim;",
	"&nwarhk;",
	"&nwarr;",
	"&nwarrow;",
	"&nwnear;",
	"&oacute;",
	"&oast;",
	"&ocir;",
	"&ocirc;",
	"&ocy;",
	"&odash;",
	"&odblac;",
	"&odiv;",
	"&odot;",
	"&odsold;",
	"&oelig;",
	"&ofcir;",
	"&ofr;",
	"&ogon;",
	"&ograve;",
	"&ogt;",
	"&ohbar;",
	"&ohm;",
	"&oint;",
	"&olarr;",
	"&olcir;",
	"&olcross;",
	"&oline;",
	"&olt;",
	"&omacr;",
	"&omega;",
	"&omicron;",
	"&omid;",
	"&ominus;",
	"&oopf;",
	"&opar;",
	"&opencurlydoublequote;",
	"&opencurlyquote;",
	"&operp;",
	"&oplus;",
	"&or;",
	"&orarr;",
	"&ord;",
	"&order;",
	"&orderof;",
	"&ordf;",
	"&ordm;",
	"&origof;",
	"&oror;",
	"&orslope;",
	"&orv;",
	"&os;",
	"&oscr;",
	"&oslash;",
	"&osol;",
	"&otilde;",
	"&otimes;",
	"&otimesas;",
	"&ouml;",
	"&ovbar;",
	"&overbar;",
	"&overbrace;",
	"&overbracket;",
	"&overparenthesis;",
	"&par;",
	"&para;",
	"&parallel;",
	"&parsim;",
	"&parsl;",
	"&part;",
	"&partiald;",
	"&pcy;",
	"&percnt;",
	"&period;",
	"&permil;",
	"&perp;",
	"&pertenk;",
	"&pfr;",
	"&phi;",
	"&phiv;",
	"&phmmat;",
	"&phone;",
	"&pi;",
	"&pitchfork;",
	"&piv;",
	"&planck;",
	"&planckh;",
	"&plankv;",
	"&plus;",
	"&plusacir;",
	"&plusb;",
	"&pluscir;",
	"&plusdo;",
	"&plusdu;",
	"&pluse;",
	"&plusminus;",
	"&plusmn;",
	"&plussim;",
	"&plustwo;",
	"&pm;",
	"&poincareplane;",
	"&pointint;",
	"&popf;",
	"&pound;",
	"&pr;",
	"&prap;",
	"&prcue;",
	"&pre;",
	"&prec;",
	"&precapprox;",
	"&preccurlyeq;",
	"&precedes;",
	"&precedesequal;",
	"&precedesslantequal;",
	"&precedestilde;",
	"&preceq;",
	"&precnapprox;",
	"&precneqq;",
	"&precnsim;",
	"&precsim;",
	"&prime;",
	"&primes;",
	"&prnap;",
	"&prne;",
	"&prnsim;",
	"&prod;",
	"&product;",
	"&profalar;",
	"&profline;",
	"&profsurf;",
	"&prop;",
	"&proportion;",
	"&proportional;",
	"&propto;",
	"&prsim;",
	"&prurel;",
	"&pscr;",
	"&psi;",
	"&puncsp;",
	"&qfr;",
	"&qint;",
	"&qopf;",
	"&qprime;",
	"&qscr;",
	"&quaternions;",
	"&quatint;",
	"&quest;",
	"&questeq;",
	"&quot;",
	"&raarr;",
	"&race;",
	"&racute;",
	"&radic;",
	"&raemptyv;",
	"&rang;",
	"&rangd;",
	"&range;",
	"&rangle;",
	"&raquo;",
	"&rarr;",
	"&rarrap;",
	"&rarrb;",
	"&rarrbfs;",
	"&rarrc;",
	"&rarrfs;",
	"&rarrhk;",
	"&rarrlp;",
	"&rarrpl;",
	"&rarrsim;",
	"&rarrtl;",
	"&rarrw;",
	"&ratail;",
	"&ratio;",
	"&rationals;",
	"&rbarr;",
	"&rbbrk;",
	"&rbrace;",
	"&rbrack;",
	"&rbrke;",
	"&rbrksld;",
	"&rbrkslu;",
	"&rcaron;",
	"&rcedil;",
	"&rceil;",
	"&rcub;",
	"&rcy;",
	"&rdca;",
	"&rdldhar;",
	"&rdquo;",
	"&rdquor;",
	"&rdsh;",
	"&re;",
	"&real;",
	"&realine;",
	"&realpart;",
	"&reals;",
	"&rect;",
	"&reg;",
	"&reverseelement;",
	"&reverseequilibrium;",
	"&reverseupequilibrium;",
	"&rfisht;",
	"&rfloor;",
	"&rfr;",
	"&rhar;",
	"&rhard;",
	"&rharu;",
	"&rharul;",
	"&rho;",
	"&rhov;",
	"&rightanglebracket;",
	"&rightarrow;",
	"&rightarrowbar;",
	"&rightarrowleftarrow;",
	"&rightarrowtail;",
	"&rightceiling;",
	"&rightdoublebracket;",
	"&rightdownteevector;",
	"&rightdownvector;",
	"&rightdownvectorbar;",
	"&rightfloor;",
	"&rightharpoondown;",
	"&rightharpoonup;",
	"&rightleftarrows;",
	"&rightleftharpoons;",
	"&rightrightarrows;",
	"&rightsquigarrow;",
	"&righttee;",
	"&rightteearrow;",
	"&rightteevector;",
	"&rightthreetimes;",
	"&righttriangle;",
	"&righttrianglebar;",
	"&righttriangleequal;",
	"&rightupdownvector;",
	"&rightupteevector;",
	"&rightupvector;",
	"&rightupvectorbar;",
	"&rightvector;",
	"&rightvectorbar;",
	"&ring;",
	"&risingdotseq;",
	"&rlarr;",
	"&rlhar;",
	"&rlm;",
	"&rmoust;",
	"&rmoustache;",
	"&rnmid;",
	"&roang;",
	"&roarr;",
	"&robrk;",
	"&ropar;",
	"&ropf;",
	"&roplus;",
	"&rotimes;",
	"&roundimplies;",
	"&rpar;",
	"&rpargt;",
	"&rppolint;",
	"&rrarr;",
	"&rrightarrow;",
	"&rsaquo;",
	"&rscr;",
	"&rsh;",
	"&rsqb;",
	"&rsquo;",
	"&rsquor;",
	"&rthree;",
	"&rtimes;",
	"&rtri;",
	"&rtrie;",
	"&rtrif;",
	"&rtriltri;",
	"&ruledelayed;",
	"&ruluhar;",
	"&rx;",
	"&sacute;",
	"&sbquo;",
	"&sc;",
	"&scap;",
	"&scaron;",
	"&sccue;",
	"&sce;",
	"&scedil;",
	"&scirc;",
	"&scnap;",
	"&scne;",
	"&scnsim;",
	"&scpolint;",
	"&scsim;",
	"&scy;",
	"&sdot;",
	"&sdotb;",
	"&sdote;",
	"&searhk;",
	"&searr;",
	"&searrow;",
	"&sect;",
	"&semi;",
	"&seswar;",
	"&setminus;",
	"&setmn;",
	"&sext;",
	"&sfr;",
	"&sfrown;",
	"&sharp;",
	"&shchcy;",
	"&shcy;",
	"&shortdownarrow;",
	"&shortleftarrow;",
	"&shortmid;",
	"&shortparallel;",
	"&shortrightarrow;",
	"&shortuparrow;",
	"&shy;",
	"&sigma;",
	"&sigmaf;",
	"&sigmav;",
	"&sim;",
	"&simdot;",
	"&sime;",
	"&simeq;",
	"&simg;",
	"&simge;",
	"&siml;",
	"&simle;",
	"&simne;",
	"&simplus;",
	"&simrarr;",
	"&slarr;",
	"&smallcircle;",
	"&smallsetminus;",
	"&smashp;",
	"&smeparsl;",
	"&smid;",
	"&smile;",
	"&smt;",
	"&smte;",
	"&smtes;",
	"&softcy;",
	"&sol;",
	"&solb;",
	"&solbar;",
	"&sopf;",
	"&spades;",
	"&spadesuit;",
	"&spar;",
	"&sqcap;",
	"&sqcaps;",
	"&sqcup;",
	"&sqcups;",
	"&sqrt;",
	"&sqsub;",
	"&sqsube;",
	"&sqsubset;",
	"&sqsubseteq;",
	"&sqsup;",
	"&sqsupe;",
	"&sqsupset;",
	"&sqsupseteq;",
	"&squ;",
	"&square;",
	"&squareintersection;",
	"&squaresubset;",
	"&squaresubsetequal;",
	"&squaresuperset;",
	"&squaresupersetequal;",
	"&squareunion;",
	"&squarf;",
	"&squf;",
	"&srarr;",
	"&sscr;",
	"&ssetmn;",
	"&ssmile;",
	"&sstarf;",
	"&star;",
	"&starf;",
	"&straightepsilon;",
	"&straightphi;",
	"&strns;",
	"&sub;",
	"&subdot;",
	"&sube;",
	"&subedot;",
	"&submult;",
	"&subne;",
	"&subplus;",
	"&subrarr;",
	"&subset;",
	"&subseteq;",
	"&subseteqq;",
	"&subsetequal;",
	"&subsetneq;",
	"&subsetneqq;",
	"&subsim;",
	"&subsub;",
	"&subsup;",
	"&succ;",
	"&succapprox;",
	"&succcurlyeq;",
	"&succeeds;",
	"&succeedsequal;",
	"&succeedsslantequal;",
	"&succeedstilde;",
	"&succeq;",
	"&succnapprox;",
	"&succneqq;",
	"&succnsim;",
	"&succsim;",
	"&suchthat;",
	"&sum;",
	"&sung;",
	"&sup1;",
	"&sup2;",
	"&sup3;",
	"&sup;",
	"&supdot;",
	"&supdsub;",
	"&supe;",
	"&supedot;",
	"&superset;",
	"&supersetequal;",
	"&suphsol;",
	"&suphsub;",
	"&suplarr;",
	"&supmult;",
	"&supne;",
	"&supplus;",
	"&supset;",
	"&supseteq;",
	"&supseteqq;",
	"&supsetneq;",
	"&supsetneqq;",
	"&supsim;",
	"&supsub;",
	"&supsup;",
	"&swarhk;",
	"&swarr;",
	"&swarrow;",
	"&swnwar;",
	"&szlig;",
	"&tab;",
	"&target;",
	"&tau;",
	"&tbrk;",
	"&tcaron;",
	"&tcedil;",
	"&tcy;",
	"&tdot;",
	"&telrec;",
	"&tfr;",
	"&there4;",
	"&therefore;",
	"&theta;",
	"&thetasym;",
	"&thetav;",
	"&thickapprox;",
	"&thicksim;",
	"&thickspace;",
	"&thinsp;",
	"&thinspace;",
	"&thkap;",
	"&thksim;",
	"&thorn;",
	"&tilde;",
	"&tildeequal;",
	"&tildefullequal;",
	"&tildetilde;",
	"&times;",
	"&timesb;",
	"&timesbar;",
	"&timesd;",
	"&tint;",
	"&toea;",
	"&top;",
	"&topbot;",
	"&topcir;",
	"&topf;",
	"&topfork;",
	"&tosa;",
	"&tprime;",
	"&trade;",
	"&triangle;",
	"&triangledown;",
	"&triangleleft;",
	"&trianglelefteq;",
	"&triangleq;",
	"&triangleright;",
	"&trianglerighteq;",
	"&tridot;",
	"&trie;",
	"&triminus;",
	"&tripledot;",
	"&triplus;",
	"&trisb;",
	"&tritime;",
	"&trpezium;",
	"&tscr;",
	"&tscy;",
	"&tshcy;",
	"&tstrok;",
	"&twixt;",
	"&twoheadleftarrow;",
	"&twoheadrightarrow;",
	"&uacute;",
	"&uarr;",
	"&uarrocir;",
	"&ubrcy;",
	"&ubreve;",
	"&ucirc;",
	"&ucy;",
	"&udarr;",
	"&udblac;",
	"&udhar;",
	"&ufisht;",
	"&ufr;",
	"&ugrave;",
	"&uhar;",
	"&uharl;",
	"&uharr;",
	"&uhblk;",
	"&ulcorn;",
	"&ulcorner;",
	"&ulcrop;",
	"&ultri;",
	"&umacr;",
	"&uml;",
	"&underbar;",
	"&underbrace;",
	"&underbracket;",
	"&underparenthesis;",
	"&union;",
	"&unionplus;",
	"&uogon;",
	"&uopf;",
	"&uparrow;",
	"&uparrowbar;",
	"&uparrowdownarrow;",
	"&updownarrow;",
	"&upequilibrium;",
	"&upharpoonleft;",
	"&upharpoonright;",
	"&uplus;",
	"&upperleftarrow;",
	"&upperrightarrow;",
	"&upsi;",
	"&upsih;",
	"&upsilon;",
	"&uptee;",
	"&upteearrow;",
	"&upuparrows;",
	"&urcorn;",
	"&urcorner;",
	"&urcrop;",
	"&uring;",
	"&urtri;",
	"&uscr;",
	"&utdot;",
	"&utilde;",
	"&utri;",
	"&utrif;",
	"&uuarr;",
	"&uuml;",
	"&uwangle;",
	"&vangrt;",
	"&varepsilon;",
	"&varkappa;",
	"&varnothing;",
	"&varphi;",
	"&varpi;",
	"&varpropto;",
	"&varr;",
	"&varrho;",
	"&varsigma;",
	"&varsubsetneq;",
	"&varsubsetneqq;",
	"&varsupsetneq;",
	"&varsupsetneqq;",
	"&vartheta;",
	"&vartriangleleft;",
	"&vartriangleright;",
	"&vbar;",
	"&vbarv;",
	"&vcy;",
	"&vdash;",
	"&vdashl;",
	"&vee;",
	"&veebar;",
	"&veeeq;",
	"&vellip;",
	"&verbar;",
	"&vert;",
	"&verticalbar;",
	"&verticalline;",
	"&verticalseparator;",
	"&verticaltilde;",
	"&verythinspace;",
	"&vfr;",
	"&vltri;",
	"&vnsub;",
	"&vnsup;",
	"&vopf;",
	"&vprop;",
	"&vrtri;",
	"&vscr;",
	"&vsubne;",
	"&vsupne;",
	"&vvdash;",
	"&vzigzag;",
	"&wcirc;",
	"&wedbar;",
	"&wedge;",
	"&wedgeq;",
	"&weierp;",
	"&wfr;",
	"&wopf;",
	"&wp;",
	"&wr;",
	"&wreath;",
	"&wscr;",
	"&xcap;",
	"&xcirc;",
	"&xcup;",
	"&xdtri;",
	"&xfr;",
	"&xharr;",
	"&xi;",
	"&xlarr;",
	"&xmap;",
	"&xnis;",
	"&xodot;",
	"&xopf;",
	"&xoplus;",
	"&xotime;",
	"&xrarr;",
	"&xscr;",
	"&xsqcup;",
	"&xuplus;",
	"&xutri;",
	"&xvee;",
	"&xwedge;",
	"&yacute;",
	"&yacy;",
	"&ycirc;",
	"&ycy;",
	"&yen;",
	"&yfr;",
	"&yicy;",
	"&yopf;",
	"&yscr;",
	"&yucy;",
	"&yuml;",
	"&zacute;",
	"&zcaron;",
	"&zcy;",
	"&zdot;",
	"&zeetrf;",
	"&zerowidthspace;",
	"&zeta;",
	"&zfr;",
	"&zhcy;",
	"&zigrarr;",
	"&zopf;",
	"&zscr;",
	"&zwj;",
	"&zwnj;"
];

const regexp$1 = /&([a-z0-9]+|#x?[0-9a-f]+);/gi;
class UnknownCharReference extends Rule {
    documentation(context) {
        return {
            description: `HTML defines a set of valid character references but ${context || "this"} is not a valid one.`,
            url: ruleDocumentationUrl("@/rules/unrecognized-char-ref.ts"),
        };
    }
    setup() {
        this.on("element:ready", (event) => {
            const node = event.target;
            /* only iterate over direct descendants */
            for (const child of node.childNodes) {
                if (child.nodeType !== NodeType.TEXT_NODE) {
                    continue;
                }
                this.findCharacterReferences(child.textContent, child.location);
            }
        });
        this.on("attr", (event) => {
            /* boolean attributes has no value so nothing to validate */
            if (!event.value) {
                return;
            }
            this.findCharacterReferences(event.value.toString(), event.valueLocation);
        });
    }
    findCharacterReferences(text, location) {
        let match;
        do {
            match = regexp$1.exec(text);
            if (match) {
                const entity = match[0];
                /* assume numeric entities are valid for now */
                if (entity.startsWith("&#")) {
                    continue;
                }
                /* ignore if this is a known character reference name */
                if (entities$1.includes(entity)) {
                    continue;
                }
                const entityLocation = sliceLocation(location, match.index, match.index + entity.length);
                this.report(null, `Unrecognized character reference "${entity}"`, entityLocation, entity);
            }
        } while (match);
    }
}

var Style$1;
(function (Style) {
    Style[Style["Any"] = 0] = "Any";
    Style[Style["AlwaysOmit"] = 1] = "AlwaysOmit";
    Style[Style["AlwaysSelfclose"] = 2] = "AlwaysSelfclose";
})(Style$1 || (Style$1 = {}));
const defaults$4 = {
    style: "omit",
};
class Void extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$4), options));
        this.style = parseStyle$1(this.options.style);
    }
    get deprecated() {
        return true;
    }
    static schema() {
        return {
            style: {
                enum: ["any", "omit", "selfclose", "selfclosing"],
                type: "string",
            },
        };
    }
    documentation() {
        return {
            description: "HTML void elements cannot have any content and must not have an end tag.",
            url: ruleDocumentationUrl("@/rules/void.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const current = event.target; // The current element being closed
            const active = event.previous; // The current active element (that is, the current element on the stack)
            if (current && current.meta) {
                this.validateCurrent(current);
            }
            if (active && active.meta) {
                this.validateActive(active, active.meta);
            }
        });
    }
    validateCurrent(node) {
        if (node.voidElement && node.closed === NodeClosed.EndTag) {
            this.report(null, `End tag for <${node.tagName}> must be omitted`, node.location);
        }
    }
    validateActive(node, meta) {
        /* ignore foreign elements, they may or may not be self-closed and both are
         * valid */
        if (meta.foreign) {
            return;
        }
        const selfOrOmitted = node.closed === NodeClosed.VoidOmitted || node.closed === NodeClosed.VoidSelfClosed;
        if (node.voidElement) {
            if (this.style === Style$1.AlwaysOmit && node.closed === NodeClosed.VoidSelfClosed) {
                this.report(node, `Expected omitted end tag <${node.tagName}> instead of self-closing element <${node.tagName}/>`);
            }
            if (this.style === Style$1.AlwaysSelfclose && node.closed === NodeClosed.VoidOmitted) {
                this.report(node, `Expected self-closing element <${node.tagName}/> instead of omitted end-tag <${node.tagName}>`);
            }
        }
        if (selfOrOmitted && node.voidElement === false) {
            this.report(node, `End tag for <${node.tagName}> must not be omitted`);
        }
    }
}
function parseStyle$1(name) {
    switch (name) {
        case "any":
            return Style$1.Any;
        case "omit":
            return Style$1.AlwaysOmit;
        case "selfclose":
        case "selfclosing":
            return Style$1.AlwaysSelfclose;
        /* istanbul ignore next: covered by schema validation */
        default:
            throw new Error(`Invalid style "${name}" for "void" rule`);
    }
}

class VoidContent extends Rule {
    documentation(tagName) {
        const doc = {
            description: "HTML void elements cannot have any content and must not have content or end tag.",
            url: ruleDocumentationUrl("@/rules/void-content.ts"),
        };
        if (tagName) {
            doc.description = `<${tagName}> is a void element and must not have content or end tag.`;
        }
        return doc;
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.target; // The current element being closed.
            if (!node) {
                return;
            }
            if (!node.voidElement) {
                return;
            }
            if (node.closed === NodeClosed.EndTag) {
                this.report(null, `End tag for <${node.tagName}> must be omitted`, node.location, node.tagName);
            }
        });
    }
}

var Style;
(function (Style) {
    Style[Style["AlwaysOmit"] = 1] = "AlwaysOmit";
    Style[Style["AlwaysSelfclose"] = 2] = "AlwaysSelfclose";
})(Style || (Style = {}));
const defaults$3 = {
    style: "omit",
};
class VoidStyle extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$3), options));
        this.style = parseStyle(this.options.style);
    }
    static schema() {
        return {
            style: {
                enum: ["omit", "selfclose", "selfclosing"],
                type: "string",
            },
        };
    }
    documentation(context) {
        const doc = {
            description: "The current configuration requires a specific style for ending void elements.",
            url: ruleDocumentationUrl("@/rules/void-style.ts"),
        };
        if (context) {
            const [desc, end] = styleDescription(context.style);
            doc.description = `The current configuration requires void elements to ${desc}, use <${context.tagName}${end}> instead.`;
        }
        return doc;
    }
    setup() {
        this.on("tag:end", (event) => {
            const active = event.previous; // The current active element (that is, the current element on the stack)
            if (active && active.meta) {
                this.validateActive(active);
            }
        });
    }
    validateActive(node) {
        /* ignore non-void elements, they must be closed with regular end tag */
        if (!node.voidElement) {
            return;
        }
        if (this.shouldBeOmitted(node)) {
            this.report(node, `Expected omitted end tag <${node.tagName}> instead of self-closing element <${node.tagName}/>`);
        }
        if (this.shouldBeSelfClosed(node)) {
            this.report(node, `Expected self-closing element <${node.tagName}/> instead of omitted end-tag <${node.tagName}>`);
        }
    }
    report(node, message) {
        const context = {
            style: this.style,
            tagName: node.tagName,
        };
        super.report(node, message, null, context);
    }
    shouldBeOmitted(node) {
        return this.style === Style.AlwaysOmit && node.closed === NodeClosed.VoidSelfClosed;
    }
    shouldBeSelfClosed(node) {
        return this.style === Style.AlwaysSelfclose && node.closed === NodeClosed.VoidOmitted;
    }
}
function parseStyle(name) {
    switch (name) {
        case "omit":
            return Style.AlwaysOmit;
        case "selfclose":
        case "selfclosing":
            return Style.AlwaysSelfclose;
        /* istanbul ignore next: covered by schema validation */
        default:
            throw new Error(`Invalid style "${name}" for "void-style" rule`);
    }
}
function styleDescription(style) {
    switch (style) {
        case Style.AlwaysOmit:
            return ["omit end tag", ""];
        case Style.AlwaysSelfclose:
            return ["be self-closed", "/"];
        // istanbul ignore next: will only happen if new styles are added, otherwise this isn't reached
        default:
            throw new Error(`Unknown style`);
    }
}

class H30 extends Rule {
    documentation() {
        return {
            description: "WCAG 2.1 requires each `<a>` anchor link to have a text describing the purpose of the link using either plain text or an `<img>` with the `alt` attribute set.",
            url: ruleDocumentationUrl("@/rules/wcag/h30.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const links = event.document.getElementsByTagName("a");
            for (const link of links) {
                /* ignore links with aria-hidden="true" */
                if (!inAccessibilityTree(link)) {
                    continue;
                }
                /* check if text content is present (or dynamic) */
                const textClassification = classifyNodeText(link);
                if (textClassification !== TextClassification.EMPTY_TEXT) {
                    continue;
                }
                /* check if image with alt-text is present */
                const images = link.querySelectorAll("img");
                if (images.some((image) => hasAltText(image))) {
                    continue;
                }
                /* check if aria-label is present on either the <a> element or a descendant */
                const labels = link.querySelectorAll("[aria-label]");
                if (hasAriaLabel(link) || labels.some((cur) => hasAriaLabel(cur))) {
                    continue;
                }
                this.report(link, "Anchor link must have a text describing its purpose");
            }
        });
    }
}

class H32 extends Rule {
    documentation() {
        return {
            description: "WCAG 2.1 requires each `<form>` element to have at least one submit button.",
            url: ruleDocumentationUrl("@/rules/wcag/h32.ts"),
        };
    }
    setup() {
        /* query all tags with form property, normally this is only the <form> tag
         * but with custom element metadata other tags might be considered form
         * (usually a component wrapping a <form> element) */
        const formTags = this.getTagsWithProperty("form");
        const formSelector = formTags.join(",");
        this.on("dom:ready", (event) => {
            const { document } = event;
            const forms = document.querySelectorAll(formSelector);
            for (const form of forms) {
                /* find nested submit buttons */
                if (hasNestedSubmit(form)) {
                    continue;
                }
                /* find explicitly associated submit buttons */
                if (hasAssociatedSubmit(document, form)) {
                    continue;
                }
                this.report(form, `<${form.tagName}> element must have a submit button`);
            }
        });
    }
}
function isSubmit(node) {
    const type = node.getAttribute("type");
    return Boolean(type && type.valueMatches(/submit|image/));
}
function isAssociated(id, node) {
    const form = node.getAttribute("form");
    return Boolean(form && form.valueMatches(id, true));
}
function hasNestedSubmit(form) {
    const matches = form
        .querySelectorAll("button,input")
        .filter(isSubmit)
        .filter((node) => !node.hasAttribute("form"));
    return matches.length > 0;
}
function hasAssociatedSubmit(document, form) {
    const { id } = form;
    if (!id) {
        return false;
    }
    const matches = document
        .querySelectorAll("button[form],input[form]")
        .filter(isSubmit)
        .filter((node) => isAssociated(id, node));
    return matches.length > 0;
}

class H36 extends Rule {
    documentation() {
        return {
            description: 'WCAG 2.1 requires all images used as submit buttons to have a textual description using the alt attribute. The alt text cannot be empty (`alt=""`).',
            url: ruleDocumentationUrl("@/rules/wcag/h36.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            /* only handle input elements */
            const node = event.previous;
            if (node.tagName !== "input")
                return;
            /* only handle images with type="image" */
            if (node.getAttributeValue("type") !== "image") {
                return;
            }
            if (!hasAltText(node)) {
                this.report(node, "image used as submit button must have alt text");
            }
        });
    }
}

const defaults$2 = {
    allowEmpty: true,
    alias: [],
};
class H37 extends Rule {
    constructor(options) {
        super(Object.assign(Object.assign({}, defaults$2), options));
        /* ensure alias is array */
        if (!Array.isArray(this.options.alias)) {
            this.options.alias = [this.options.alias];
        }
    }
    static schema() {
        return {
            alias: {
                anyOf: [
                    {
                        items: {
                            type: "string",
                        },
                        type: "array",
                    },
                    {
                        type: "string",
                    },
                ],
            },
            allowEmpty: {
                type: "boolean",
            },
        };
    }
    documentation() {
        return {
            description: "Both HTML5 and WCAG 2.0 requires images to have a alternative text for each image.",
            url: ruleDocumentationUrl("@/rules/wcag/h37.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.previous;
            /* only validate images */
            if (!node || node.tagName !== "img") {
                return;
            }
            /* ignore images with aria-hidden="true" or role="presentation" */
            if (!inAccessibilityTree(node)) {
                return;
            }
            /* validate plain alt-attribute */
            const alt = node.getAttributeValue("alt");
            if (alt || (alt === "" && this.options.allowEmpty)) {
                return;
            }
            /* validate if any non-empty alias is present */
            for (const attr of this.options.alias) {
                if (node.getAttribute(attr)) {
                    return;
                }
            }
            this.report(node, '<img> is missing required "alt" attribute', node.location);
        });
    }
}

class H67 extends Rule {
    documentation() {
        return {
            description: "A decorative image cannot have a title attribute. Either remove `title` or add a descriptive `alt` text.",
            url: ruleDocumentationUrl("@/rules/wcag/h67.ts"),
        };
    }
    setup() {
        this.on("tag:end", (event) => {
            const node = event.target;
            /* only validate images */
            if (!node || node.tagName !== "img") {
                return;
            }
            /* ignore images without title */
            const title = node.getAttribute("title");
            if (!title || title.value === "") {
                return;
            }
            /* ignore elements with non-empty alt-text */
            const alt = node.getAttributeValue("alt");
            if (alt && alt !== "") {
                return;
            }
            this.report(node, "<img> with empty alt text cannot have title attribute", title.keyLocation);
        });
    }
}

class H71 extends Rule {
    documentation() {
        return {
            description: "H71: Providing a description for groups of form controls using fieldset and legend elements",
            url: ruleDocumentationUrl("@/rules/wcag/h71.ts"),
        };
    }
    setup() {
        this.on("dom:ready", (event) => {
            const { document } = event;
            const fieldsets = document.querySelectorAll(this.selector);
            for (const fieldset of fieldsets) {
                this.validate(fieldset);
            }
        });
    }
    validate(fieldset) {
        const legend = fieldset.querySelectorAll("> legend");
        if (legend.length === 0) {
            this.reportNode(fieldset);
        }
    }
    reportNode(node) {
        super.report(node, `${node.annotatedName} must have a <legend> as the first child`);
    }
    get selector() {
        return this.getTagsDerivedFrom("fieldset").join(",");
    }
}

const bundledRules$1 = {
    "wcag/h30": H30,
    "wcag/h32": H32,
    "wcag/h36": H36,
    "wcag/h37": H37,
    "wcag/h67": H67,
    "wcag/h71": H71,
};

const bundledRules = Object.assign({ "allowed-links": AllowedLinks, "aria-label-misuse": AriaLabelMisuse, "attr-case": AttrCase, "attr-delimiter": AttrDelimiter, "attr-pattern": AttrPattern, "attr-quotes": AttrQuotes, "attr-spacing": AttrSpacing, "attribute-allowed-values": AttributeAllowedValues, "attribute-boolean-style": AttributeBooleanStyle, "attribute-empty-style": AttributeEmptyStyle, "class-pattern": ClassPattern, "close-attr": CloseAttr, "close-order": CloseOrder, deprecated: Deprecated, "deprecated-rule": DeprecatedRule, "doctype-html": NoStyleTag$1, "doctype-style": DoctypeStyle, "element-case": ElementCase, "element-name": ElementName, "element-permitted-content": ElementPermittedContent, "element-permitted-occurrences": ElementPermittedOccurrences, "element-permitted-order": ElementPermittedOrder, "element-required-attributes": ElementRequiredAttributes, "element-required-content": ElementRequiredContent, "empty-heading": EmptyHeading, "empty-title": EmptyTitle, "heading-level": HeadingLevel, "id-pattern": IdPattern, "input-attributes": InputAttributes, "input-missing-label": InputMissingLabel, "long-title": LongTitle, "meta-refresh": MetaRefresh, "missing-doctype": MissingDoctype, "multiple-labeled-controls": MultipleLabeledControls, "no-autoplay": NoAutoplay, "no-conditional-comment": NoConditionalComment, "no-deprecated-attr": NoDeprecatedAttr, "no-dup-attr": NoDupAttr, "no-dup-class": NoDupClass, "no-dup-id": NoDupID, "no-implicit-close": NoImplicitClose, "no-inline-style": NoInlineStyle, "no-missing-references": NoMissingReferences, "no-multiple-main": NoMultipleMain, "no-raw-characters": NoRawCharacters, "no-redundant-for": NoRedundantFor, "no-redundant-role": NoRedundantRole, "no-self-closing": NoSelfClosing, "no-style-tag": NoStyleTag, "no-trailing-whitespace": NoTrailingWhitespace, "no-unknown-elements": NoUnknownElements, "no-utf8-bom": NoUtf8Bom, "prefer-button": PreferButton, "prefer-native-element": PreferNativeElement, "prefer-tbody": PreferTbody, "require-sri": RequireSri, "script-element": ScriptElement, "script-type": ScriptType, "svg-focusable": SvgFocusable, "text-content": TextContent, "unrecognized-char-ref": UnknownCharReference, void: Void, "void-content": VoidContent, "void-style": VoidStyle }, bundledRules$1);

var defaultConfig = {};

const config$3 = {
    rules: {
        "aria-label-misuse": "error",
        "deprecated-rule": "warn",
        "empty-heading": "error",
        "empty-title": "error",
        "meta-refresh": "error",
        "multiple-labeled-controls": "error",
        "no-autoplay": ["error", { include: ["audio", "video"] }],
        "no-dup-id": "error",
        "no-redundant-for": "error",
        "no-redundant-role": "error",
        "prefer-native-element": "error",
        "svg-focusable": "error",
        "text-content": "error",
        "wcag/h30": "error",
        "wcag/h32": "error",
        "wcag/h36": "error",
        "wcag/h37": "error",
        "wcag/h67": "error",
        "wcag/h71": "error",
    },
};

const config$2 = {
    rules: {
        "input-missing-label": "error",
        "heading-level": "error",
        "missing-doctype": "error",
        "no-missing-references": "error",
        "require-sri": "error",
    },
};

const config$1 = {
    rules: {
        "aria-label-misuse": "error",
        "attr-case": "error",
        "attr-delimiter": "error",
        "attr-quotes": "error",
        "attr-spacing": "error",
        "attribute-allowed-values": "error",
        "attribute-boolean-style": "error",
        "attribute-empty-style": "error",
        "close-attr": "error",
        "close-order": "error",
        deprecated: "error",
        "deprecated-rule": "warn",
        "doctype-html": "error",
        "doctype-style": "error",
        "element-case": "error",
        "element-name": "error",
        "element-permitted-content": "error",
        "element-permitted-occurrences": "error",
        "element-permitted-order": "error",
        "element-required-attributes": "error",
        "element-required-content": "error",
        "empty-heading": "error",
        "empty-title": "error",
        "input-attributes": "error",
        "long-title": "error",
        "meta-refresh": "error",
        "multiple-labeled-controls": "error",
        "no-autoplay": ["error", { include: ["audio", "video"] }],
        "no-conditional-comment": "error",
        "no-deprecated-attr": "error",
        "no-dup-attr": "error",
        "no-dup-class": "error",
        "no-dup-id": "error",
        "no-implicit-close": "error",
        "no-inline-style": "error",
        "no-multiple-main": "error",
        "no-raw-characters": "error",
        "no-redundant-for": "error",
        "no-redundant-role": "error",
        "no-self-closing": "error",
        "no-trailing-whitespace": "error",
        "no-utf8-bom": "error",
        "prefer-button": "error",
        "prefer-native-element": "error",
        "prefer-tbody": "error",
        "script-element": "error",
        "script-type": "error",
        "svg-focusable": "error",
        "text-content": "error",
        "unrecognized-char-ref": "error",
        void: "off",
        "void-content": "error",
        "void-style": "error",
        "wcag/h30": "error",
        "wcag/h32": "error",
        "wcag/h36": "error",
        "wcag/h37": "error",
        "wcag/h67": "error",
        "wcag/h71": "error",
    },
};

const config = {
    rules: {
        "attr-spacing": "error",
        "attribute-allowed-values": "error",
        "close-attr": "error",
        "close-order": "error",
        deprecated: "error",
        "deprecated-rule": "warn",
        "doctype-html": "error",
        "element-name": "error",
        "element-permitted-content": "error",
        "element-permitted-occurrences": "error",
        "element-permitted-order": "error",
        "element-required-attributes": "error",
        "element-required-content": "error",
        "multiple-labeled-controls": "error",
        "no-deprecated-attr": "error",
        "no-dup-attr": "error",
        "no-dup-id": "error",
        "no-multiple-main": "error",
        "no-raw-characters": ["error", { relaxed: true }],
        "script-element": "error",
        "unrecognized-char-ref": "error",
        "void-content": "error",
    },
};

const presets = {
    "html-validate:a17y": config$3,
    "html-validate:document": config$2,
    "html-validate:recommended": config$1,
    "html-validate:standard": config,
    /* @deprecated aliases */
    "htmlvalidate:recommended": config$1,
    "htmlvalidate:document": config$2,
};

/**
 * A resolved configuration is a normalized configuration with all extends,
 * plugins etc resolved.
 */
class ResolvedConfig {
    constructor({ metaTable, plugins, rules, transformers }) {
        this.metaTable = metaTable;
        this.plugins = plugins;
        this.rules = rules;
        this.transformers = transformers;
    }
    getMetaTable() {
        return this.metaTable;
    }
    getPlugins() {
        return this.plugins;
    }
    getRules() {
        return this.rules;
    }
    /**
     * Transform a source.
     *
     * When transforming zero or more new sources will be generated.
     *
     * @param source - Current source to transform.
     * @param filename - If set it is the filename used to match
     * transformer. Default is to use filename from source.
     * @returns A list of transformed sources ready for validation.
     */
    transformSource(source, filename) {
        const transformer = this.findTransformer(filename || source.filename);
        const context = {
            hasChain: (filename) => {
                return !!this.findTransformer(filename);
            },
            chain: (source, filename) => {
                return this.transformSource(source, filename);
            },
        };
        if (transformer) {
            try {
                return Array.from(transformer.fn.call(context, source), (cur) => {
                    /* keep track of which transformers that has been run on this source
                     * by appending this entry to the transformedBy array */
                    cur.transformedBy = cur.transformedBy || [];
                    cur.transformedBy.push(transformer.name);
                    return cur;
                });
            }
            catch (err) {
                throw new NestedError(`When transforming "${source.filename}": ${err.message}`, err);
            }
        }
        else {
            return [source];
        }
    }
    /**
     * Wrapper around [[transformSource]] which reads a file before passing it
     * as-is to transformSource.
     *
     * @param source - Filename to transform (according to configured
     * transformations)
     * @returns A list of transformed sources ready for validation.
     */
    transformFilename(filename) {
        const data = fs.readFileSync(filename, { encoding: "utf8" });
        const source = {
            data,
            filename,
            line: 1,
            column: 1,
            offset: 0,
            originalData: data,
        };
        return this.transformSource(source);
    }
    /**
     * Returns true if a transformer matches given filename.
     */
    canTransform(filename) {
        const entry = this.findTransformer(filename);
        return !!entry;
    }
    findTransformer(filename) {
        const match = this.transformers.find((entry) => entry.pattern.test(filename));
        return match !== null && match !== void 0 ? match : null;
    }
}

let rootDirCache = null;
const ajv = new Ajv({ strict: true, strictTuples: true, strictTypes: true });
ajv.addMetaSchema(ajvSchemaDraft);
const validator = ajv.compile(configurationSchema);
function overwriteMerge(a, b) {
    return b;
}
function mergeInternal(base, rhs) {
    const dst = deepmerge(base, Object.assign(Object.assign({}, rhs), { rules: {} }));
    /* rules need some special care, should overwrite arrays instead of
     * concaternation, i.e. ["error", {...options}] should not be merged by
     * appending to old value */
    if (rhs.rules) {
        dst.rules = deepmerge(dst.rules, rhs.rules, { arrayMerge: overwriteMerge });
    }
    /* root property is merged with boolean "or" since it should always be truthy
     * if any config has it set. */
    const root = base.root || rhs.root;
    if (root) {
        dst.root = root;
    }
    return dst;
}
function loadFromFile(filename) {
    let json;
    try {
        /* load using require as it can process both js and json */
        json = requireUncached(filename);
    }
    catch (err) {
        throw new ConfigError(`Failed to read configuration from "${filename}"`, err);
    }
    /* expand any relative paths */
    for (const key of ["extends", "elements", "plugins"]) {
        if (!json[key])
            continue;
        json[key] = json[key].map((ref) => {
            return Config.expandRelative(ref, path.dirname(filename));
        });
    }
    return json;
}
/**
 * Configuration holder.
 *
 * Each file being validated will have a unique instance of this class.
 */
class Config {
    constructor(options) {
        var _a;
        this.transformers = [];
        const initial = {
            extends: [],
            plugins: [],
            rules: {},
            transform: {},
        };
        this.config = mergeInternal(initial, options || {});
        this.metaTable = null;
        this.rootDir = this.findRootDir();
        this.initialized = false;
        /* load plugins */
        this.plugins = this.loadPlugins(this.config.plugins || []);
        this.configurations = this.loadConfigurations(this.plugins);
        this.extendMeta(this.plugins);
        /* process extended configs */
        for (const extend of (_a = this.config.extends) !== null && _a !== void 0 ? _a : []) {
            this.config = this.extendConfig(extend);
        }
        /* rules explicitly set by passed options should have precedence over any
         * extended rules, not the other way around. */
        if (options && options.rules) {
            this.config = mergeInternal(this.config, { rules: options.rules });
        }
    }
    /**
     * Create a new blank configuration. See also `Config.defaultConfig()`.
     */
    static empty() {
        return new Config({
            extends: [],
            rules: {},
            plugins: [],
            transform: {},
        });
    }
    /**
     * Create configuration from object.
     */
    static fromObject(options, filename = null) {
        Config.validate(options, filename);
        return new Config(options);
    }
    /**
     * Read configuration from filename.
     *
     * Note: this reads configuration data from a file. If you intent to load
     * configuration for a file to validate use `ConfigLoader.fromTarget()`.
     *
     * @param filename - The file to read from or one of the presets such as
     * `html-validate:recommended`.
     */
    static fromFile(filename) {
        const configdata = loadFromFile(filename);
        return Config.fromObject(configdata, filename);
    }
    /**
     * Validate configuration data.
     *
     * Throws SchemaValidationError if invalid.
     */
    static validate(configData, filename = null) {
        var _a;
        const valid = validator(configData);
        if (!valid) {
            throw new SchemaValidationError(filename, `Invalid configuration`, configData, configurationSchema, (_a = validator.errors) !== null && _a !== void 0 ? _a : []);
        }
        if (configData.rules) {
            const normalizedRules = Config.getRulesObject(configData.rules);
            for (const [ruleId, [, ruleOptions]] of normalizedRules.entries()) {
                const cls = bundledRules[ruleId];
                const path = `/rules/${ruleId}/1`;
                Rule.validateOptions(cls, ruleId, path, ruleOptions, filename, configData);
            }
        }
    }
    /**
     * Load a default configuration object.
     */
    static defaultConfig() {
        return new Config(defaultConfig);
    }
    /**
     * Initialize plugins, transforms etc.
     *
     * Must be called before trying to use config. Can safely be called multiple
     * times.
     */
    init() {
        if (this.initialized) {
            return;
        }
        /* precompile transform patterns */
        this.transformers = this.precompileTransformers(this.config.transform || {});
        this.initialized = true;
    }
    /**
     * Returns true if this configuration is marked as "root".
     */
    isRootFound() {
        return Boolean(this.config.root);
    }
    /**
     * Returns a new configuration as a merge of the two. Entries from the passed
     * object takes priority over this object.
     *
     * @param rhs - Configuration to merge with this one.
     */
    merge(rhs) {
        return new Config(mergeInternal(this.config, rhs.config));
    }
    extendConfig(entry) {
        let base;
        if (this.configurations.has(entry)) {
            base = this.configurations.get(entry);
        }
        else {
            base = Config.fromFile(entry).config;
        }
        return mergeInternal(this.config, base);
    }
    /**
     * Get element metadata.
     */
    getMetaTable() {
        /* use cached table if it exists */
        if (this.metaTable) {
            return this.metaTable;
        }
        const metaTable = new MetaTable();
        const source = this.config.elements || ["html5"];
        /* extend validation schema from plugins */
        for (const plugin of this.getPlugins()) {
            if (plugin.elementSchema) {
                metaTable.extendValidationSchema(plugin.elementSchema);
            }
        }
        /* load from all entries */
        for (const entry of source) {
            /* load meta directly from entry */
            if (typeof entry !== "string") {
                metaTable.loadFromObject(entry);
                continue;
            }
            let filename;
            /* try searching builtin metadata */
            filename = path.join(projectRoot, "elements", `${entry}.json`);
            if (fs.existsSync(filename)) {
                metaTable.loadFromFile(filename);
                continue;
            }
            /* try as regular file */
            filename = entry.replace("<rootDir>", this.rootDir);
            if (fs.existsSync(filename)) {
                metaTable.loadFromFile(filename);
                continue;
            }
            /* assume it is loadable with require() */
            try {
                metaTable.loadFromObject(legacyRequire(entry));
            }
            catch (err) {
                throw new ConfigError(`Failed to load elements from "${entry}": ${err.message}`, err);
            }
        }
        metaTable.init();
        return (this.metaTable = metaTable);
    }
    /**
     * @internal exposed for testing only
     */
    static expandRelative(src, currentPath) {
        if (src[0] === ".") {
            return path.normalize(`${currentPath}/${src}`);
        }
        return src;
    }
    /**
     * Get a copy of internal configuration data.
     *
     * @internal primary purpose is unittests
     */
    get() {
        const config = Object.assign({}, this.config);
        if (config.elements) {
            config.elements = config.elements.map((cur) => {
                if (typeof cur === "string") {
                    return cur.replace(this.rootDir, "<rootDir>");
                }
                else {
                    return cur;
                }
            });
        }
        return config;
    }
    /**
     * Get all configured rules, their severity and options.
     */
    getRules() {
        var _a;
        return Config.getRulesObject((_a = this.config.rules) !== null && _a !== void 0 ? _a : {});
    }
    static getRulesObject(src) {
        const rules = new Map();
        for (const [ruleId, data] of Object.entries(src)) {
            let options = data;
            if (!Array.isArray(options)) {
                options = [options, {}];
            }
            else if (options.length === 1) {
                options = [options[0], {}];
            }
            const severity = parseSeverity(options[0]);
            rules.set(ruleId, [severity, options[1]]);
        }
        return rules;
    }
    /**
     * Get all configured plugins.
     */
    getPlugins() {
        return this.plugins;
    }
    loadPlugins(plugins) {
        return plugins.map((moduleName) => {
            try {
                const plugin = legacyRequire(moduleName.replace("<rootDir>", this.rootDir));
                plugin.name = plugin.name || moduleName;
                plugin.originalName = moduleName;
                return plugin;
            }
            catch (err) {
                throw new ConfigError(`Failed to load plugin "${moduleName}": ${err}`, err);
            }
        });
    }
    loadConfigurations(plugins) {
        const configs = new Map();
        /* builtin presets */
        for (const [name, config] of Object.entries(presets)) {
            Config.validate(config, name);
            configs.set(name, config);
        }
        /* presets from plugins */
        for (const plugin of plugins) {
            for (const [name, config] of Object.entries(plugin.configs || {})) {
                if (!config)
                    continue;
                Config.validate(config, name);
                /* add configuration with name provided by plugin */
                configs.set(`${plugin.name}:${name}`, config);
                /* add configuration with name provided by user (in config file) */
                if (plugin.name !== plugin.originalName) {
                    configs.set(`${plugin.originalName}:${name}`, config);
                }
            }
        }
        return configs;
    }
    extendMeta(plugins) {
        for (const plugin of plugins) {
            if (!plugin.elementSchema) {
                continue;
            }
            const { properties } = plugin.elementSchema;
            if (!properties) {
                continue;
            }
            for (const [key, schema] of Object.entries(properties)) {
                if (schema.copyable && !MetaCopyableProperty.includes(key)) {
                    MetaCopyableProperty.push(key);
                }
            }
        }
    }
    /**
     * Resolve all configuration and return a [[ResolvedConfig]] instance.
     *
     * A resolved configuration will merge all extended configs and load all
     * plugins and transformers, and normalize the rest of the configuration.
     */
    resolve() {
        return new ResolvedConfig(this.resolveData());
    }
    /**
     * Same as [[resolve]] but returns the raw configuration data instead of
     * [[ResolvedConfig]] instance. Mainly used for testing.
     *
     * @internal
     */
    resolveData() {
        return {
            metaTable: this.getMetaTable(),
            plugins: this.getPlugins(),
            rules: this.getRules(),
            transformers: this.transformers,
        };
    }
    precompileTransformers(transform) {
        return Object.entries(transform).map(([pattern, name]) => {
            try {
                const fn = this.getTransformFunction(name);
                const version = fn.api || 0;
                /* check if transformer version is supported */
                if (version !== TRANSFORMER_API.VERSION) {
                    throw new ConfigError(`Transformer uses API version ${version} but only version ${TRANSFORMER_API.VERSION} is supported`);
                }
                return {
                    // eslint-disable-next-line security/detect-non-literal-regexp
                    pattern: new RegExp(pattern),
                    name,
                    fn,
                };
            }
            catch (err) {
                if (err instanceof ConfigError) {
                    throw new ConfigError(`Failed to load transformer "${name}": ${err.message}`, err);
                }
                else {
                    throw new ConfigError(`Failed to load transformer "${name}"`, err);
                }
            }
        });
    }
    /**
     * Get transformation function requested by configuration.
     *
     * Searches:
     *
     * - Named transformers from plugins.
     * - Unnamed transformer from plugin.
     * - Standalone modules (local or node_modules)
     *
     * @param name - Key from configuration
     */
    getTransformFunction(name) {
        /* try to match a named transformer from plugin */
        const match = name.match(/(.*):(.*)/);
        if (match) {
            const [, pluginName, key] = match;
            return this.getNamedTransformerFromPlugin(name, pluginName, key);
        }
        /* try to match an unnamed transformer from plugin */
        const plugin = this.plugins.find((cur) => cur.name === name);
        if (plugin) {
            return this.getUnnamedTransformerFromPlugin(name, plugin);
        }
        /* assume transformer refers to a regular module */
        return this.getTransformerFromModule(name);
    }
    /**
     * @param name - Original name from configuration
     * @param pluginName - Name of plugin
     * @param key - Name of transform (from plugin)
     */
    getNamedTransformerFromPlugin(name, pluginName, key) {
        const plugin = this.plugins.find((cur) => cur.name === pluginName);
        if (!plugin) {
            throw new ConfigError(`No plugin named "${pluginName}" has been loaded`);
        }
        if (!plugin.transformer) {
            throw new ConfigError(`Plugin does not expose any transformer`);
        }
        if (typeof plugin.transformer === "function") {
            throw new ConfigError(`Transformer "${name}" refers to named transformer but plugin exposes only unnamed, use "${pluginName}" instead.`);
        }
        const transformer = plugin.transformer[key];
        if (!transformer) {
            throw new ConfigError(`Plugin "${pluginName}" does not expose a transformer named "${key}".`);
        }
        return transformer;
    }
    /**
     * @param name - Original name from configuration
     * @param plugin - Plugin instance
     */
    getUnnamedTransformerFromPlugin(name, plugin) {
        if (!plugin.transformer) {
            throw new ConfigError(`Plugin does not expose any transformer`);
        }
        if (typeof plugin.transformer !== "function") {
            if (plugin.transformer.default) {
                return plugin.transformer.default;
            }
            throw new ConfigError(`Transformer "${name}" refers to unnamed transformer but plugin exposes only named.`);
        }
        return plugin.transformer;
    }
    getTransformerFromModule(name) {
        /* expand <rootDir> */
        const moduleName = name.replace("<rootDir>", this.rootDir);
        const fn = legacyRequire(moduleName);
        /* sanity check */
        if (typeof fn !== "function") {
            /* this is not a proper transformer, is it a plugin exposing a transformer? */
            if (fn.transformer) {
                throw new ConfigError(`Module is not a valid transformer. This looks like a plugin, did you forget to load the plugin first?`);
            }
            throw new ConfigError(`Module is not a valid transformer.`);
        }
        return fn;
    }
    /**
     * @internal
     */
    get rootDirCache() {
        /* return global instance */
        return rootDirCache;
    }
    set rootDirCache(value) {
        /* set global instance */
        rootDirCache = value;
    }
    findRootDir() {
        const cache = this.rootDirCache;
        if (cache !== null) {
            return cache;
        }
        /* try to locate package.json */
        let current = process.cwd();
        // eslint-disable-next-line no-constant-condition
        while (true) {
            const search = path.join(current, "package.json");
            if (fs.existsSync(search)) {
                return (this.rootDirCache = current);
            }
            /* get the parent directory */
            const child = current;
            current = path.dirname(current);
            /* stop if this is the root directory */
            if (current === child) {
                break;
            }
        }
        /* default to working directory if no package.json is found */
        return (this.rootDirCache = process.cwd());
    }
}

/**
 * Configuration loader interface.
 *
 * A configuration loader takes a handle (typically a filename) and returns a
 * configuration for it.
 *
 * @public
 */
class ConfigLoader {
    constructor(config, configFactory = Config) {
        const defaults = configFactory.empty();
        this.configFactory = configFactory;
        this.globalConfig = defaults.merge(config ? this.loadFromObject(config) : this.defaultConfig());
    }
    empty() {
        return this.configFactory.empty();
    }
    loadFromObject(options, filename) {
        return this.configFactory.fromObject(options, filename);
    }
    loadFromFile(filename) {
        return this.configFactory.fromFile(filename);
    }
}

/*! *****************************************************************************
Copyright (c) Microsoft Corporation. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License"); you may not use
this file except in compliance with the License. You may obtain a copy of the
License at http://www.apache.org/licenses/LICENSE-2.0

THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
MERCHANTABLITY OR NON-INFRINGEMENT.

See the Apache Version 2.0 License for specific language governing permissions
and limitations under the License.
***************************************************************************** */

function __rest(s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) if (e.indexOf(p[i]) < 0)
            t[p[i]] = s[p[i]];
    return t;
}

class EventHandler {
    constructor() {
        this.listeners = {};
    }
    /**
     * Add an event listener.
     *
     * @param event - Event names (comma separated) or '*' for any event.
     * @param callback - Called any time even triggers.
     * @returns Unregistration function.
     */
    on(event, callback) {
        const names = event.split(",").map((x) => x.trim());
        for (const name of names) {
            this.listeners[name] = this.listeners[name] || [];
            this.listeners[name].push(callback);
        }
        return () => {
            for (const name of names) {
                this.listeners[name] = this.listeners[name].filter((fn) => {
                    return fn !== callback;
                });
            }
        };
    }
    /**
     * Add a onetime event listener. The listener will automatically be removed
     * after being triggered once.
     *
     * @param event - Event names (comma separated) or '*' for any event.
     * @param callback - Called any time even triggers.
     * @returns Unregistration function.
     */
    once(event, callback) {
        const deregister = this.on(event, (event, data) => {
            callback(event, data);
            deregister();
        });
        return deregister;
    }
    /**
     * Trigger event causing all listeners to be called.
     *
     * @param event - Event name.
     * @param data - Event data.
     */
    /* eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types */
    trigger(event, data) {
        var _a, _b;
        const callbacks = [...((_a = this.listeners[event]) !== null && _a !== void 0 ? _a : []), ...((_b = this.listeners["*"]) !== null && _b !== void 0 ? _b : [])];
        callbacks.forEach((listener) => {
            listener.call(null, event, data);
        });
    }
}

const regexp = /<!(?:--)?\[(.*?)\](?:--)?>/g;
function* parseConditionalComment(comment, commentLocation) {
    let match;
    while ((match = regexp.exec(comment)) !== null) {
        const expression = match[1];
        const begin = match.index;
        const end = begin + match[0].length;
        const location = sliceLocation(commentLocation, begin, end, comment);
        yield {
            expression,
            location,
        };
    }
}

class ParserError extends Error {
    constructor(location, message) {
        super(message);
        this.location = location;
    }
}

/**
 * Parse HTML document into a DOM tree.
 */
class Parser {
    /**
     * Create a new parser instance.
     *
     * @param config - Configuration
     */
    constructor(config) {
        this.event = new EventHandler();
        this.dom = null;
        this.metaTable = config.getMetaTable();
    }
    /**
     * Parse HTML markup.
     *
     * @param source - HTML markup.
     * @returns DOM tree representing the HTML markup.
     */
    // eslint-disable-next-line complexity
    parseHtml(source) {
        var _a, _b, _c, _d;
        if (typeof source === "string") {
            source = {
                data: source,
                filename: "inline",
                line: 1,
                column: 1,
                offset: 0,
            };
        }
        /* reset DOM in case there are multiple calls in the same session */
        this.dom = new DOMTree({
            filename: (_a = source.filename) !== null && _a !== void 0 ? _a : "",
            offset: (_b = source.offset) !== null && _b !== void 0 ? _b : 0,
            line: (_c = source.line) !== null && _c !== void 0 ? _c : 1,
            column: (_d = source.column) !== null && _d !== void 0 ? _d : 1,
            size: 0,
        });
        /* trigger any rules waiting for DOM load event */
        this.trigger("dom:load", {
            source,
            location: null,
        });
        const lexer = new Lexer();
        const tokenStream = lexer.tokenize(source);
        /* consume all tokens from the stream */
        let it = this.next(tokenStream);
        while (!it.done) {
            const token = it.value;
            switch (token.type) {
                case TokenType.UNICODE_BOM:
                    /* ignore */
                    break;
                case TokenType.TAG_OPEN:
                    this.consumeTag(source, token, tokenStream);
                    break;
                case TokenType.WHITESPACE:
                    this.trigger("whitespace", {
                        text: token.data[0],
                        location: token.location,
                    });
                    this.appendText(token.data[0], token.location);
                    break;
                case TokenType.DIRECTIVE:
                    this.consumeDirective(token);
                    break;
                case TokenType.CONDITIONAL:
                    this.trigger("conditional", {
                        condition: token.data[1],
                        location: token.location,
                    });
                    break;
                case TokenType.COMMENT:
                    this.consumeComment(token);
                    break;
                case TokenType.DOCTYPE_OPEN:
                    this.consumeDoctype(token, tokenStream);
                    break;
                case TokenType.TEXT:
                case TokenType.TEMPLATING:
                    this.appendText(token.data, token.location);
                    break;
                case TokenType.EOF:
                    this.closeTree(source, token.location);
                    break;
            }
            it = this.next(tokenStream);
        }
        /* resolve any dynamic meta element properties */
        this.dom.resolveMeta(this.metaTable);
        /* trigger any rules waiting for DOM ready */
        this.trigger("dom:ready", {
            document: this.dom,
            source,
            /* disable location for this event so rules can use implicit node location
             * instead */
            location: null,
        });
        return this.dom.root;
    }
    /**
     * Detect optional end tag.
     *
     * Some tags have optional end tags (e.g. <ul><li>foo<li>bar</ul> is
     * valid). The parser handles this by checking if the element on top of the
     * stack when is allowed to omit.
     */
    closeOptional(token) {
        /* if the element doesn't have metadata it cannot have optional end
         * tags. Period. */
        const active = this.dom.getActive();
        if (!(active.meta && active.meta.implicitClosed)) {
            return false;
        }
        const tagName = token.data[2];
        const open = !token.data[1];
        const meta = active.meta.implicitClosed;
        if (open) {
            /* a new element is opened, check if the new element should close the
             * previous */
            return meta.includes(tagName);
        }
        else {
            /* if we are explicitly closing the active element, ignore implicit */
            if (active.is(tagName)) {
                return false;
            }
            /* the parent element is closed, check if the active element would be
             * implicitly closed when parent is. */
            return Boolean(active.parent && active.parent.is(tagName) && meta.includes(active.tagName));
        }
    }
    /* eslint-disable-next-line complexity, sonarjs/cognitive-complexity */
    consumeTag(source, startToken, tokenStream) {
        const tokens = Array.from(this.consumeUntil(tokenStream, TokenType.TAG_CLOSE, startToken.location));
        const endToken = tokens.slice(-1)[0];
        const closeOptional = this.closeOptional(startToken);
        const parent = closeOptional ? this.dom.getActive().parent : this.dom.getActive();
        const node = HtmlElement.fromTokens(startToken, endToken, parent, this.metaTable);
        const isStartTag = !startToken.data[1];
        const isClosing = !isStartTag || node.closed !== NodeClosed.Open;
        const isForeign = node.meta && node.meta.foreign;
        /* if the previous tag to be implicitly closed by the current tag we close
         * it and pop it from the stack before continuing processing this tag */
        if (closeOptional) {
            const active = this.dom.getActive();
            active.closed = NodeClosed.ImplicitClosed;
            this.closeElement(source, node, active, startToken.location);
            this.dom.popActive();
        }
        if (isStartTag) {
            this.dom.pushActive(node);
            this.trigger("tag:start", {
                target: node,
                location: startToken.location,
            });
        }
        for (let i = 0; i < tokens.length; i++) {
            const token = tokens[i];
            switch (token.type) {
                case TokenType.WHITESPACE:
                    break;
                case TokenType.ATTR_NAME:
                    this.consumeAttribute(source, node, token, tokens[i + 1]);
                    break;
            }
        }
        /* emit tag:ready unless this is a end tag */
        if (isStartTag) {
            this.trigger("tag:ready", {
                target: node,
                location: endToken.location,
            });
        }
        if (isClosing) {
            const active = this.dom.getActive();
            /* if this is not an open tag it is a close tag and thus we force it to be
             * one, in case it is detected as void */
            if (!isStartTag) {
                node.closed = NodeClosed.EndTag;
            }
            this.closeElement(source, node, active, endToken.location);
            /* if this element is closed with an end tag but is would it will not be
             * closed again (it is already closed automatically since it is
             * void). Closing again will have side-effects as it will close the parent
             * and cause a mess later. */
            const voidClosed = !isStartTag && node.voidElement;
            if (!voidClosed) {
                this.dom.popActive();
            }
        }
        else if (isForeign) {
            /* consume the body of the foreign element so it won't be part of the
             * document (only the root foreign element is).  */
            this.discardForeignBody(source, node.tagName, tokenStream, startToken.location);
        }
    }
    closeElement(source, node, active, location) {
        /* call processElement hook */
        this.processElement(active, source);
        /* trigger event for the closing of the element (the </> tag)*/
        const event = {
            target: node,
            previous: active,
            location,
        };
        this.trigger("tag:end", event);
        /* trigger event for for an element being fully constructed. Special care
         * for void elements explicit closed <input></input> */
        if (active && !active.isRootElement()) {
            this.trigger("element:ready", {
                target: active,
                location: active.location,
            });
        }
    }
    processElement(node, source) {
        /* enable cache on node now that it is fully constructed */
        node.cacheEnable();
        if (source.hooks && source.hooks.processElement) {
            const processElement = source.hooks.processElement;
            const metaTable = this.metaTable;
            const context = {
                getMetaFor(tagName) {
                    return metaTable.getMetaFor(tagName);
                },
            };
            processElement.call(context, node);
        }
    }
    /**
     * Discard tokens until the end tag for the foreign element is found.
     */
    discardForeignBody(source, foreignTagName, tokenStream, errorLocation) {
        /* consume elements until the end tag for this foreign element is found */
        let nested = 1;
        let startToken;
        let endToken;
        do {
            /* search for tags */
            const tokens = Array.from(this.consumeUntil(tokenStream, TokenType.TAG_OPEN, errorLocation));
            const [last] = tokens.slice(-1);
            const [, tagClosed, tagName] = last.data;
            /* keep going unless the new tag matches the foreign root element */
            if (tagName !== foreignTagName) {
                continue;
            }
            /* locate end token and determine if this is a self-closed tag */
            const endTokens = Array.from(this.consumeUntil(tokenStream, TokenType.TAG_CLOSE, last.location));
            endToken = endTokens.slice(-1)[0];
            const selfClosed = endToken.data[0] === "/>";
            /* since foreign element may be nested keep a count for the number of
             * opened/closed elements */
            if (tagClosed) {
                startToken = last;
                nested--;
            }
            else if (!selfClosed) {
                nested++;
            }
        } while (nested > 0);
        /* istanbul ignore next: this should never happen because `consumeUntil`
         * would have thrown errors however typescript does not know that */
        if (!startToken || !endToken) {
            return;
        }
        const active = this.dom.getActive();
        const node = HtmlElement.fromTokens(startToken, endToken, active, this.metaTable);
        this.closeElement(source, node, active, endToken.location);
        this.dom.popActive();
    }
    consumeAttribute(source, node, token, next) {
        const keyLocation = this.getAttributeKeyLocation(token);
        const valueLocation = this.getAttributeValueLocation(next);
        const location = this.getAttributeLocation(token, next);
        const haveValue = next && next.type === TokenType.ATTR_VALUE;
        const attrData = {
            key: token.data[1],
            value: null,
            quote: null,
        };
        if (next && haveValue) {
            const [, , value, quote] = next.data;
            attrData.value = value !== null && value !== void 0 ? value : null;
            attrData.quote = quote !== null && quote !== void 0 ? quote : null;
        }
        /* get callback to process attributes, default is to just return attribute
         * data right away but a transformer may override it to allow aliasing
         * attributes, e.g ng-attr-foo or v-bind:foo */
        let processAttribute = (attr) => [attr];
        if (source.hooks && source.hooks.processAttribute) {
            processAttribute = source.hooks.processAttribute;
        }
        /* handle deprecated callbacks */
        let iterator;
        const legacy = processAttribute.call({}, attrData);
        if (typeof legacy[Symbol.iterator] !== "function") {
            /* AttributeData */
            iterator = [attrData];
        }
        else {
            /* Iterable<AttributeData> */
            iterator = legacy;
        }
        /* process attribute(s) */
        for (const attr of iterator) {
            const event = {
                target: node,
                key: attr.key,
                value: attr.value,
                quote: attr.quote,
                originalAttribute: attr.originalAttribute,
                location,
                keyLocation,
                valueLocation,
            };
            this.trigger("attr", event);
            node.setAttribute(attr.key, attr.value, keyLocation, valueLocation, attr.originalAttribute);
        }
    }
    /**
     * Takes attribute key token an returns location.
     */
    getAttributeKeyLocation(token) {
        return token.location;
    }
    /**
     * Take attribute value token and return a new location referring to only the
     * value.
     *
     * foo="bar"    foo='bar'    foo=bar    foo      foo=""
     *      ^^^          ^^^         ^^^    (null)   (null)
     */
    getAttributeValueLocation(token) {
        if (!token || token.type !== TokenType.ATTR_VALUE || token.data[2] === "") {
            return null;
        }
        const quote = token.data[3];
        if (quote) {
            return sliceLocation(token.location, 2, -1);
        }
        else {
            return sliceLocation(token.location, 1);
        }
    }
    /**
     * Take attribute key and value token an returns a new location referring to
     * an aggregate location covering key, quotes if present and value.
     */
    getAttributeLocation(key, value) {
        var _a;
        const begin = key.location;
        const end = value && value.type === TokenType.ATTR_VALUE ? value.location : undefined;
        return {
            filename: begin.filename,
            line: begin.line,
            column: begin.column,
            size: begin.size + ((_a = end === null || end === void 0 ? void 0 : end.size) !== null && _a !== void 0 ? _a : 0),
            offset: begin.offset,
        };
    }
    consumeDirective(token) {
        const directive = token.data[1];
        const match = directive.match(/^([a-zA-Z0-9-]+)\s*(.*?)(?:\s*:\s*(.*))?$/);
        if (!match) {
            throw new Error(`Failed to parse directive "${directive}"`);
        }
        const [, action, data, comment] = match;
        this.trigger("directive", {
            action,
            data,
            comment: comment || "",
            location: token.location,
        });
    }
    /**
     * Consumes comment token.
     *
     * Tries to find IE conditional comments and emits conditional token if found.
     */
    consumeComment(token) {
        const comment = token.data[0];
        for (const conditional of parseConditionalComment(comment, token.location)) {
            this.trigger("conditional", {
                condition: conditional.expression,
                location: conditional.location,
            });
        }
    }
    /**
     * Consumes doctype tokens. Emits doctype event.
     */
    consumeDoctype(startToken, tokenStream) {
        const tokens = Array.from(this.consumeUntil(tokenStream, TokenType.DOCTYPE_CLOSE, startToken.location));
        const doctype = tokens[0]; /* first token is the doctype, second is the closing ">" */
        const value = doctype.data[0];
        this.dom.doctype = value;
        this.trigger("doctype", {
            tag: startToken.data[1],
            value,
            valueLocation: tokens[0].location,
            location: startToken.location,
        });
    }
    /**
     * Return a list of tokens found until the expected token was found.
     *
     * @param errorLocation - What location to use if an error occurs
     */
    *consumeUntil(tokenStream, search, errorLocation) {
        let it = this.next(tokenStream);
        while (!it.done) {
            const token = it.value;
            yield token;
            if (token.type === search)
                return;
            it = this.next(tokenStream);
        }
        throw new ParserError(errorLocation, `stream ended before ${TokenType[search]} token was found`);
    }
    next(tokenStream) {
        const it = tokenStream.next();
        if (!it.done) {
            const token = it.value;
            this.trigger("token", {
                location: token.location,
                type: token.type,
                data: token.data ? Array.from(token.data) : undefined,
            });
        }
        return it;
    }
    on(event, listener) {
        return this.event.on(event, listener);
    }
    once(event, listener) {
        return this.event.once(event, listener);
    }
    /**
     * Defer execution. Will call function sometime later.
     *
     * @param cb - Callback to execute later.
     */
    defer(cb) {
        this.event.once("*", cb);
    }
    trigger(event, data) {
        if (typeof data.location === "undefined") {
            throw new Error("Triggered event must contain location");
        }
        this.event.trigger(event, data);
    }
    /**
     * @internal
     */
    getEventHandler() {
        return this.event;
    }
    /**
     * Appends a text node to the current element on the stack.
     */
    appendText(text, location) {
        this.dom.getActive().appendText(text, location);
    }
    /**
     * Trigger close events for any still open elements.
     */
    closeTree(source, location) {
        let active;
        while ((active = this.dom.getActive()) && !active.isRootElement()) {
            this.closeElement(source, null, active, location);
            this.dom.popActive();
        }
    }
}

class Reporter {
    constructor() {
        this.result = {};
    }
    /**
     * Merge two or more reports into a single one.
     */
    static merge(reports) {
        const valid = reports.every((report) => report.valid);
        const merged = {};
        reports.forEach((report) => {
            report.results.forEach((result) => {
                const key = result.filePath;
                if (key in merged) {
                    merged[key].messages = [...merged[key].messages, ...result.messages];
                }
                else {
                    merged[key] = Object.assign({}, result);
                }
            });
        });
        const results = Object.values(merged).map((result) => {
            /* recalculate error- and warning-count */
            result.errorCount = countErrors(result.messages);
            result.warningCount = countWarnings(result.messages);
            return result;
        });
        return {
            valid,
            results,
            errorCount: sumErrors(results),
            warningCount: sumWarnings(results),
        };
    }
    add(rule, message, severity, node, location, context) {
        var _a;
        if (!(location.filename in this.result)) {
            this.result[location.filename] = [];
        }
        this.result[location.filename].push({
            ruleId: rule.name,
            ruleUrl: (_a = rule.documentation(context)) === null || _a === void 0 ? void 0 : _a.url,
            severity,
            message,
            offset: location.offset,
            line: location.line,
            column: location.column,
            size: location.size || 0,
            selector: node ? node.generateSelector() : null,
            context,
        });
    }
    addManual(filename, message) {
        if (!(filename in this.result)) {
            this.result[filename] = [];
        }
        this.result[filename].push(message);
    }
    save(sources) {
        const report = {
            valid: this.isValid(),
            results: Object.keys(this.result).map((filePath) => {
                const messages = Array.from(this.result[filePath]).sort(messageSort);
                const source = (sources || []).find((source) => { var _a; return filePath === ((_a = source.filename) !== null && _a !== void 0 ? _a : ""); });
                return {
                    filePath,
                    messages,
                    errorCount: countErrors(messages),
                    warningCount: countWarnings(messages),
                    source: source ? source.originalData || source.data : null,
                };
            }),
            errorCount: 0,
            warningCount: 0,
        };
        report.errorCount = sumErrors(report.results);
        report.warningCount = sumWarnings(report.results);
        return report;
    }
    isValid() {
        const numErrors = Object.values(this.result).reduce((sum, messages) => {
            return sum + countErrors(messages);
        }, 0);
        return numErrors === 0;
    }
}
function countErrors(messages) {
    return messages.filter((m) => m.severity === Severity.ERROR).length;
}
function countWarnings(messages) {
    return messages.filter((m) => m.severity === Severity.WARN).length;
}
function sumErrors(results) {
    return results.reduce((sum, result) => {
        return sum + result.errorCount;
    }, 0);
}
function sumWarnings(results) {
    return results.reduce((sum, result) => {
        return sum + result.warningCount;
    }, 0);
}
function messageSort(a, b) {
    if (a.line < b.line) {
        return -1;
    }
    if (a.line > b.line) {
        return 1;
    }
    if (a.column < b.column) {
        return -1;
    }
    if (a.column > b.column) {
        return 1;
    }
    return 0;
}

class Engine {
    constructor(config, configData, ParserClass) {
        this.report = new Reporter();
        this.configData = configData;
        this.config = config;
        this.ParserClass = ParserClass;
        /* initialize plugins and rules */
        const result = this.initPlugins(this.config);
        this.availableRules = Object.assign(Object.assign({}, bundledRules), result.availableRules);
    }
    /**
     * Lint sources and return report
     *
     * @param src - Parsed source.
     * @returns Report output.
     */
    lint(sources) {
        for (const source of sources) {
            /* create parser for source */
            const parser = this.instantiateParser();
            /* setup plugins and rules */
            const { rules } = this.setupPlugins(source, this.config, parser);
            /* create a faux location at the start of the stream for the next events */
            const location = {
                filename: source.filename,
                line: 1,
                column: 1,
                offset: 0,
                size: 1,
            };
            /* trigger configuration ready event */
            const configEvent = {
                location,
                config: this.configData,
                rules,
            };
            parser.trigger("config:ready", configEvent);
            /* trigger source ready event */
            /* eslint-disable-next-line @typescript-eslint/no-unused-vars -- object destructured on purpose to remove property */
            const sourceData = __rest(source, ["hooks"]);
            const sourceEvent = {
                location,
                source: sourceData,
            };
            parser.trigger("source:ready", sourceEvent);
            /* setup directive handling */
            parser.on("directive", (_, event) => {
                this.processDirective(event, parser, rules);
            });
            /* parse token stream */
            try {
                parser.parseHtml(source);
            }
            catch (e) {
                if (e instanceof InvalidTokenError || e instanceof ParserError) {
                    this.reportError("parser-error", e.message, e.location);
                }
                else {
                    throw e;
                }
            }
        }
        /* generate results from report */
        return this.report.save(sources);
    }
    /**
     * Returns a list of all events generated while parsing the source.
     *
     * For verbosity, token events are ignored (use [[dumpTokens]] to inspect
     * token stream).
     */
    dumpEvents(source) {
        const parser = this.instantiateParser();
        const lines = [];
        parser.on("*", (event, data) => {
            /* ignore token events as it becomes to verbose */
            if (event === "token") {
                return;
            }
            lines.push({ event, data });
        });
        source.forEach((src) => parser.parseHtml(src));
        return lines;
    }
    dumpTokens(source) {
        const lexer = new Lexer();
        const lines = [];
        for (const src of source) {
            for (const token of lexer.tokenize(src)) {
                const data = token.data ? token.data[0] : null;
                lines.push({
                    token: TokenType[token.type],
                    data,
                    location: `${token.location.filename}:${token.location.line}:${token.location.column}`,
                });
            }
        }
        return lines;
    }
    dumpTree(source) {
        /* @todo handle dumping each tree */
        const parser = this.instantiateParser();
        const document = parser.parseHtml(source[0]);
        const lines = [];
        function decoration(node) {
            let output = "";
            if (node.hasAttribute("id")) {
                output += `#${node.id}`;
            }
            if (node.hasAttribute("class")) {
                output += `.${node.classList.join(".")}`;
            }
            return output;
        }
        function writeNode(node, level, sibling) {
            if (node.parent) {
                const indent = "  ".repeat(level - 1);
                const l = node.childElements.length > 0 ? "┬" : "─";
                const b = sibling < node.parent.childElements.length - 1 ? "├" : "└";
                lines.push(`${indent}${b}─${l} ${node.tagName}${decoration(node)}`);
            }
            else {
                lines.push("(root)");
            }
            node.childElements.forEach((child, index) => writeNode(child, level + 1, index));
        }
        writeNode(document, 0, 0);
        return lines;
    }
    /**
     * Get rule documentation.
     */
    getRuleDocumentation(ruleId, context // eslint-disable-line @typescript-eslint/explicit-module-boundary-types
    ) {
        const rules = this.config.getRules();
        if (rules.has(ruleId)) {
            const [, options] = rules.get(ruleId);
            const rule = this.instantiateRule(ruleId, options);
            return rule.documentation(context);
        }
        else {
            return null;
        }
    }
    /**
     * Create a new parser instance with the current configuration.
     *
     * @internal
     */
    instantiateParser() {
        return new this.ParserClass(this.config);
    }
    processDirective(event, parser, allRules) {
        const rules = event.data
            .split(",")
            .map((name) => name.trim())
            .map((name) => allRules[name])
            .filter((rule) => rule); /* filter out missing rules */
        switch (event.action) {
            case "enable":
                this.processEnableDirective(rules, parser);
                break;
            case "disable":
                this.processDisableDirective(rules, parser);
                break;
            case "disable-block":
                this.processDisableBlockDirective(rules, parser);
                break;
            case "disable-next":
                this.processDisableNextDirective(rules, parser);
                break;
            default:
                this.reportError("parser-error", `Unknown directive "${event.action}"`, event.location);
                break;
        }
    }
    processEnableDirective(rules, parser) {
        for (const rule of rules) {
            rule.setEnabled(true);
            if (rule.getSeverity() === Severity.DISABLED) {
                rule.setServerity(Severity.ERROR);
            }
        }
        /* enable rules on node */
        parser.on("tag:start", (event, data) => {
            data.target.enableRules(rules.map((rule) => rule.name));
        });
    }
    processDisableDirective(rules, parser) {
        for (const rule of rules) {
            rule.setEnabled(false);
        }
        /* disable rules on node */
        parser.on("tag:start", (event, data) => {
            data.target.disableRules(rules.map((rule) => rule.name));
        });
    }
    processDisableBlockDirective(rules, parser) {
        let directiveBlock = null;
        for (const rule of rules) {
            rule.setEnabled(false);
        }
        const unregisterOpen = parser.on("tag:start", (event, data) => {
            var _a, _b;
            /* wait for a tag to open and find the current block by using its parent */
            if (directiveBlock === null) {
                directiveBlock = (_b = (_a = data.target.parent) === null || _a === void 0 ? void 0 : _a.unique) !== null && _b !== void 0 ? _b : null;
            }
            /* disable rules directly on the node so it will be recorded for later,
             * more specifically when using the domtree to trigger errors */
            data.target.disableRules(rules.map((rule) => rule.name));
        });
        const unregisterClose = parser.on("tag:end", (event, data) => {
            /* if the directive is the last thing in a block no id would be set */
            const lastNode = directiveBlock === null;
            /* test if the block is being closed by checking the parent of the block
             * element is being closed */
            const parentClosed = directiveBlock === data.previous.unique;
            /* remove listeners and restore state */
            if (lastNode || parentClosed) {
                unregisterClose();
                unregisterOpen();
                for (const rule of rules) {
                    rule.setEnabled(true);
                }
            }
        });
    }
    processDisableNextDirective(rules, parser) {
        for (const rule of rules) {
            rule.setEnabled(false);
        }
        /* disable rules directly on the node so it will be recorded for later,
         * more specifically when using the domtree to trigger errors */
        const unregister = parser.on("tag:start", (event, data) => {
            data.target.disableRules(rules.map((rule) => rule.name));
        });
        /* disable directive after next event occurs */
        parser.once("tag:ready, tag:end, attr", () => {
            unregister();
            parser.defer(() => {
                for (const rule of rules) {
                    rule.setEnabled(true);
                }
            });
        });
    }
    /*
     * Initialize all plugins. This should only be done once for all sessions.
     */
    initPlugins(config) {
        for (const plugin of config.getPlugins()) {
            if (plugin.init) {
                plugin.init();
            }
        }
        return {
            availableRules: this.initRules(config),
        };
    }
    /**
     * Initializes all rules from plugins and returns an object with a mapping
     * between rule name and its constructor.
     */
    initRules(config) {
        const availableRules = {};
        for (const plugin of config.getPlugins()) {
            for (const [name, rule] of Object.entries(plugin.rules || {})) {
                if (!rule)
                    continue;
                availableRules[name] = rule;
            }
        }
        return availableRules;
    }
    /**
     * Setup all plugins for this session.
     */
    setupPlugins(source, config, parser) {
        const eventHandler = parser.getEventHandler();
        for (const plugin of config.getPlugins()) {
            if (plugin.setup) {
                plugin.setup(source, eventHandler);
            }
        }
        return {
            rules: this.setupRules(config, parser),
        };
    }
    /**
     * Load and setup all rules for current configuration.
     */
    setupRules(config, parser) {
        const rules = {};
        for (const [ruleId, [severity, options]] of config.getRules().entries()) {
            rules[ruleId] = this.loadRule(ruleId, config, severity, options, parser, this.report);
        }
        return rules;
    }
    /**
     * Load and setup a rule using current config.
     */
    loadRule(ruleId, config, severity, options, parser, report) {
        const meta = config.getMetaTable();
        const rule = this.instantiateRule(ruleId, options);
        rule.name = ruleId;
        rule.init(parser, report, severity, meta);
        /* call setup callback if present */
        if (rule.setup) {
            rule.setup();
        }
        return rule;
    }
    instantiateRule(name, options) {
        if (this.availableRules[name]) {
            const RuleConstructor = this.availableRules[name];
            return new RuleConstructor(options);
        }
        else {
            return this.missingRule(name);
        }
    }
    missingRule(name) {
        return new (class MissingRule extends Rule {
            setup() {
                this.on("dom:load", () => {
                    this.report(null, `Definition for rule '${name}' was not found`);
                });
            }
        })();
    }
    reportError(ruleId, message, location) {
        this.report.addManual(location.filename, {
            ruleId,
            severity: Severity.ERROR,
            message,
            offset: location.offset,
            line: location.line,
            column: location.column,
            size: location.size || 0,
            selector: null,
        });
    }
}

/**
 * @internal
 */
function findConfigurationFiles(directory) {
    return ["json", "cjs", "js"]
        .map((extension) => path.join(directory, `.htmlvalidate.${extension}`))
        .filter((filePath) => fs.existsSync(filePath));
}
/**
 * Loads configuration by traversing filesystem.
 *
 * Configuration is read from three sources and in the following order:
 *
 * 1. Global configuration passed to constructor.
 * 2. Configuration files found when traversing the directory structure.
 * 3. Override passed to this function.
 *
 * The following configuration filenames are searched:
 *
 * - `.htmlvalidate.json`
 * - `.htmlvalidate.js`
 * - `.htmlvalidate.cjs`
 *
 * Global configuration is used when no configuration file is found. The
 * result is always merged with override if present.
 *
 * The `root` property set to `true` affects the configuration as following:
 *
 * 1. If set in override the override is returned as-is.
 * 2. If set in the global config the override is merged into global and
 * returned. No configuration files are searched.
 * 3. Setting `root` in configuration file only stops directory traversal.
 */
class FileSystemConfigLoader extends ConfigLoader {
    /**
     * @param config - Global configuration
     * @param configFactory - Optional configuration factory
     */
    constructor(config, configFactory = Config) {
        super(config, configFactory);
        this.cache = new Map();
    }
    /**
     * Get configuration for given filename.
     *
     * @param filename - Filename to get configuration for.
     * @param configOverride - Configuration to merge final result with.
     */
    getConfigFor(filename, configOverride) {
        /* special case when the overridden configuration is marked as root, should
         * not try to load any more configuration files */
        const override = this.loadFromObject(configOverride || {});
        if (override.isRootFound()) {
            override.init();
            return override;
        }
        /* special case when the global configuration is marked as root, should not
         * try to load and more configuration files */
        if (this.globalConfig.isRootFound()) {
            const merged = this.globalConfig.merge(override);
            merged.init();
            return merged;
        }
        const config = this.fromFilename(filename);
        const merged = config ? config.merge(override) : this.globalConfig.merge(override);
        merged.init();
        return merged;
    }
    /**
     * Flush configuration cache.
     *
     * @param filename - If given only the cache for that file is flushed.
     */
    flushCache(filename) {
        if (filename) {
            this.cache.delete(filename);
        }
        else {
            this.cache.clear();
        }
    }
    /**
     * Load raw configuration from directory traversal.
     *
     * This configuration is not merged with global configuration and may return
     * `null` if no configuration files are found.
     */
    fromFilename(filename) {
        var _a;
        if (filename === "inline") {
            return null;
        }
        if (this.cache.has(filename)) {
            return (_a = this.cache.get(filename)) !== null && _a !== void 0 ? _a : null;
        }
        let found = false;
        let current = path.resolve(path.dirname(filename));
        let config = this.empty();
        // eslint-disable-next-line no-constant-condition
        while (true) {
            /* search configuration files in current directory */
            for (const configFile of findConfigurationFiles(current)) {
                const local = this.loadFromFile(configFile);
                found = true;
                config = local.merge(config);
            }
            /* stop if a configuration with "root" is set to true */
            if (config.isRootFound()) {
                break;
            }
            /* get the parent directory */
            const child = current;
            current = path.dirname(current);
            /* stop if this is the root directory */
            if (current === child) {
                break;
            }
        }
        /* no config was found by loader, return null and let caller decide what to do */
        if (!found) {
            this.cache.set(filename, null);
            return null;
        }
        this.cache.set(filename, config);
        return config;
    }
    defaultConfig() {
        return this.configFactory.defaultConfig();
    }
}

function isSourceHooks(value) {
    if (!value || typeof value === "string") {
        return false;
    }
    return Boolean(value.processAttribute || value.processElement);
}
function isConfigData(value) {
    if (!value || typeof value === "string") {
        return false;
    }
    return !(value.processAttribute || value.processElement);
}
/**
 * Primary API for using HTML-validate.
 *
 * Provides high-level abstractions for common operations.
 */
class HtmlValidate {
    constructor(arg) {
        const [loader, config] = arg instanceof ConfigLoader ? [arg, undefined] : [undefined, arg];
        this.configLoader = loader !== null && loader !== void 0 ? loader : new FileSystemConfigLoader(config);
    }
    validateString(str, arg1, arg2, arg3) {
        const filename = typeof arg1 === "string" ? arg1 : "inline";
        const options = isConfigData(arg1) ? arg1 : isConfigData(arg2) ? arg2 : undefined;
        const hooks = isSourceHooks(arg1) ? arg1 : isSourceHooks(arg2) ? arg2 : arg3;
        const source = {
            data: str,
            filename,
            line: 1,
            column: 1,
            offset: 0,
            hooks,
        };
        return this.validateSource(source, options);
    }
    /**
     * Parse and validate HTML from [[Source]].
     *
     * @public
     * @param input - Source to parse.
     * @returns Report output.
     */
    validateSource(input, configOverride) {
        const config = this.getConfigFor(input.filename, configOverride);
        const resolved = config.resolve();
        const source = resolved.transformSource(input);
        const engine = new Engine(resolved, config.get(), Parser);
        return engine.lint(source);
    }
    /**
     * Parse and validate HTML from file.
     *
     * @public
     * @param filename - Filename to read and parse.
     * @returns Report output.
     */
    validateFile(filename) {
        const config = this.getConfigFor(filename);
        const resolved = config.resolve();
        const source = resolved.transformFilename(filename);
        const engine = new Engine(resolved, config.get(), Parser);
        return engine.lint(source);
    }
    /**
     * Parse and validate HTML from multiple files. Result is merged together to a
     * single report.
     *
     * @param filenames - Filenames to read and parse.
     * @returns Report output.
     */
    validateMultipleFiles(filenames) {
        return Reporter.merge(filenames.map((filename) => this.validateFile(filename)));
    }
    /**
     * Returns true if the given filename can be validated.
     *
     * A file is considered to be validatable if the extension is `.html` or if a
     * transformer matches the filename.
     *
     * This is mostly useful for tooling to determine whenever to validate the
     * file or not. CLI tools will run on all the given files anyway.
     */
    canValidate(filename) {
        /* .html is always supported */
        const extension = path.extname(filename).toLowerCase();
        if (extension === ".html") {
            return true;
        }
        /* test if there is a matching transformer */
        const config = this.getConfigFor(filename);
        const resolved = config.resolve();
        return resolved.canTransform(filename);
    }
    /**
     * Tokenize filename and output all tokens.
     *
     * Using CLI this is enabled with `--dump-tokens`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to tokenize.
     */
    dumpTokens(filename) {
        const config = this.getConfigFor(filename);
        const resolved = config.resolve();
        const source = resolved.transformFilename(filename);
        const engine = new Engine(resolved, config.get(), Parser);
        return engine.dumpTokens(source);
    }
    /**
     * Parse filename and output all events.
     *
     * Using CLI this is enabled with `--dump-events`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to dump events from.
     */
    dumpEvents(filename) {
        const config = this.getConfigFor(filename);
        const resolved = config.resolve();
        const source = resolved.transformFilename(filename);
        const engine = new Engine(resolved, config.get(), Parser);
        return engine.dumpEvents(source);
    }
    /**
     * Parse filename and output DOM tree.
     *
     * Using CLI this is enabled with `--dump-tree`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to dump DOM tree from.
     */
    dumpTree(filename) {
        const config = this.getConfigFor(filename);
        const resolved = config.resolve();
        const source = resolved.transformFilename(filename);
        const engine = new Engine(resolved, config.get(), Parser);
        return engine.dumpTree(source);
    }
    /**
     * Transform filename and output source data.
     *
     * Using CLI this is enabled with `--dump-source`. Mostly useful for
     * debugging.
     *
     * @param filename - Filename to dump source from.
     */
    dumpSource(filename) {
        const config = this.getConfigFor(filename);
        const resolved = config.resolve();
        const sources = resolved.transformFilename(filename);
        return sources.reduce((result, source) => {
            result.push(`Source ${source.filename}@${source.line}:${source.column} (offset: ${source.offset})`);
            if (source.transformedBy) {
                result.push("Transformed by:");
                result = result.concat(source.transformedBy.reverse().map((name) => ` - ${name}`));
            }
            if (source.hooks && Object.keys(source.hooks).length > 0) {
                result.push("Hooks");
                for (const [key, present] of Object.entries(source.hooks)) {
                    if (present) {
                        result.push(` - ${key}`);
                    }
                }
            }
            result.push("---");
            result = result.concat(source.data.split("\n"));
            result.push("---");
            return result;
        }, []);
    }
    /**
     * Get effective configuration schema.
     */
    getConfigurationSchema() {
        return configurationSchema;
    }
    /**
     * Get effective metadata element schema.
     *
     * If a filename is given the configured plugins can extend the
     * schema. Filename must not be an existing file or a filetype normally
     * handled by html-validate but the path will be used when resolving
     * configuration. As a rule-of-thumb, set it to the elements json file.
     */
    getElementsSchema(filename) {
        const config = this.getConfigFor(filename !== null && filename !== void 0 ? filename : "inline");
        const metaTable = config.getMetaTable();
        return metaTable.getJSONSchema();
    }
    /**
     * Get contextual documentation for the given rule.
     *
     * Typical usage:
     *
     * ```js
     * const report = htmlvalidate.validateFile("my-file.html");
     * for (const result of report.results){
     *   const config = htmlvalidate.getConfigFor(result.filePath);
     *   for (const message of result.messages){
     *     const documentation = htmlvalidate.getRuleDocumentation(message.ruleId, config, message.context);
     *     // do something with documentation
     *   }
     * }
     * ```
     *
     * @param ruleId - Rule to get documentation for.
     * @param config - If set it provides more accurate description by using the
     * correct configuration for the file.
     * @param context - If set to `Message.context` some rules can provide
     * contextual details and suggestions.
     */
    getRuleDocumentation(ruleId, config = null, context = null) {
        const c = config || this.getConfigFor("inline");
        const engine = new Engine(c.resolve(), c.get(), Parser);
        return engine.getRuleDocumentation(ruleId, context);
    }
    /**
     * Create a parser configured for given filename.
     *
     * @param source - Source to use.
     */
    getParserFor(source) {
        const config = this.getConfigFor(source.filename);
        return new Parser(config.resolve());
    }
    /**
     * Get configuration for given filename.
     *
     * See [[FileSystemConfigLoader]] for details.
     *
     * @public
     * @param filename - Filename to get configuration for.
     * @param configOverride - Configuration to apply last.
     */
    getConfigFor(filename, configOverride) {
        return this.configLoader.getConfigFor(filename, configOverride);
    }
    /**
     * Flush configuration cache. Clears full cache unless a filename is given.
     *
     * See [[FileSystemConfigLoader]] for details.
     *
     * @public
     * @param filename - If set, only flush cache for given filename.
     */
    flushConfigCache(filename) {
        this.configLoader.flushCache(filename);
    }
}

/**
 * The static configuration loader does not do any per-handle lookup. Only the
 * global or per-call configuration is used.
 *
 * In practice this means no configuration is fetch by traversing the
 * filesystem.
 */
class StaticConfigLoader extends ConfigLoader {
    getConfigFor(handle, configOverride) {
        const override = this.loadFromObject(configOverride || {});
        if (override.isRootFound()) {
            override.init();
            return override;
        }
        const merged = this.globalConfig.merge(override);
        merged.init();
        return merged;
    }
    flushCache() {
        /* do nothing */
    }
    defaultConfig() {
        return this.loadFromObject({
            extends: ["html-validate:recommended"],
            elements: ["html5"],
        });
    }
}

const defaults$1 = {
    silent: false,
    version,
    logger(text) {
        /* eslint-disable-next-line no-console */
        console.error(kleur.red(text));
    },
};
/**
 * Tests if plugin is compatible with html-validate library. Unless the `silent`
 * option is used a warning is displayed on the console.
 *
 * @param name - Name of plugin
 * @param declared - What library versions the plugin support (e.g. declared peerDependencies)
 * @returns - `true` if version is compatible
 */
function compatibilityCheck(name, declared, options) {
    const { silent, version: current, logger } = Object.assign(Object.assign({}, defaults$1), options);
    const valid = semver.satisfies(current, declared);
    if (valid || silent) {
        return valid;
    }
    const text = [
        "-----------------------------------------------------------------------------------------------------",
        `${name} requires html-validate version "${declared}" but current installed version is ${current}`,
        "This is not a supported configuration. Please install a supported version before reporting bugs.",
        "-----------------------------------------------------------------------------------------------------",
    ].join("\n");
    logger(text);
    return false;
}

const ruleIds = new Set(Object.keys(bundledRules));
/**
 * Returns true if given ruleId is an existing builtin rule. It does not handle
 * rules loaded via plugins.
 *
 * Can be used to create forward/backward compatibility by checking if a rule
 * exists to enable/disable it.
 *
 * @param ruleId - Rule id to check
 * @returns `true` if rule exists
 */
function ruleExists(ruleId) {
    return ruleIds.has(ruleId);
}

const entities = {
    ">": "&gt;",
    "<": "&lt;",
    "'": "&apos;",
    '"': "&quot;",
    "&": "&amp;",
};
function xmlescape(src) {
    return src.toString().replace(/[><'"&]/g, (match) => {
        return entities[match];
    });
}
function getMessageType(message) {
    switch (message.severity) {
        case 2:
            return "error";
        case 1:
            return "warning";
        default:
            return "error";
    }
}
function checkstyleFormatter(results) {
    let output = "";
    output += `<?xml version="1.0" encoding="utf-8"?>\n`;
    output += `<checkstyle version="4.3">\n`;
    results.forEach((result) => {
        const messages = result.messages;
        output += `  <file name="${xmlescape(result.filePath)}">\n`;
        messages.forEach((message) => {
            const ruleId = xmlescape(`htmlvalidate.rules.${message.ruleId}`);
            output += "    ";
            output += [
                `<error line="${xmlescape(message.line)}"`,
                `column="${xmlescape(message.column)}"`,
                `severity="${xmlescape(getMessageType(message))}"`,
                `message="${xmlescape(message.message)} (${message.ruleId})"`,
                `source="${ruleId}" />`,
            ].join(" ");
            output += "\n";
        });
        output += "  </file>\n";
    });
    output += "</checkstyle>\n";
    return output;
}
const formatter$4 = checkstyleFormatter;

const defaults = {
    showLink: true,
};
/**
 * Codeframe formatter based on ESLint codeframe.
 */
/**
 * Given a word and a count, append an s if count is not one.
 * @param word - A word in its singular form.
 * @param count - A number controlling whether word should be pluralized.
 * @returns The original word with an s on the end if count is not one.
 */
function pluralize(word, count) {
    return count === 1 ? word : `${word}s`;
}
/**
 * Gets a formatted relative file path from an absolute path and a line/column in the file.
 * @param filePath - The absolute file path to format.
 * @param line - The line from the file to use for formatting.
 * @param column -The column from the file to use for formatting.
 * @returns The formatted file path.
 */
function formatFilePath(filePath, line, column) {
    let relPath = path.relative(process.cwd(), filePath);
    /* istanbul ignore next: safety check from original implementation */
    if (line && column) {
        relPath += `:${line}:${column}`;
    }
    return kleur.green(relPath);
}
function getStartLocation(message) {
    return {
        line: message.line,
        column: message.column,
    };
}
function getEndLocation(message, source) {
    let line = message.line;
    let column = message.column;
    for (let i = 0; i < message.size; i++) {
        if (source.charAt(message.offset + i) === "\n") {
            line++;
            column = 0;
        }
        else {
            column++;
        }
    }
    return { line, column };
}
/**
 * Gets the formatted output for a given message.
 * @param message - The object that represents this message.
 * @param parentResult - The result object that this message belongs to.
 * @returns The formatted output.
 */
function formatMessage(message, parentResult, options) {
    const type = message.severity === 2 ? kleur.red("error") : kleur.yellow("warning");
    const msg = `${kleur.bold(message.message.replace(/([^ ])\.$/, "$1"))}`;
    const ruleId = kleur.dim(`(${message.ruleId})`);
    const filePath = formatFilePath(parentResult.filePath, message.line, message.column);
    const sourceCode = parentResult.source;
    /* istanbul ignore next: safety check from original implementation */
    const firstLine = [
        `${type}:`,
        `${msg}`,
        ruleId ? `${ruleId}` : "",
        sourceCode ? `at ${filePath}:` : `at ${filePath}`,
    ]
        .filter(String)
        .join(" ");
    const result = [firstLine];
    /* istanbul ignore next: safety check from original implementation */
    if (sourceCode) {
        result.push(codeFrameColumns(sourceCode, {
            start: getStartLocation(message),
            end: getEndLocation(message, sourceCode),
        }, { highlightCode: false }));
    }
    if (options.showLink && message.ruleUrl) {
        result.push(`${kleur.bold("Details:")} ${message.ruleUrl}`);
    }
    return result.join("\n");
}
/**
 * Gets the formatted output summary for a given number of errors and warnings.
 * @param errors - The number of errors.
 * @param warnings - The number of warnings.
 * @returns The formatted output summary.
 */
function formatSummary(errors, warnings) {
    const summaryColor = errors > 0 ? "red" : "yellow";
    const summary = [];
    if (errors > 0) {
        summary.push(`${errors} ${pluralize("error", errors)}`);
    }
    if (warnings > 0) {
        summary.push(`${warnings} ${pluralize("warning", warnings)}`);
    }
    return kleur[summaryColor]().bold(`${summary.join(" and ")} found.`);
}
function codeframe(results, options) {
    const merged = Object.assign(Object.assign({}, defaults), options);
    let errors = 0;
    let warnings = 0;
    const resultsWithMessages = results.filter((result) => result.messages.length > 0);
    let output = resultsWithMessages
        .reduce((resultsOutput, result) => {
        const messages = result.messages.map((message) => {
            return `${formatMessage(message, result, merged)}\n\n`;
        });
        errors += result.errorCount;
        warnings += result.warningCount;
        return resultsOutput.concat(messages);
    }, [])
        .join("\n");
    output += "\n";
    output += formatSummary(errors, warnings);
    output += "\n";
    return errors + warnings > 0 ? output : "";
}
const formatter$3 = codeframe;

function jsonFormatter(results) {
    return JSON.stringify(results);
}
const formatter$2 = jsonFormatter;

function linkSummary(results) {
    const urls = results.reduce((result, it) => {
        const urls = it.messages
            .map((error) => error.ruleUrl)
            .filter((error) => Boolean(error));
        return [...result, ...urls];
    }, []);
    const unique = Array.from(new Set(urls));
    if (unique.length === 0) {
        return "";
    }
    const lines = unique.map((url) => `  ${url}\n`);
    return `\n${kleur.bold("More information")}:\n${lines.join("")}\n`;
}
function stylish(results) {
    const errors = stylishImpl(results.map((it) => (Object.assign(Object.assign({}, it), { fixableErrorCount: 0, fixableWarningCount: 0 }))));
    const links = linkSummary(results);
    return `${errors}${links}`;
}
const formatter$1 = stylish;

function textFormatter(results) {
    let output = "";
    let total = 0;
    results.forEach((result) => {
        const messages = result.messages;
        if (messages.length === 0) {
            return;
        }
        total += messages.length;
        output += messages
            .map((message) => {
            let messageType;
            if (message.severity === 2) {
                messageType = "error";
            }
            else {
                messageType = "warning";
            }
            const location = `${result.filePath}:${message.line}:${message.column}`;
            return `${location}: ${messageType} [${message.ruleId}] ${message.message}\n`;
        })
            .join("");
    });
    return total > 0 ? output : "";
}
const formatter = textFormatter;

const availableFormatters = {
    checkstyle: formatter$4,
    codeframe: formatter$3,
    json: formatter$2,
    stylish: formatter$1,
    text: formatter,
};
/**
 * Get formatter function by name.
 *
 * @param name - Name of formatter.
 * @returns Formatter function or null if it doesn't exist.
 */
function getFormatter(name) {
    var _a;
    return (_a = availableFormatters[name]) !== null && _a !== void 0 ? _a : null;
}

export { Config as C, DynamicValue as D, EventHandler as E, FileSystemConfigLoader as F, HtmlValidate as H, MetaTable as M, NodeClosed as N, Parser as P, Rule as R, Severity as S, TextNode as T, UserError as U, ConfigError as a, ConfigLoader as b, StaticConfigLoader as c, HtmlElement as d, SchemaValidationError as e, MetaCopyableProperty as f, Reporter as g, TemplateExtractor as h, getFormatter as i, compatibilityCheck as j, TokenType as k, legacyRequire as l, bugs as m, name as n, presets as p, ruleExists as r, version as v };
//# sourceMappingURL=core.js.map
