import * as jestDiffDefault from 'jest-diff';
import jestDiffDefault__default from 'jest-diff';
import { k as TokenType, H as HtmlValidate } from './core.js';
import deepmerge from 'deepmerge';

/* eslint-disable @typescript-eslint/ban-ts-comment */
/* ignore typing for compatibility so it will seem "impossible" but different version will yield different source */
const diffCandidates = [
    // @ts-ignore
    jestDiffDefault__default === null || jestDiffDefault__default === void 0 ? void 0 : jestDiffDefault__default.diff,
    // @ts-ignore
    jestDiffDefault__default,
    // @ts-ignore
    jestDiffDefault === null || jestDiffDefault === void 0 ? void 0 : jestDiffDefault.diff,
    // @ts-ignore
    jestDiffDefault,
];
const isFunction = (fn) => typeof fn === "function";
/* istanbul ignore next: covered by compatibility tests but not a single pass */
/* @ts-ignore assume one of the candidate matches, there will be a reasonable error later on if not */
const diff = diffCandidates.find(isFunction);

function isThenable(src) {
    return src && typeof src === "object" && typeof src.then === "function";
}

/**
 * Creates a wrapped function based on the passed function.
 *
 * The returned function takes either a `T` or `Promise<T>`. If `T` the result
 * will be synchronous or if `Promise<T>` the result will be asynchronous.
 *
 * In practice this means that if you pass a synchronous object into it you will
 * maintain synchronous code but if you pass an asynchronous object you must
 * await the result.
 */
function diverge(fn) {
    function diverged(actual, ...args) {
        if (isThenable(actual)) {
            return actual.then((resolved) => fn.call(this, resolved, ...args));
        }
        else {
            return fn.call(this, actual, ...args);
        }
    }
    return diverged;
}

/**
 * Takes all messages from all files and flattens to a single array.
 */
function flattenMessages(report) {
    return report.results.reduce((aggregated, result) => {
        return aggregated.concat(result.messages);
    }, []);
}

function toBeValid(report) {
    if (report.valid) {
        return {
            pass: true,
            message: /* istanbul ignore next */ () => "Result should not contain error",
        };
    }
    else {
        const firstError = report.results[0].messages[0];
        return {
            pass: false,
            message: () => `Result should be valid but had error "${firstError.message}"`,
        };
    }
}
var toBeValid$1 = diverge(toBeValid);

function toBeInvalid(report) {
    if (report.valid) {
        return {
            pass: false,
            message: () => "Result should be invalid but had no errors",
        };
    }
    else {
        return {
            pass: true,
            message: /* istanbul ignore next */ () => "Result should not contain error",
        };
    }
}
var toBeInvalid$1 = diverge(toBeInvalid);

/* eslint-disable @typescript-eslint/explicit-module-boundary-types, prefer-template */
function toBeToken(actual, expected) {
    const token = actual.value;
    // istanbul ignore next: TokenMatcher requires "type" property to be set, this is just a failsafe
    if (token.type) {
        token.type = TokenType[token.type];
    }
    // istanbul ignore next: TokenMatcher requires "type" property to be set, this is just a failsafe
    if (expected.type) {
        expected.type = TokenType[expected.type];
    }
    const matcher = expect.objectContaining(expected);
    const pass = this.equals(token, matcher);
    const diffString = diff(matcher, token, { expand: this.expand });
    const message = () => this.utils.matcherHint(".toBeToken") +
        "\n\n" +
        "Expected token to equal:\n" +
        `  ${this.utils.printExpected(matcher)}\n` +
        "Received:\n" +
        `  ${this.utils.printReceived(token)}` +
        /* istanbul ignore next */ (diffString ? `\n\nDifference:\n\n${diffString}` : "");
    return { pass, message };
}
var toBeToken$1 = diverge(toBeToken);

/* eslint-disable @typescript-eslint/ban-ts-comment, prefer-template */
function isMessage(arg) {
    return (arg &&
        (arg.ruleId ||
            arg.severity ||
            arg.message ||
            arg.offset ||
            arg.line ||
            arg.column ||
            arg.size ||
            arg.selector ||
            arg.context));
}
function isConfig(arg) {
    return (arg && (arg.root || arg.extends || arg.elements || arg.plugin || arg.transform || arg.rules));
}
function isString(arg) {
    return typeof arg === "string";
}
function getMarkup(src) {
    // @ts-ignore DOM library not available
    if (typeof HTMLElement !== "undefined" && src instanceof HTMLElement) {
        return src.outerHTML;
    }
    /* istanbul ignore else: prototype only allows string or HTMLElement */
    if (typeof src === "string") {
        return src;
    }
    else {
        throw new Error(`Failed to get markup from "${typeof src}" argument`);
    }
}
function toHTMLValidate(actual, arg0, arg1, arg2) {
    const markup = getMarkup(actual);
    const message = isMessage(arg0) ? arg0 : undefined;
    const config = isConfig(arg0) ? arg0 : isConfig(arg1) ? arg1 : undefined;
    const filename = isString(arg0) ? arg0 : isString(arg1) ? arg1 : arg2;
    return toHTMLValidateImpl.call(this, markup, message, config, filename);
}
function toHTMLValidateImpl(actual, expectedError, userConfig, filename) {
    const defaultConfig = {
        rules: {
            /* jsdom normalizes style so disabling rule when using this matcher or it
             * gets quite noisy when configured with self-closing */
            "void-style": "off",
        },
    };
    const config = deepmerge(defaultConfig, userConfig || {});
    const actualFilename = filename || this.testPath;
    const htmlvalidate = new HtmlValidate();
    const report = htmlvalidate.validateString(actual, actualFilename, config);
    const pass = report.valid;
    if (pass) {
        return { pass, message: () => "HTML is valid when an error was expected" };
    }
    else {
        if (expectedError) {
            const matcher = expect.arrayContaining([expect.objectContaining(expectedError)]);
            const errorPass = this.equals(report.results[0].messages, matcher);
            const diffString = diff(matcher, report.results[0].messages, {
                expand: this.expand,
                aAnnotation: "Expected error",
                bAnnotation: "Actual error",
            });
            const hint = this.utils.matcherHint(".not.toHTMLValidate", undefined, undefined, {
                comment: "expected error",
            });
            const expectedErrorMessage = () => [
                hint,
                "",
                "Expected error to be present:",
                this.utils.printExpected(expectedError),
                /* istanbul ignore next */ diffString ? `\n${diffString}` : "",
            ].join("\n");
            return { pass: !errorPass, message: expectedErrorMessage };
        }
        const errors = report.results[0].messages.map((message) => `  ${message.message} [${message.ruleId}]`);
        return {
            pass,
            message: () => ["Expected HTML to be valid but had the following errors:", ""].concat(errors).join("\n"),
        };
    }
}
var toHTMLValidate$1 = diverge(toHTMLValidate);

/* eslint-disable prefer-template */
function toHaveError(actual, ruleId, message, context // eslint-disable-line @typescript-eslint/explicit-module-boundary-types
) {
    const flattened = flattenMessages(actual);
    const expected = { ruleId, message };
    if (context) {
        expected.context = context;
    }
    const matcher = [expect.objectContaining(expected)];
    const pass = this.equals(flattened, matcher);
    const diffString = diff(matcher, flattened, { expand: this.expand });
    const resultMessage = () => this.utils.matcherHint(".toHaveError") +
        "\n\n" +
        "Expected error to equal:\n" +
        `  ${this.utils.printExpected(matcher)}\n` +
        "Received:\n" +
        `  ${this.utils.printReceived(flattened)}` +
        /* istanbul ignore next */ (diffString ? `\n\nDifference:\n\n${diffString}` : "");
    return { pass, message: resultMessage };
}
var toHaveError$1 = diverge(toHaveError);

/* eslint-disable prefer-template */
function toHaveErrors(report, errors) {
    const flattened = flattenMessages(report);
    const matcher = errors.map((entry) => {
        if (Array.isArray(entry)) {
            const [ruleId, message] = entry;
            return expect.objectContaining({ ruleId, message });
        }
        else {
            return expect.objectContaining(entry);
        }
    });
    const pass = this.equals(flattened, matcher);
    const diffString = diff(matcher, flattened, { expand: this.expand });
    const resultMessage = () => this.utils.matcherHint(".toHaveErrors") +
        "\n\n" +
        "Expected error to equal:\n" +
        `  ${this.utils.printExpected(matcher)}\n` +
        "Received:\n" +
        `  ${this.utils.printReceived(flattened)}` +
        /* istanbul ignore next */ (diffString ? `\n\nDifference:\n\n${diffString}` : "");
    return { pass, message: resultMessage };
}
var toHaveErrors$1 = diverge(toHaveErrors);

expect.extend({
    toBeValid: toBeValid$1,
    toBeInvalid: toBeInvalid$1,
    toHaveError: toHaveError$1,
    toHaveErrors: toHaveErrors$1,
    toBeToken: toBeToken$1,
    toHTMLValidate: toHTMLValidate$1,
});
//# sourceMappingURL=jest-lib.js.map
