package haxe.net;

import haxe.io.Bytes;

class Crypto {
	static public function getSecureRandomBytes(length:Int):Bytes {
        var reason = '';
        try {
            #if flash
                return Bytes.ofData(untyped __global__["flash.crypto.generateRandomBytes"](length));
            #elseif js
                untyped __js__('var Crypto = typeof crypto === "undefined" ? require("crypto") : crypto');
                var bytes:Dynamic = untyped __js__("(Crypto.randomBytes) ? Crypto.randomBytes({0}) : Crypto.getRandomValues(new Uint8Array({0}))", length);
                var out = Bytes.alloc(length);
                for (n in 0 ... length) out.set(n, bytes[n]);
                return out;
            #elseif python
                var out = Bytes.alloc(length);
                var bytes = RandomOs.urandom(length);
                for (n in 0 ... length) out.set(n, bytes[n]);
                return out;
            #elseif java
                return Bytes.ofData(java.security.SecureRandom.getSeed(length));
            #elseif cs
                var out = Bytes.alloc(length);
                var rng = new cs.system.security.cryptography.RNGCryptoServiceProvider();
                rng.GetBytes(out.getData());
                return out;
            #elseif sys
                // https://en.wikipedia.org/wiki//dev/random 
				// The WIKIpedia tells that you canNOT use the device KsecDD just like /dev/urandom or /dev/random!!
                var out = Bytes.alloc(length);
				#if windows
					return  haxe.io.Bytes.ofString(getRandomString(16));
					// This will not work
                    var input = sys.io.File.read("\\Device\\KsecDD");

                #else
                    var input = sys.io.File.read("/dev/urandom");
                #end
                input.readBytes(out, 0, length);
                input.close();
                return out;
            #end
        } catch (e:Dynamic) {
            reason = '$e';
        }
        throw "Can't find a secure source of random bytes. Reason: " + reason;
    }

	
	
	public static function getRandomString(length:Int, ?charactersToUse = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"):String
	{
		var str = "";
		for (i in 0...length){
			str += charactersToUse.charAt( Math.floor((Math.random() *  (Date.now().getTime() % (charactersToUse.length) ) )));
		}
		return str;
	} //getRandomString
	
}