"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const notepack_io_1 = __importDefault(require("notepack.io"));
const timer_1 = __importDefault(require("@gamestdio/timer"));
const events_1 = require("events");
const RemoteClient_1 = require("./presence/RemoteClient");
const SchemaSerializer_1 = require("./serializer/SchemaSerializer");
const Protocol_1 = require("./Protocol");
const Utils_1 = require("./Utils");
const Debug_1 = require("./Debug");
const DEFAULT_PATCH_RATE = 1000 / 20; // 20fps (50ms)
const DEFAULT_SIMULATION_INTERVAL = 1000 / 60; // 60fps (16.66ms)
const DEFAULT_SEAT_RESERVATION_TIME = Number(process.env.COLYSEUS_SEAT_RESERVATION_TIME || 5);
class Room extends events_1.EventEmitter {
    constructor(presence) {
        super();
        this.clock = new timer_1.default();
        this.maxClients = Infinity;
        this.patchRate = DEFAULT_PATCH_RATE;
        this.autoDispose = true;
        this.metadata = null;
        this.clients = [];
        this.remoteClients = {};
        // seat reservation & reconnection
        this.seatReservationTime = DEFAULT_SEAT_RESERVATION_TIME;
        this.reservedSeats = new Set();
        this.reservedSeatTimeouts = {};
        this.reconnections = {};
        this.isDisconnecting = false;
        this._serializer = this._getSerializer();
        this._afterNextPatchBroadcasts = [];
        this._locked = false;
        this._lockedExplicitly = false;
        this._maxClientsReached = false;
        this.presence = presence;
        this.once('dispose', () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield this._dispose();
            }
            catch (e) {
                Debug_1.debugAndPrintError(`onDispose error: ${(e && e.message || e || 'promise rejected')}`);
            }
            this.emit('disconnect');
        }));
        this.setPatchRate(this.patchRate);
    }
    requestJoin(options, isNew) {
        return 1;
    }
    onAuth(options) {
        return true;
    }
    get locked() {
        return this._locked;
    }
    hasReachedMaxClients() {
        return (this.clients.length + this.reservedSeats.size) >= this.maxClients;
    }
    setSeatReservationTime(seconds) {
        this.seatReservationTime = seconds;
        return this;
    }
    hasReservedSeat(sessionId) {
        return this.reservedSeats.has(sessionId);
    }
    setSimulationInterval(callback, delay = DEFAULT_SIMULATION_INTERVAL) {
        // clear previous interval in case called setSimulationInterval more than once
        if (this._simulationInterval) {
            clearInterval(this._simulationInterval);
        }
        this._simulationInterval = setInterval(() => {
            this.clock.tick();
            callback(this.clock.deltaTime);
        }, delay);
    }
    setPatchRate(milliseconds) {
        // clear previous interval in case called setPatchRate more than once
        if (this._patchInterval) {
            clearInterval(this._patchInterval);
            this._patchInterval = undefined;
        }
        if (milliseconds !== null && milliseconds !== 0) {
            this._patchInterval = setInterval(() => {
                this.broadcastPatch();
                this.broadcastAfterPatch();
            }, milliseconds);
        }
    }
    setState(newState) {
        this.clock.start();
        this._serializer.reset(newState);
        this.state = newState;
    }
    setMetadata(meta) {
        this.metadata = meta;
    }
    lock() {
        // rooms locked internally aren't explicit locks.
        this._lockedExplicitly = (arguments[0] === undefined);
        // skip if already locked.
        if (this._locked) {
            return;
        }
        this.emit('lock');
        this._locked = true;
    }
    unlock() {
        // only internal usage passes arguments to this function.
        if (arguments[0] === undefined) {
            this._lockedExplicitly = false;
        }
        // skip if already locked
        if (!this._locked) {
            return;
        }
        this.emit('unlock');
        this._locked = false;
    }
    send(client, data) {
        Protocol_1.send[Protocol_1.Protocol.ROOM_DATA](client, data);
    }
    broadcast(data, options = {}) {
        if (options.afterNextPatch) {
            delete options.afterNextPatch;
            this._afterNextPatchBroadcasts.push([data, options]);
            return true;
        }
        // no data given, try to broadcast patched state
        if (!data) {
            throw new Error('Room#broadcast: \'data\' is required to broadcast.');
        }
        // encode all messages with msgpack
        if (!(data instanceof Buffer)) {
            data = notepack_io_1.default.encode(data);
        }
        let numClients = this.clients.length;
        while (numClients--) {
            const client = this.clients[numClients];
            if (options.except !== client) {
                Protocol_1.send[Protocol_1.Protocol.ROOM_DATA](client, data, false);
            }
        }
        return true;
    }
    getAvailableData() {
        return __awaiter(this, void 0, void 0, function* () {
            return {
                clients: this.clients.length,
                maxClients: this.maxClients,
                metadata: this.metadata,
                roomId: this.roomId,
            };
        });
    }
    disconnect() {
        this.isDisconnecting = true;
        this.autoDispose = true;
        const delayedDisconnection = new Promise((resolve) => this.once('disconnect', () => resolve()));
        let numClients = this.clients.length;
        if (numClients > 0) {
            // prevent new clients to join while this room is disconnecting.
            this.lock();
            // clients may have `async onLeave`, room will be disposed after they all run
            while (numClients--) {
                const client = this.clients[numClients];
                const reconnection = this.reconnections[client.sessionId];
                if (reconnection) {
                    reconnection.reject();
                }
                else {
                    client.close(Protocol_1.WS_CLOSE_CONSENTED);
                }
            }
        }
        else {
            // no clients connected, dispose immediately.
            this.emit('dispose');
        }
        return delayedDisconnection;
    }
    // see @serialize decorator.
    get serializer() { return this._serializer.id; }
    _getSerializer() {
        return new SchemaSerializer_1.SchemaSerializer();
    }
    sendState(client) {
        Protocol_1.send[Protocol_1.Protocol.ROOM_STATE](client, this._serializer.getFullState(client));
    }
    broadcastPatch() {
        if (!this._simulationInterval) {
            this.clock.tick();
        }
        if (!this.state) {
            Debug_1.debugPatch('trying to broadcast null state. you should call #setState');
            return false;
        }
        return this._serializer.applyPatches(this.clients, this.state);
    }
    broadcastAfterPatch() {
        const length = this._afterNextPatchBroadcasts.length;
        if (length > 0) {
            for (let i = 0; i < length; i++) {
                this.broadcast.apply(this, this._afterNextPatchBroadcasts[i]);
            }
            // new messages may have been added in the meantime,
            // let's splice the ones that have been processed
            this._afterNextPatchBroadcasts.splice(0, length);
        }
    }
    allowReconnection(client, seconds = 15) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.isDisconnecting) {
                throw new Error('disconnecting');
            }
            yield this._reserveSeat(client, seconds, true);
            // keep reconnection reference in case the user reconnects into this room.
            const reconnection = new Utils_1.Deferred();
            this.reconnections[client.sessionId] = reconnection;
            // expire seat reservation after timeout
            this.reservedSeatTimeouts[client.sessionId] = setTimeout(() => reconnection.reject(false), seconds * 1000);
            const cleanup = () => {
                this.reservedSeats.delete(client.sessionId);
                delete this.reconnections[client.sessionId];
                delete this.reservedSeatTimeouts[client.sessionId];
            };
            reconnection.
                then(() => {
                clearTimeout(this.reservedSeatTimeouts[client.sessionId]);
                cleanup();
            }).
                catch(() => {
                cleanup();
                this._disposeIfEmpty();
            });
            return yield reconnection.promise;
        });
    }
    _reserveSeat(client, seconds = this.seatReservationTime, allowReconnection = false) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!allowReconnection && this.hasReachedMaxClients()) {
                return false;
            }
            this.reservedSeats.add(client.sessionId);
            yield this.presence.setex(`${this.roomId}:${client.id}`, client.sessionId, seconds);
            if (allowReconnection) {
                // store reference of the roomId this client is allowed to reconnect to.
                yield this.presence.setex(client.sessionId, this.roomId, seconds);
            }
            else {
                this.reservedSeatTimeouts[client.sessionId] = setTimeout(() => this.reservedSeats.delete(client.sessionId), seconds * 1000);
                this.resetAutoDisposeTimeout(seconds);
            }
            return true;
        });
    }
    resetAutoDisposeTimeout(timeoutInSeconds) {
        clearTimeout(this._autoDisposeTimeout);
        if (!this.autoDispose) {
            return;
        }
        this._autoDisposeTimeout = setTimeout(() => {
            this._autoDisposeTimeout = undefined;
            this._disposeIfEmpty();
        }, timeoutInSeconds * 1000);
    }
    _disposeIfEmpty() {
        const willDispose = (this.autoDispose &&
            this._autoDisposeTimeout === undefined &&
            this.clients.length === 0 &&
            this.reservedSeats.size === 0);
        if (willDispose) {
            this.emit('dispose');
        }
        return willDispose;
    }
    _dispose() {
        let userReturnData;
        if (this.onDispose) {
            userReturnData = this.onDispose();
        }
        if (this._patchInterval) {
            clearInterval(this._patchInterval);
            this._patchInterval = undefined;
        }
        if (this._simulationInterval) {
            clearInterval(this._simulationInterval);
            this._simulationInterval = undefined;
        }
        // clear all timeouts/intervals + force to stop ticking
        this.clock.clear();
        this.clock.stop();
        return userReturnData || Promise.resolve();
    }
    // allow remote clients to trigger events on themselves
    _emitOnClient(sessionId, event, args) {
        const remoteClient = this.remoteClients[sessionId];
        if (!remoteClient) {
            Debug_1.debugAndPrintError(`trying to send event ("${event}") to non-existing remote client (${sessionId})`);
            return;
        }
        if (typeof (event) !== 'string') {
            remoteClient.emit('message', event);
        }
        else {
            remoteClient.emit(event, args);
        }
    }
    _onMessage(client, message) {
        message = Protocol_1.decode(message);
        if (!message) {
            Debug_1.debugAndPrintError(`${this.roomName} (${this.roomId}), couldn't decode message: ${message}`);
            return;
        }
        if (message[0] === Protocol_1.Protocol.ROOM_DATA) {
            this.onMessage(client, message[2]);
        }
        else if (message[0] === Protocol_1.Protocol.LEAVE_ROOM) {
            // stop interpreting messages from this client
            client.removeAllListeners('message');
            // prevent "onLeave" from being called twice in case the connection is forcibly closed
            client.removeAllListeners('close');
            // only effectively close connection when "onLeave" is fulfilled
            this._onLeave(client, Protocol_1.WS_CLOSE_CONSENTED).then(() => client.close());
        }
        else {
            this.onMessage(client, message);
        }
    }
    _onJoin(client, options, auth) {
        // create remote client instance.
        if (client.remote) {
            client = (new RemoteClient_1.RemoteClient(client, this.roomId, this.presence));
            this.remoteClients[client.sessionId] = client;
        }
        this.clients.push(client);
        // delete seat reservation
        this.reservedSeats.delete(client.sessionId);
        if (this.reservedSeatTimeouts[client.sessionId]) {
            clearTimeout(this.reservedSeatTimeouts[client.sessionId]);
            delete this.reservedSeatTimeouts[client.sessionId];
        }
        // clear auto-dispose timeout.
        if (this._autoDisposeTimeout) {
            clearTimeout(this._autoDisposeTimeout);
            this._autoDisposeTimeout = undefined;
        }
        // lock automatically when maxClients is reached
        if (!this._locked && this.clients.length === this.maxClients) {
            this._maxClientsReached = true;
            this.lock.call(this, true);
        }
        // confirm room id that matches the room name requested to join
        Protocol_1.send[Protocol_1.Protocol.JOIN_ROOM](client, client.sessionId, this._serializer.id, this._serializer.handshake && this._serializer.handshake());
        // bind onLeave method.
        client.on('message', this._onMessage.bind(this, client));
        client.once('close', this._onLeave.bind(this, client));
        // send current state when new client joins the room
        if (this.state) {
            this.sendState(client);
        }
        const reconnection = this.reconnections[client.sessionId];
        if (reconnection) {
            reconnection.resolve(client);
        }
        else {
            // emit 'join' to room handler
            this.emit('join', client);
            return this.onJoin && this.onJoin(client, options, auth);
        }
    }
    _onLeave(client, code) {
        return __awaiter(this, void 0, void 0, function* () {
            // call abstract 'onLeave' method only if the client has been successfully accepted.
            if (Utils_1.spliceOne(this.clients, this.clients.indexOf(client)) && this.onLeave) {
                try {
                    yield this.onLeave(client, (code === Protocol_1.WS_CLOSE_CONSENTED));
                }
                catch (e) {
                    Debug_1.debugAndPrintError(`onLeave error: ${(e && e.message || e || 'promise rejected')}`);
                }
            }
            this.emit('leave', client);
            // remove remote client reference
            if (client instanceof RemoteClient_1.RemoteClient) {
                delete this.remoteClients[client.sessionId];
            }
            // dispose immediatelly if client reconnection isn't set up.
            const willDispose = this._disposeIfEmpty();
            // unlock if room is available for new connections
            if (!willDispose && this._maxClientsReached && !this._lockedExplicitly) {
                this._maxClientsReached = false;
                this.unlock.call(this, true);
            }
        });
    }
}
exports.Room = Room;
