"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_1 = require("@colyseus/schema");
const Protocol_1 = require("../Protocol");
class SchemaSerializer {
    constructor() {
        this.id = 'schema';
        this.hasFiltersByClient = false;
    }
    reset(newState) {
        if (!(newState instanceof schema_1.Schema)) {
            throw new Error(`SchemaSerializer error. See: https://docs.colyseus.io/migrating/0.10/#new-default-serializer`);
        }
        this.state = newState;
        this.hasFiltersByClient = this.hasFilter(newState._schema, newState._filters);
    }
    getFullState(client) {
        return (this.hasFiltersByClient)
            ? this.state.encodeAllFiltered(client)
            : this.state.encodeAll();
    }
    applyPatches(clients) {
        const hasChanges = this.state.$changed;
        if (hasChanges) {
            let numClients = clients.length;
            if (!this.hasFiltersByClient) {
                // encode changes once, for all clients
                const patches = this.state.encode();
                while (numClients--) {
                    const client = clients[numClients];
                    Protocol_1.send[Protocol_1.Protocol.ROOM_STATE_PATCH](client, patches);
                }
            }
            else {
                // encode state multiple times, for each client
                while (numClients--) {
                    const client = clients[numClients];
                    Protocol_1.send[Protocol_1.Protocol.ROOM_STATE_PATCH](client, this.state.encodeFiltered(client));
                }
                throw new Error('filters are not fully implemented yet.');
                // this.state.markAsUnchanged();
            }
        }
        return hasChanges;
    }
    handshake() {
        return this.state && schema_1.Reflection.encode(this.state);
    }
    hasFilter(schema, filters) {
        if (!filters) {
            return false;
        }
        for (const fieldName in schema) {
            if (filters[fieldName]) {
                return true;
            }
            else if (typeof (schema[fieldName]) === 'function') {
                const childSchema = schema[fieldName]._schema;
                const childFilters = schema[fieldName]._filters;
                return this.hasFilter(childSchema, childFilters);
            }
            else if (Array.isArray(schema[fieldName])) {
                const childSchema = schema[fieldName][0]._schema;
                const childFilters = schema[fieldName][0]._filters;
                return this.hasFilter(childSchema, childFilters);
            }
            else if (schema[fieldName].map) {
                const childSchema = schema[fieldName].map._schema;
                const childFilters = schema[fieldName].map._filters;
                return this.hasFilter(childSchema, childFilters);
            }
        }
        return false;
    }
}
exports.SchemaSerializer = SchemaSerializer;
